//
// NAME: CIDLib_Collection.Hpp
//
// DESCRIPTION: 
//
//  This header implements the base class for all template collections. All
//  of the collections derive from TCollection and must implement its virtual
//  interface. This interface allows for things like adding new elements,
//  checking the number of elements, checking for empty, checking for full.
//  I.e. things that do not depend upon any particular ordering or storage
//  style or knowledge of the storage mechanism.
//
//  This class implements the MLockable interface but only actually honors
//  it when the derived class asks to. In which case it will create a mutex
//  and pass on the lock/unlock calls to it.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 12/24/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TCollection
//  PREFIX: curs
// ----------------------------------------------------------------------------
template <class TElem> class TCollection : public TObject, public MLockable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        ~TCollection()
        {
            if (__pmtxLock)
                delete __pmtxLock;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid
        Lock(const tCIDLib::TCard4 c4Timeout = kCIDLib::c4MaxWait) const
        {
            // If this one is lockable, then do the lock
            if (__pmtxLock)
                __pmtxLock->Lock(c4Timeout);
        }

        tCIDLib::TVoid Unlock() const
        {
            // If this one is lockable, then do the unlock
            if (__pmtxLock)
                __pmtxLock->Unlock();
        }


        // --------------------------------------------------------------------
        //  Public, pure virtual methods
        // --------------------------------------------------------------------
        virtual tCIDLib::TVoid Add
        (
            const   TElem&                  objNew
        ) = 0;

        virtual tCIDLib::TBoolean bFlushCur() = 0;

        virtual tCIDLib::TBoolean bIsEmpty() const = 0;

        virtual tCIDLib::TBoolean bNext() = 0;

        virtual tCIDLib::TBoolean bResetIter() = 0;

        virtual tCIDLib::TCard4 c4ElemCount() const = 0;

        virtual tCIDLib::TCard4 c4MaxElemCount() const = 0;

        virtual tCIDLib::TVoid Flush() = 0;

        virtual TElem& objCur() const = 0;

        virtual TElemCursor<TElem>* pcursNew() const = 0;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsMTSafe() const
        {
            return (__pmtxLock != 0);
        }

        tCIDLib::EMTStates eMTState() const
        {
            if (__pmtxLock)
                return tCIDLib::EMTState_Safe;
            return tCIDLib::EMTState_Unsafe;
        }

        tCIDLib::TCard4 c4SerialNum() const
        {
            TLocker lockCol(this);
            tCIDLib::TCard4 c4Ret = __c4SerialNum;
            return c4Ret;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        TCollection(const tCIDLib::EMTStates eMTSafe = tCIDLib::EMTState_Unsafe) :

            __c4SerialNum(0)
            , __pmtxLock(0)
        {
            __CheckFirstOfType();

            if (eMTSafe)
                __pmtxLock = new TMutex;
        }

        TCollection(const TCollection<TElem>& colToCopy) :

            __c4SerialNum(0)
            , __pmtxLock(0)
        {
            __CheckFirstOfType();

            if (colToCopy.__pmtxLock)
                __pmtxLock = new TMutex;
        }


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TCard4 _c4IncSerialNum()
        {
            TLocker lockCol(this);
            __c4SerialNum++;
            return __c4SerialNum;
        }


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TCollection<TElem>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;
                TString strTmp(L"TCollectionOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4SerialNum
        //      This is a serial number from the collection. It is bumped up
        //      every time a node is added or removed. Cursors watch it to
        //      know if they are out of date.
        //
        //  __pmtxLock
        //      This is the optional mutex that allows this collection to
        //      be lockable. The derived class indicates to our constructor
        //      whether this collection should be mutlti-thread safe.
        // --------------------------------------------------------------------
        tCIDLib::TCard4     __c4SerialNum;
        TMutex*             __pmtxLock;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a collection of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean    __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TCollection<TElem>,TObject)
};

template <class TElem> tCIDLib::TBoolean TCollection<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TCollection<TElem>::clsThis;

#pragma pack(pop)
