//
// NAME: CIDLib_CollectSearch.Hpp
//
// DESCRIPTION: 
//
//  This header defines the TColSearch class, which provides a means to
//  search a collection for objects that match a passed value. This class can
//  be a totally generic template since it just uses a cursor and a provided
//  user comparison function.
//
//  Note that some collections are specialized and also provide their own
//  search mechanisms, such as mapping collections that have key based
//  searches. This mechanism can be used for them, given the correct comparator
//  object, but it wouldn't be worth it usually (though it can provide
//  generality to do so.)
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 12/24/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TColSearch
//  PREFIX: cols
// ----------------------------------------------------------------------------
template <class TCol, class TElem> class TColSearch : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Public, static data
        // --------------------------------------------------------------------
        static const TClass clsThis;


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TColSearch( const   TCol* const             pcolToSearch
                    , const TObjEq<TElem>* const    pcompToAdopt) :

            __bAdoptedComp(kCIDLib::True)
            , __cursThis(pcolToSearch)
            , __pcolToSearch(pcolToSearch)
            , __pcompToUse(pcompToAdopt)
        {
            // Set the class name on the first creation of a bag of this type
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TColSearchOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        TColSearch( const   TCol* const             pcolToSearch
                    , const TObjEq<TElem>&          compToUse) :

            __bAdoptedComp(kCIDLib::False)
            , __cursThis(pcolToSearch)
            , __pcolToSearch(pcolToSearch)
            , __pcompToUse(&compToUse)
        {
            // Set the class name on the first creation of a bag of this type
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TColSearchOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        ~TColSearch()
        {
            if (__bAdoptedComp)
                delete __pcompToUse;
        }


        // --------------------------------------------------------------------
        //  Public inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsDescendantOf(const TClass& clsTarget) const
        {
            if (clsTarget == clsThis)
                return kCIDLib::True;
            return TObject::bIsDescendantOf(clsTarget);
        }

        const TClass& clsIsA() const
        {
            return clsThis;
        }

        const TClass& clsParent() const
        {
            return TObject::clsThis;
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bFind(const TElem& objToFind)
        {
            // Reset the cursor for a new search
            if (!__cursThis.bReset())
                return kCIDLib::False;

            // Check each element for equality with the passed object
            do
            {
                if (__pcompToUse->bEqual(__cursThis.objCur(), objToFind))
                    return kCIDLib::True;
            }   while (__cursThis.bNext());
            return kCIDLib::False;
        }

        tCIDLib::TBoolean bFindNext(const TElem& objToFind)
        {
            //
            //  Move up the cursor before we start, so that we start after
            //  the last match. If this hits the end, then we are done.
            //
            if (!__cursThis.bNext())
                return kCIDLib::False;

            // Check each element for equality with the passed object
            do
            {
                if (__pcompToUse->bEqual(__cursThis.objCur(), objToFind))
                    return kCIDLib::True;
            }   while (__cursThis.bNext());
            return kCIDLib::False;
        }

        tCIDLib::TBoolean bFindNextDifferent(const TElem& objToFind)
        {
            //
            //  Move up the cursor before we start, so that we start after
            //  the last match. If this hits the end, then we are done.
            //
            if (!__cursThis.bNext())
                return kCIDLib::False;

            // Check each element for inequality with the passed object
            do
            {
                if (!__pcompToUse->bEqual(__cursThis.objCur(), objToFind))
                    return kCIDLib::True;
            }   while (__cursThis.bNext());
            return kCIDLib::False;
        }

        const TElem& objCur() const
        {
            return __cursThis.objCur();
        }

        TCol::TCursor cursCur() const
        {
            return __cursThis;
        }


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TColSearch();

        TColSearch(const TColSearch<TCol,TElem>&);

        tCIDLib::TVoid operator=(const TColSearch<TCol,TElem>&);



        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __bAdoptedComp
        //      This indicates whether we own the comparison object or are just
        //      referencing it.
        //      
        //  __cursThis
        //      This is the cursor that we are using to iterate the collection
        //      and maintain the 
        //
        //  __pcolToSearch
        //      This is a pointer to the collection that we are searching.
        //
        //  __pcompToUse
        //      This is the comparator object that we are to use.
        // --------------------------------------------------------------------
        tCIDLib::TBoolean       __bAdoptedComp;
        TCol::TCursor           __cursThis;
        const TCol*             __pcolToSearch;
        const TObjEq<TElem>*    __pcompToUse;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a collection search of a particular
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;
};

template <class TCol, class TElem> tCIDLib::TBoolean TColSearch<TCol,TElem>::__bFirstOfType = kCIDLib::False;
template <class TCol, class TElem> const TClass TColSearch<TCol,TElem>::clsThis;

#pragma pack(pop)
