//
// NAME: CIDLib_CollectCursor.Hpp
//
// DESCRIPTION: 
//
//  This header defines the base collection cursor classes. Collection cursors
//  allow code to move around within a collection without much knowledge of
//  the particular type of collection being manipulated and without having a
//  the problem of multi-thread collection access that would occur if the
//  collection itself provided the iteration.
//
//  Actually collections themselves do provide an iterator, but its for one user
//  at a time and provides writable access. Cursors provide read only access to
//  the elements but there can be multiple of them for a collection at once.
//
//  The basic TElemCursor class provides just forward iteration. Derived from
//  it is TBiElemCursor which provides bidirectional iteration by adding
//  a bPrevious() method.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 12/24/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TElemCursor
//  PREFIX: curs
// ----------------------------------------------------------------------------
template <class TElem> class TElemCursor : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        ~TElemCursor() {}


        // --------------------------------------------------------------------
        //  Public, pure virtual methods
        // --------------------------------------------------------------------
        virtual tCIDLib::TBoolean bIsEmpty() const = 0;

        virtual tCIDLib::TBoolean bIsValid() const = 0;

        virtual tCIDLib::TBoolean bNext() = 0;

        virtual tCIDLib::TBoolean bReset() = 0;

        virtual tCIDLib::TCard4 c4ElemCount() const = 0;

        virtual const TElem& objCur() const = 0;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TCard4 c4SerialNum() const
        {
            return __c4SerialNum;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Hidden construtors and operators
        // --------------------------------------------------------------------
        TElemCursor(const tCIDLib::TCard4 c4SerialNum) :

            __c4SerialNum(c4SerialNum)
        {
            if (__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;
                TString  strTmp(L"TElemCursorOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        TElemCursor(const TElemCursor<TElem>& cursToCopy) :

            __c4SerialNum(cursToCopy.__c4SerialNum)
        {
        }

        TElemCursor<TElem>& operator=(const TElemCursor<TElem>& cursToAssign)
        {
            __c4SerialNum = cursToAssign.__c4SerialNum;
            return *this;
        }


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _CheckSerialNum(const tCIDLib::TCard4 c4ToCheck) const
        {
            if (c4ToCheck != __c4SerialNum)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_StaleCursor
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
        }

        tCIDLib::TCard4 _c4SerialNum(const tCIDLib::TCard4 c4New)
        {
            __c4SerialNum = c4New;
            return __c4SerialNum;
        }


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TElemCursor();


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4SerialNum
        //      This is a serial number from the collection. Its stored when
        //      this collection is first created or synced back up by a
        //      bReset() call. This lets the derived cursor class know if
        //      it is out of date with respect to the collection it is
        //      cursoring.
        // --------------------------------------------------------------------
        tCIDLib::TCard4     __c4SerialNum;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a collection of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean    __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TElemCursor<TElem>,TObject)
};

template <class TElem> tCIDLib::TBoolean TElemCursor<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TElemCursor<TElem>::clsThis;



// ----------------------------------------------------------------------------
//   CLASS: TBiElemCursor
//  PREFIX: curs
// ----------------------------------------------------------------------------
template <class TElem> class TBiElemCursor : public TElemCursor<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TBiElemCursor(const tCIDLib::TCard4 c4SerialNum) :

            TElemCursor<TElem>(c4SerialNum)
        {
            if (__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;
                TString  strTmp(L"TBiElemCursorOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        TBiElemCursor(const TBiElemCursor<TElem>& cursToCopy) :

            TElemCursor<TElem>(cursToCopy)
        {
        }

        ~TBiElemCursor() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TBiElemCursor<TElem>& operator=(const TBiElemCursor<TElem>& cursToAssign)
        {
            if (this == &cursToAssign)
                return *this;

            TParent::operator=(cursToAssign);
            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, pure virtual methods
        // --------------------------------------------------------------------
        virtual tCIDLib::TBoolean bPrevious() = 0;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TBiElemCursor();


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a collection of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean    __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TElemCursor<TElem>,TObject)
};

template <class TElem> tCIDLib::TBoolean TBiElemCursor<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TBiElemCursor<TElem>::clsThis;

#pragma pack(pop)
