//
// NAME: CIDLib_Class.Hpp
//
// DESCRIPTION: 
//
//  This header holds one of the most primal classes in the system. It is
//  the class class. Every class is given a static const member of this type
//  which holds its class type. At the same time it is not the most primal
//  since it is derived from TObject. But this one is still more primal in
//  a way because all of the RTTI methods depend upon it. This of course
//  makes this class special due to the circular nature of its relationship
//  with TObject.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/02/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  Because of the very special nature of this class, it cannot use
//      the standard members macros (because of order of inclusion stuff.)
//      So it does the standard RTTI members manually.
//


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
class   TString;


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//  CLASS: TClass
// PREFIX: cls
// ----------------------------------------------------------------------------
class CIDLIBEXP TClass :

    public TObject, public MFormattable, public MStreamable, public MDuplicable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors.
        // --------------------------------------------------------------------
        TClass();

        TClass
        (
            const   tCIDLib::Tch* const     pszClassName
        );

        TClass
        (
            const   TString&                strClassName
        );

        TClass
        (
            const   TClass&                 clsToCopy
        );

        ~TClass();


        // --------------------------------------------------------------------
        //  This is a special constructor that will register the class
        //  for dynamic creation (usually used via the RTTIData() macro.
        // --------------------------------------------------------------------
        TClass
        (
            const   tCIDLib::Tch* const     pszClassName
            ,       tCIDLib::TObjFactory    pfnFactory
        );


        // --------------------------------------------------------------------
        //  Public, static data members
        // --------------------------------------------------------------------
        static const TClass   clsThis;


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator<
        (
            const   TClass&                 clsToCompare
        )   const;

        tCIDLib::TBoolean operator<=
        (
            const   TClass&                 clsToCompare
        )   const;

        tCIDLib::TBoolean operator>
        (
            const   TClass&                 clsToCompare
        )   const;

        tCIDLib::TBoolean operator>=
        (
            const   TClass&                 clsToCompare
        )   const;

        tCIDLib::TBoolean operator==
        (
            const   TClass&                 clsToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TClass&                 clsToTest
        )   const;

        operator const tCIDLib::Tch*() const;

        TClass& operator=
        (
            const   TClass&                 clsToAssign
        );


        // --------------------------------------------------------------------
        //  Public static methods
        //
        //  clsForType(), pobjMakeNewOfType(), and RegisterClass() are all
        //  implemented in CIDLib_TypeRegistry.Cpp!
        // --------------------------------------------------------------------
        static TClass clsForType
        (
            const   TString&                strTypeName
        );

        static TObject* pobjMakeNewOfType
        (
            const   tCIDLib::Tch* const     pszTypeName
        );

        static TObject* pobjMakeNewOfType
        (
            const   TString&                strTypeName
        );

        static tCIDLib::TVoid RegisterClass
        (
            const   TString&                strToRegister
            ,       tCIDLib::TObjFactory    pFactoryFunc
        );

        static tCIDLib::TVoid TestCanCast
        (
            const   TObject&                objToCast
            , const TClass&                 clsToCastTo
        );

        static tCIDLib::TVoid TestCanCast
        (
            const   TObject&                objToCast
            , const tCIDLib::Tch* const     pszTypeName
        );


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsDescendantOf
        (
            const   TClass&              clsTarget
        )   const;

        const TClass& clsIsA() const;

        const TClass& clsParent() const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bNameValid() const;

        tCIDLib::THashVal hshInternal() const;

        tCIDLib::THashVal hshCalcHash
        (
            const   tCIDLib::TCard4      c4Modulus
        )   const;

        TObject* pobjMakeNew() const;

        tCIDLib::TVoid QueryClassName
        (
                    TString&             strName
        )   const;

        const tCIDLib::Tch* pszClassName() const;


    protected   :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmDest
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private         :
        // --------------------------------------------------------------------
        //  Private types
        // --------------------------------------------------------------------
        typedef TObject TParent;


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4BufChars
        //      This is the number of characters that the class name buffer
        //      can currently hold. This lets us not necessarily reallocate
        //      it if a new name is assigned that is <= the length of the
        //      current name.
        //
        //  __hshName
        //      This is used to store the hash value for the name. Since
        //      class objects are often compared in time critical code (as
        //      part of RTTI checks), this helps speed things up a lot.
        //
        //  __pszClassName
        //      This is the storage for the class name.
        // --------------------------------------------------------------------
        tCIDLib::TCard4     __c4BufChars;
        tCIDLib::THashVal   __hshName;
        tCIDLib::Tch*       __pszClassName;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        DefPolyDup(TClass)
};

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TClass::operator>=(const TClass& clsToCompare) const
{
    return !operator<(clsToCompare);
}

inline tCIDLib::TBoolean TClass::operator<=(const TClass& clsToCompare) const
{
    return !operator>(clsToCompare);
}

inline tCIDLib::TBoolean TClass::operator!=(const TClass& clsToTest) const
{
    return !operator==(clsToTest);
}

inline TClass::operator const tCIDLib::Tch*() const
{
    return __pszClassName;
}

// ----------------------------------------------------------------------------
//  Public, inherited methods
// ----------------------------------------------------------------------------
inline const TClass& TClass::clsIsA() const
{
    return clsThis;
}

inline const TClass& TClass::clsParent() const
{
    return TParent::clsIsA();
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::THashVal TClass::hshInternal() const
{
    return __hshName;
}

inline const tCIDLib::Tch* TClass::pszClassName() const
{
    return __pszClassName;
}


// -----------------------------------------------------------------------------
//  Some template based functions that build upon the pobjMakeNewOfType()
//  method of the facility class above. These automatically invoke the method
//  and do the correct casting.
// -----------------------------------------------------------------------------
template <class T>
tCIDLib::TVoid MakeNewOfType(T*& pToFill, const tCIDLib::Tch* pszClassName)
{
    // Create it as a base object type first, since we know that's ok
    TObject* const pobjTmp = TClass::pobjMakeNewOfType(pszClassName);

    // Put a janitor on it until we test the cast
    TJanitor<TObject> janTest(pobjTmp);

    // Confirm that the user requested cast will be safe
    TClass::TestCanCast(*pobjTmp, T::clsThis);

    // Do the cast and orphan the object
    pToFill = (T*)pobjTmp;
    janTest.Orphan();
}

template <class T>
tCIDLib::TVoid MakeNewOfType(T*& pToFill, const TString& strClassName)
{
    // Create it as a base object type first, since we know that's ok
    TObject* const pobjTmp = TClass::pobjMakeNewOfType(strClassName);

    // Put a janitor on it until we test the cast
    TJanitor<TObject> janTest(pobjTmp);

    // Confirm that the user requested cast will be safe
    TClass::TestCanCast(*pobjTmp, T::clsThis);

    // Do the cast and orphan the object now
    pToFill = (T*)pobjTmp;
    janTest.Orphan();
}



// -----------------------------------------------------------------------------
//  These template classes do dynamic casting in the standard C++ style. They
//  return 0 if the cast is not valid.
// -----------------------------------------------------------------------------
template <class T> T* pDynamicCast(TObject* const pobjToCast)
{
    if (pobjToCast->bIsDescendantOf(T::clsThis))
        return (T*)pobjToCast;
    return 0;
}

template <class T> const T* pDynamicCast(const TObject* const pobjToCast)
{
    if (pobjToCast->bIsDescendantOf(T::clsThis))
        return (const T*)pobjToCast;
    return 0;
}
