//
// NAME: CIDLib_Class.Cpp
//
// DESCRIPTION:
//
//  This module implements the TClass class, which is the RTTI class for
//  the CIDLib system. It represents a class in the system.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 06/20/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//   CLASS: TClass
//  PREFIX: cls
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TClass: Static members
//
//  clsThis
//      We cannot use the standard macro to generate these because of circular
//      references at this level.
// ----------------------------------------------------------------------------
const TClass TClass::clsThis(L"TClass");


// ----------------------------------------------------------------------------
//  TClass: Constructors and Destructors
// ----------------------------------------------------------------------------

TClass::TClass() :

    __c4BufChars(64)
    , __hshName(0)
    , __pszClassName(0)
{
    //
    //  Allocate a reasonable sized string that will hold a likely class
    //  name without reallocation.
    //
    __pszClassName = new tCIDLib::Tch[65];
    __pszClassName[0] = 0;
}

TClass::TClass(const TClass& clsSrc) :

    __c4BufChars(clsSrc.__c4BufChars)
    , __hshName(clsSrc.__hshName)
    , __pszClassName(0)
{
    __pszClassName = TRawStr::pszReplicate(clsSrc.__pszClassName);
}

TClass::TClass(const TString& strClassName) :

    __c4BufChars(0)
    , __hshName(0)
    , __pszClassName(0)
{
    // Replicate the name and store its hash
    __pszClassName = TRawStr::pszReplicate(strClassName.pszData());
    __hshName = TRawStr::hshHashStr(__pszClassName, kCIDLib::c4ClassModulus);

    // Store the buffer size
    __c4BufChars = TRawStr::c4StrLen(__pszClassName);
}

TClass::TClass(const tCIDLib::Tch* const pszClassName) :

    __c4BufChars(0)
    , __hshName(0)
    , __pszClassName(0)
{
    if (!pszClassName)
    {
        tCIDLib::TZStr1K szTmp;
        TRawStr::CopyCatStr
        (
            szTmp
            , c4MaxBufChars(szTmp)
            , L"TClass: Invalid class name ("
            , pszClassName
            , L")"
        );
        _MSGPOPUP_(szTmp)
        TFacCIDLib::ExitProcess(tCIDLib::EExit_FatalError);
    }

    // Replicate the passed class name
    __pszClassName = TRawStr::pszReplicate(pszClassName);

    // Calculate the hash value
    __hshName = TRawStr::hshHashStr(__pszClassName, kCIDLib::c4ClassModulus);

    // Store the buffer size
    __c4BufChars = TRawStr::c4StrLen(__pszClassName);
}

TClass::TClass( const   tCIDLib::Tch* const     pszClassName
                ,       tCIDLib::TObjFactory    pfnFactory) :
    __c4BufChars(0)
    , __hshName(0)
    , __pszClassName(0)
{
    if (!pszClassName)
    {
        tCIDLib::TZStr1K szTmp;
        TRawStr::CopyCatStr
        (
            szTmp
            , c4MaxBufChars(szTmp)
            , L"TClass: Invalid class name ("
            , pszClassName
            , L")"
        );
        _MSGPOPUP_(szTmp)
        TFacCIDLib::ExitProcess(tCIDLib::EExit_FatalError);
    }

    // Replicate the passed class name
    __pszClassName = TRawStr::pszReplicate(pszClassName);

    // Calculate the hash value
    __hshName = TRawStr::hshHashStr(__pszClassName, kCIDLib::c4ClassModulus);

    // Store the buffer size
    __c4BufChars = TRawStr::c4StrLen(__pszClassName);

    // And now register the class
    TClass::RegisterClass(__pszClassName, pfnFactory);
}

TClass::~TClass()
{
    delete __pszClassName;
}


// ----------------------------------------------------------------------------
//  TClass: Public operators
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TClass::operator<(const TClass& clsToCompare) const
{
    tCIDLib::TInt4 i4Res = TRawStr::eCompareStr
    (
        __pszClassName
        , clsToCompare.__pszClassName
    );

    if (i4Res < 0)
        return kCIDLib::True;

    return kCIDLib::False;
}

tCIDLib::TBoolean TClass::operator>(const TClass& clsToCompare) const
{
    // It could be equal, so compare our string to the passed one
    tCIDLib::TInt4 i4Res = TRawStr::eCompareStr
    (
        __pszClassName
        , clsToCompare.__pszClassName
    );

    if (i4Res > 0)
        return kCIDLib::True;

    return kCIDLib::False;
}


tCIDLib::TBoolean TClass::operator==(const TClass& clsToTest) const
{
    // Check the hash value first
    if (__hshName != clsToTest.__hshName)
        return kCIDLib::False;

    // Could be equal, so compare the names
    if (!TRawStr::eCompareStr(__pszClassName, clsToTest.__pszClassName))
        return kCIDLib::True;

    return kCIDLib::False;
}


TClass& TClass::operator=(const TClass& clsSrc)
{
    if (this == &clsSrc)
        return *this;

    // Copy over the name string
    tCIDLib::TCard4 c4Len = 0;
    
    tCIDLib::TCard4 c4NewChars = TRawStr::c4StrLen(clsSrc.__pszClassName);

    // If the current buffer is not big enough, reallocate
    if (__c4BufChars < c4NewChars)
    {
        delete __pszClassName;
        __pszClassName = 0;
        __pszClassName = new tCIDLib::Tch[c4NewChars+1];

        // Store the new buffer size
        __c4BufChars = c4NewChars;
    }

    // Copy over the source name
    TRawStr::CopyStr(__pszClassName, clsSrc.__pszClassName, c4NewChars);

    // Copy over the hash value
    __hshName = clsSrc.__hshName;

    return *this;
}


// ----------------------------------------------------------------------------
//  TClass: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TClass::TestCanCast(const   TObject&    objToCast
                    , const TClass&     clsToCastTo)
{
    if (!objToCast.bIsDescendantOf(clsToCastTo))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcRTTI_BadCast
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , objToCast.clsIsA()
            , clsToCastTo
        );
    }
}

tCIDLib::TVoid
TClass::TestCanCast(const   TObject&            objToCast
                    , const tCIDLib::Tch* const pszClassName)
{
    // Find the class object for this name first
    TClass clsToCastTo = clsForType(pszClassName);

    if (!objToCast.bIsDescendantOf(clsToCastTo))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcRTTI_BadCast
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , objToCast.clsIsA()
            , clsToCastTo
        );
    }
}


// ----------------------------------------------------------------------------
//  TClass: Public, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean
TClass::bIsDescendantOf(const TClass& clsTarget) const
{
    if (clsTarget.__hshName != __hshName)
        return TParent::bIsDescendantOf(clsTarget);

    if (!TRawStr::eCompareStr
    (
        clsTarget.__pszClassName
        , clsThis.__pszClassName))
    {
        return kCIDLib::True;
    }

    return TParent::bIsDescendantOf(clsTarget);
}


// ----------------------------------------------------------------------------
//  TClass: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TClass::bNameValid() const
{
    if (__pszClassName)
        return kCIDLib::True;

    return kCIDLib::False;
}


tCIDLib::THashVal
TClass::hshCalcHash(const tCIDLib::TCard4 c4Modulus) const
{
    return TRawStr::hshHashStr(__pszClassName, c4Modulus);
}


TObject* TClass::pobjMakeNew() const
{
    return TClass::pobjMakeNewOfType(__pszClassName);
}


tCIDLib::TVoid TClass::QueryClassName(TString& strName) const
{
    strName = __pszClassName;
}


// ----------------------------------------------------------------------------
//  TClass: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TClass::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo << __pszClassName;
}


tCIDLib::TVoid TClass::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Parse out our hash value
    strmToReadFrom >> __hshName;

    // Parse out the length and buffer size into temps
    tCIDLib::TCard4 c4Len;
    tCIDLib::TCard4 c4Buf;
    strmToReadFrom >> c4Buf;
    strmToReadFrom >> c4Len;

    // Test for a 0 sized buffer, which never should happen
    if (!c4Buf)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_BadNameCount
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // Reallocate the buffer if needed
    if (c4Buf != __c4BufChars)
    {
        delete __pszClassName;
        __pszClassName = 0;
        __pszClassName = new tCIDLib::Tch[c4Buf + 1];
        __pszClassName[0] = 0;

        // Store the new buffer size
        __c4BufChars = c4Buf;
    }

    // Parse out our text now
    strmToReadFrom.ReadRawBuffer(__pszClassName, c4Len);

    // Cap of the stream
    __pszClassName[c4Len] = 0;

    //
    //  Hash it and compare with the hash we read in. If not the same then
    //  are are not reading in the right place.
    //
    tCIDLib::THashVal hshTmp = TRawStr::hshHashStr
    (
        __pszClassName
        , kCIDLib::c4ClassModulus
    );

    if (hshTmp != __hshName)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcClass_BadHash
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , TString(__pszClassName)
        );
    }
}

tCIDLib::TVoid TClass::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Format our hash value and buffer size
    strmToWriteTo << __hshName;
    strmToWriteTo << __c4BufChars;

    // Write out how many characters we have
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(__pszClassName);
    strmToWriteTo << c4Len;

    // Format our class name into the buffer, only the chars used, no nul
    strmToWriteTo.WriteRawBuffer(__pszClassName, c4Len);
}
