//
// NAME: CIDLib_BinaryStream.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_BinaryStream.Cpp module. This module
//  implements a binary stream, i.e. it reads and writes data in binary
//  format as apposed to text format. Like all streams its actual data
//  source/sink is provided via an implementation object so it can read
//  write anything that an implementation object is provided for.
//
//  Also defined here is TBinStreamImpl, which defines the interface for a
//  binary stream implementation class. TBinaryStream is written in terms
//  of this abstract class, so it is independent of where data physically
//  comes from or goes to.
//
//  Also provided here are two template methods that are used for polymorphic
//  reads and writes on a binary stream. These guys know how to store type info
//  along with the object. They also know how to read that type info back
//  in and use dynamic type support to create a new object of that type and
//  read it back in. This is a very important (and suprisingly easy)
//  mechanism which supports a lot of higher level magic. They are implemented
//  as templates so as to be very typesafe. They are implemented in terms of
//  the MStreamable mixin interface.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/28/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  Derived classes of TBinaryStream can call the default constructor
//      then gen up an implmenetation object and call _AdoptImplObject().
//      The public has to provide the object during construction.
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
class   TMemBuf;

// -----------------------------------------------------------------------------
//   CLASS: TBinStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------
class CIDLIBEXP TBinStreamImpl : public TObject
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        ~TBinStreamImpl() {}


        // ---------------------------------------------------------------------
        //  Public, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TBoolean bEndOfStream() const = 0;

        virtual tCIDLib::TFilePos fposCurPos() const = 0;

        virtual tCIDLib::TFilePos fposLogicalEnd() const = 0;

        virtual tCIDLib::TFilePos fposPhysicalEnd() const = 0;

        virtual tCIDLib::TFilePos fposSeekToEnd() = 0;

        virtual tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        ) = 0;

        virtual tCIDLib::TVoid ReadBytes
        (
                    tCIDLib::TVoid* const   pBuffer
            , const tCIDLib::TCard4         c4BytesToRead
            ,       tCIDLib::TCard4&        c4Actual
        ) = 0;

        virtual tCIDLib::TVoid Reset() = 0;

        virtual tCIDLib::TVoid TruncateAtZero() = 0;

        virtual tCIDLib::TVoid WriteBytes
        (
            const   tCIDLib::TVoid* const   pBuffer
            , const tCIDLib::TCard4         c4BytesToWrite
            ,       tCIDLib::TCard4&        c4Actual
        ) = 0;


    protected :
        // ---------------------------------------------------------------------
        //  Hidden constructors
        // ---------------------------------------------------------------------
        TBinStreamImpl() {}


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TBinStreamImpl(const TBinStreamImpl&);

        tCIDLib::TVoid operator=(const TBinStreamImpl&);


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TBinStreamImpl,TObject)
};



// ----------------------------------------------------------------------------
//   CLASS: TBinaryStream
//  PREFIX: strm
// ----------------------------------------------------------------------------
class CIDLIBEXP TBinaryStream : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Public class types. These are generally cast to a byte when
        //  written out!
        // --------------------------------------------------------------------
        enum EStreamMarkers
        {
            EStreamMarker_None          = 0
            , EStreamMarker_TypeName    = 0xFA
            , EStreamMarker_EndObject   = 0xFB
            , EStreamMarker_StartObject = 0xFC
            , EStreamMarker_Frame       = 0xFD

            // Cannot be over 255!
            , EStreamMarkers_Min        = 0xFA
            , EStreamMarkers_Max        = 0xFD
        };


        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TBinaryStream
        (
                    TBinStreamImpl* const   pstrmiToAdopt
        );

        ~TBinaryStream();


        // ---------------------------------------------------------------------
        //  Public, static methods
        // ---------------------------------------------------------------------
        static tCIDLib::TVoid CheckRelationship
        (
             const   TObject* const         pobjTest
             , const TClass&                clsTest
        );


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TBinaryStream& operator>>
        (
                    tCIDLib::TBoolean&      bToFill
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TCard1&        c1ToFill
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TCard2&        c2ToFill
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TCard4&        c4ToFill
        );

        TBinaryStream& operator>>
        (
                    EStreamMarkers&         eMarker
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TFloat4&       f4ToFill
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TFloat8&       f8ToFill
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TInt1&         i1ToFill
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TInt2&         i2ToFill
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TInt4&         i4ToFill
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TUInt&         uToFill
        );

        TBinaryStream& operator>>
        (
                    tCIDLib::TInt8&         i8ToFill
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TBoolean       bToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TCard1         c1ToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TCard2         c2ToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TCard4         c4ToWrite
        );

        TBinaryStream& operator<<
        (
            const   EStreamMarkers          eMarker
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TFloat4        f4ToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TFloat8&       f8ToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TInt1          i1ToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TInt2          i2ToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TInt4          i4ToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TUInt          uToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::TInt8          i8ToWrite
        );

        TBinaryStream& operator<<
        (
            const   tCIDLib::Tch* const     pszToWrite
        );


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bEndOfStream() const;

        TClass clsReadClassInfo();

        tCIDLib::TInt4 eReadEnum();

        tCIDLib::TFilePos fposCurPos() const;

        tCIDLib::TFilePos fposLogicalEnd() const;

        tCIDLib::TFilePos fposPhysicalEnd() const;

        tCIDLib::TFilePos fposSeekToEnd();

        tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        );

        tCIDLib::TVoid Reset();

        tCIDLib::TVoid ReadBuffer
        (
                    TMemBuf&                mbufTarget
            , const tCIDLib::TCard4         c4Count = kCIDLib::c4MaxCard
            ,       tCIDLib::TCard4&        c4Actual = NUL_TCard4
        );

        tCIDLib::TVoid ReadCard1Array
        (
                    tCIDLib::TCard1* const  ac1List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid ReadCard2Array
        (
                    tCIDLib::TCard2* const  ac2List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid ReadCard4Array
        (
                    tCIDLib::TCard4* const  ac4List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid ReadInt1Array
        (
                    tCIDLib::TInt1* const   ai1List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid ReadInt2Array
        (
                    tCIDLib::TInt2* const   ai2List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid ReadInt4Array
        (
                    tCIDLib::TInt4* const   ai4List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid ReadFloat4Array
        (
                    tCIDLib::TFloat4* const af4List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid ReadFloat8Array
        (
                    tCIDLib::TFloat8* const af8List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid ReadRawBuffer
        (
                    tCIDLib::TVoid* const   pBufToFill
            , const tCIDLib::TCard4         c4Count
            ,       tCIDLib::TCard4&        c4Actual = NUL_TCard4
        );

        tCIDLib::TVoid TruncateAtZero();

        tCIDLib::TVoid WriteBuffer
        (
            const   TMemBuf&                mbufSrc
            , const tCIDLib::TCard4         c4Count = kCIDLib::c4MaxCard
            ,       tCIDLib::TCard4&        c4Actual = NUL_TCard4
        );

        tCIDLib::TVoid WriteClassInfo
        (
            const   TClass&                 clsToWrite
        );

        tCIDLib::TVoid WriteCard1Array
        (
            const   tCIDLib::TCard1* const  ac1List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid WriteCard2Array
        (
            const   tCIDLib::TCard2* const  ac2List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid WriteCard4Array
        (
            const   tCIDLib::TCard4* const  ac4List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid WriteEnum
        (
            const   tCIDLib::TInt4          i4Enum
        );

        tCIDLib::TVoid WriteFloat4Array
        (
            const   tCIDLib::TFloat4* const af4List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid WriteFloat8Array
        (
            const   tCIDLib::TFloat8* const af8List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid WriteInt1Array
        (
            const   tCIDLib::TInt1* const   ai1List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid WriteInt2Array
        (
            const   tCIDLib::TInt2*  const  ai2List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid WriteInt4Array
        (
            const   tCIDLib::TInt4*  const  ai4List
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TVoid WriteRawBuffer
        (
            const   tCIDLib::TVoid* const   pBufToUse
            , const tCIDLib::TCard4         c4Count
            ,       tCIDLib::TCard4&        c4Actual = NUL_TCard4
        );


    protected   :
        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        TBinaryStream();


        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _AdoptImplObject
        (
                    TBinStreamImpl* const   pstrmiToAdopt
        );

        TBinStreamImpl& _strmiThis();

        const TBinStreamImpl& _strmiThis() const;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TBinaryStream(const TBinaryStream&);

        tCIDLib::TVoid operator=(const TBinaryStream&);


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __pstrmiThis
        //      This is a pointer to our stream implementation object. We
        //      adopt it and kill it when we die. We basically pass most of
        //      methods on to it or write them in terms of it.
        // ---------------------------------------------------------------------
        TBinStreamImpl* __pstrmiThis;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TBinaryStream,TObject)
};

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  TBinaryStream: Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TBinaryStream::bEndOfStream() const
{
    return _strmiThis().bEndOfStream();
}

inline tCIDLib::TInt4 TBinaryStream::eReadEnum()
{
    tCIDLib::TInt4  i4Enum;
    *this >> i4Enum;
    return i4Enum;
}

inline tCIDLib::TFilePos TBinaryStream::fposCurPos() const
{
    return _strmiThis().fposCurPos();
}

inline tCIDLib::TFilePos TBinaryStream::fposLogicalEnd() const
{
    return _strmiThis().fposLogicalEnd();
}

inline tCIDLib::TFilePos TBinaryStream::fposPhysicalEnd() const
{
    return _strmiThis().fposPhysicalEnd();
}

inline tCIDLib::TFilePos TBinaryStream::fposSeekToEnd()
{
    return _strmiThis().fposSeekToEnd();
}

inline tCIDLib::TFilePos
TBinaryStream::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    return _strmiThis().fposSkipForwardBy(c4SkipBy);
}

inline tCIDLib::TVoid TBinaryStream::Reset()
{
    _strmiThis().Reset();
}

inline tCIDLib::TVoid TBinaryStream::TruncateAtZero()
{
    _strmiThis().TruncateAtZero();
}

inline tCIDLib::TVoid TBinaryStream::WriteEnum(const tCIDLib::TInt4 i4Enum)
{
    *this << i4Enum;
}



// ----------------------------------------------------------------------------
//  These are template functions that greatly simplify the streaming in and
//  out of objects polymorphically. It keeps it all typesafe and makes sure
//  that the object gotten really supports the streaming operators. The
//  actual streaming operators are in CIDLib_Streamable.Hpp since they have
//  to be friends of the streamable mixin.
// ----------------------------------------------------------------------------
template <class T> tCIDLib::TVoid
PolymorphicRead(T*& pobjToFill, TBinaryStream& strmToReadFrom)
{
    // Init caller's pointer to 0 in case of failure
    pobjToFill = 0;

    //
    //  Stream in the class name for this object, and use that class
    //  name to create a new object, which we look at first as a 
    //  void pointer.
    //
    TClass              clsType(strmToReadFrom.clsReadClassInfo());
    TObject* const      pobjNew = clsType.pobjMakeNew();
    TJanitor<TObject>   janNewObj(pobjNew);

    //
    //  Make sure that the new type has a valid relationship with the
    //  type of the pointer we have to fill in. The class type we read
    //  in must be a class of or derived from the instantiation type.
    //
    TBinaryStream::CheckRelationship(pobjNew, T::clsThis);

    // Its ok so we can cast our pointer safely
    T* pobjTyped = (T*)pobjNew;

    //
    //  And now let it stream itself in. If this class does not support
    //  the streaming API, it will be caught here at compile time.
    //
    strmToReadFrom >> *pobjTyped;

    // Orphan the new object and give back the pointer
    janNewObj.Orphan();
    pobjToFill = pobjTyped;
}

template <class T> tCIDLib::TVoid
PolymorphicWrite(const T* const pobjToWrite, TBinaryStream& strmToWriteTo)
{
    strmToWriteTo.WriteClassInfo(pobjToWrite->clsIsA());
    strmToWriteTo << *pobjToWrite;
}
