//
// NAME: CIDLib_BinaryStream.Cpp
//
// DESCRIPTION: 
//
//  This module implements a simple derivative of the basic stream class.
//  This one is a binary mode stream, and is the class in terms of which the
//  mixin class, MStreamable, is written. MStreamable's _StreamFrom()
//  and _StreamTo() take a binary stream (or derived) class.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/26/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard members macros
// ----------------------------------------------------------------------------
RTTIData(TBinaryStream,TObject)
RTTIData(TBinStreamImpl,TObject)


// ----------------------------------------------------------------------------
//  TBinaryStream: Public static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TBinaryStream::CheckRelationship(   const   TObject* const  pobjTest
                                    , const TClass&         clsTest)
{
    if (!pobjTest->bIsDescendantOf(clsTest))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NotDerivedClass
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , pobjTest->clsIsA()
            , clsTest
        );
    }
}


// ----------------------------------------------------------------------------
//  TBinaryStream: Constructors and destructors
// ----------------------------------------------------------------------------

TBinaryStream::TBinaryStream() :

    __pstrmiThis(0)
{
}

TBinaryStream::TBinaryStream(TBinStreamImpl* const pstrmiToAdopt) :

    __pstrmiThis(pstrmiToAdopt)
{
}

TBinaryStream::~TBinaryStream()
{
    delete __pstrmiThis;
    __pstrmiThis = 0;
}


// ----------------------------------------------------------------------------
//  TBinaryStream: Public operators
// ----------------------------------------------------------------------------

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TBoolean& bToFill)
{
    ReadRawBuffer(&bToFill, sizeof(bToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TCard1& c1ToFill)
{
    ReadRawBuffer(&c1ToFill, sizeof(c1ToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TCard2& c2ToFill)
{
    ReadRawBuffer(&c2ToFill, sizeof(c2ToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TCard4& c4ToFill)
{
    ReadRawBuffer(&c4ToFill, sizeof(c4ToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(EStreamMarkers& eMarker)
{
    tCIDLib::TCard1 c1Tmp;
    ReadRawBuffer(&c1Tmp, sizeof(c1Tmp));
    if ((c1Tmp < EStreamMarkers_Min) || (c1Tmp > EStreamMarkers_Max))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_BadMarker
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_StreamFormat
            , TCardinal(c1Tmp)
        );
    }
    eMarker = static_cast<EStreamMarkers>(c1Tmp);
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TInt1& i1ToFill)
{
    ReadRawBuffer(&i1ToFill, sizeof(i1ToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TInt2& i2ToFill)
{
    ReadRawBuffer(&i2ToFill, sizeof(i2ToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TInt4& i4ToFill)
{
    ReadRawBuffer(&i4ToFill, sizeof(i4ToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TUInt& uToFill)
{
    ReadRawBuffer(&uToFill, sizeof(uToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TInt8& i8ToFill)
{
    ReadRawBuffer(&i8ToFill, sizeof(i8ToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TFloat4& f4ToFill)
{
    ReadRawBuffer(&f4ToFill, sizeof(f4ToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator>>(tCIDLib::TFloat8& f8ToFill)
{
    ReadRawBuffer(&f8ToFill, sizeof(f8ToFill));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TBoolean bVal)
{
    WriteRawBuffer(&bVal, sizeof(bVal));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TCard1 c1Val)
{
    WriteRawBuffer(&c1Val, sizeof(c1Val));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TCard2 c2Val)
{
    WriteRawBuffer(&c2Val, sizeof(c2Val));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TCard4 c4Val)
{
    WriteRawBuffer(&c4Val, sizeof(c4Val));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const EStreamMarkers eMarker)
{
    tCIDLib::TCard1 c1Tmp = static_cast<tCIDLib::TCard1>(eMarker);
    WriteRawBuffer(&c1Tmp, sizeof(c1Tmp));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TFloat4 f4Val)
{
    WriteRawBuffer(&f4Val, sizeof(f4Val));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TFloat8& f8Val)
{
    WriteRawBuffer(&f8Val, sizeof(f8Val));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TInt1 i1Val)
{
    WriteRawBuffer(&i1Val, sizeof(i1Val));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TInt2 i2Val)
{
    WriteRawBuffer(&i2Val, sizeof(i2Val));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TInt4 i4Val)
{
    WriteRawBuffer(&i4Val, sizeof(i4Val));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TUInt uVal)
{
    WriteRawBuffer(&uVal, sizeof(uVal));
    return *this;
}

TBinaryStream& TBinaryStream::operator<<(const tCIDLib::TInt8 i8Val)
{
    WriteRawBuffer(&i8Val, sizeof(i8Val));
    return *this;
}

TBinaryStream&
TBinaryStream::operator<<(const tCIDLib::Tch* const pszToWrite)
{
    WriteRawBuffer(pszToWrite, TRawStr::c4StrLen(pszToWrite));
    return *this;
}


// ----------------------------------------------------------------------------
//  TBinaryStream: Public, non-virtual methods
// ----------------------------------------------------------------------------

TClass TBinaryStream::clsReadClassInfo()
{
    //
    //  First we need to read the type string in. It will start with the
    //  eTypeName marker in a byte, followed by a char count in a Card2, then
    //  that many chars of the name. The null is not stored.
    //
    EStreamMarkers eMarker;
    *this >> eMarker;
    if (eMarker != EStreamMarker_TypeName)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoTypeMarker
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_StreamFormat
            , TInteger(eMarker)
        );
    }

    //
    //  Next should be the number of chars of the name. We know that type
    //  names are limited to 256 chars, so its just stored in a short.
    //
    tCIDLib::TCard1 c1Count(0);
    *this >> c1Count;
    if (!c1Count)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_BadNameCount
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_StreamFormat
        );
    }

    //
    //  Read in the number of bytes required and add a null terminator
    //  to it. By not passing an 'actual bytes read' parm to the raw buf
    //  read we insure an exception if there is not enough data.
    //
    tCIDLib::Tsch szName[kCIDLib::c1MaxCard+1];
    ReadRawBuffer(szName, c1Count);

    // Put in the nul terminator
    szName[c1Count] = 0;

    // Convert it to UNICode now
    tCIDLib::Tch szUNIName[kCIDLib::c1MaxCard+1];
    TRawStr::pszConvert(szName, szUNIName, kCIDLib::c1MaxCard);

    // Hash it so we can compare with the stored hash
    tCIDLib::THashVal hshNewName = TRawStr::hshHashStr
    (
        szUNIName
        , kCIDLib::c4ClassModulus
    );

    //
    //  Read in the hash that was stored after the string bytes. We know
    //  its always less than the modulus, so we stored it in a byte.
    //
    tCIDLib::TCard1 c1StoredHash(0);
    *this >> c1StoredHash;

    //
    //  If it does not equal the hash we just generated, then we obviously
    //  are not dealing with a full deck here.
    //
    if (hshNewName != tCIDLib::THashVal(c1StoredHash))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_BadStoredHash
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_StreamFormat
            , TCardinal(c1StoredHash)
            , TCardinal(hshNewName)
        );
    }

    //
    //  Return a copy of the class object for this class, if any. If not,
    //  then this logs an exception.
    //
    return TClass::clsForType(szUNIName);
}

tCIDLib::TVoid
TBinaryStream::ReadBuffer(          TMemBuf&            mbufTarget
                            , const tCIDLib::TCard4     c4Count
                            ,       tCIDLib::TCard4&    c4Actual)
{
    tCIDLib::TCard4 c4ActualCount = c4Count;

    if (c4Count == kCIDLib::c4MaxCard)
        c4ActualCount = mbufTarget.c4Size();

    // Ask the implementation to read us a buffer.
    _strmiThis().ReadBytes
    (
        (tCIDLib::TCard1*)mbufTarget.pData()
        , c4ActualCount
        , c4Actual
    );
}

tCIDLib::TVoid
TBinaryStream::ReadRawBuffer(       tCIDLib::TVoid* const   pBufToFill
                            , const tCIDLib::TCard4         c4Count
                            ,       tCIDLib::TCard4&        c4Actual)
{
    // Ask the implementation to read us a buffer.
    _strmiThis().ReadBytes
    (
        (tCIDLib::TCard1*)pBufToFill
        , c4Count
        , c4Actual
    );
}

tCIDLib::TVoid
TBinaryStream::ReadCard1Array(          tCIDLib::TCard1* const  ac1List
                                , const tCIDLib::TCard4         c4Count)
{
    ReadRawBuffer(ac1List, c4Count);
}

tCIDLib::TVoid
TBinaryStream::ReadCard2Array(          tCIDLib::TCard2* const  ac2List
                                , const tCIDLib::TCard4         c4Count)
{
    ReadRawBuffer(ac2List, sizeof(tCIDLib::TCard2)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::ReadCard4Array(          tCIDLib::TCard4* const  ac4List
                                , const tCIDLib::TCard4         c4Count)
{
    ReadRawBuffer(ac4List, sizeof(tCIDLib::TCard4)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::ReadInt1Array(           tCIDLib::TInt1* const   ai1List
                                , const tCIDLib::TCard4         c4Count)
{
    ReadRawBuffer(ai1List, c4Count);
}

tCIDLib::TVoid
TBinaryStream::ReadInt2Array(           tCIDLib::TInt2* const   ai2List
                                , const tCIDLib::TCard4         c4Count)
{
    ReadRawBuffer(ai2List, sizeof(tCIDLib::TInt2)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::ReadInt4Array(           tCIDLib::TInt4* const   ai4List
                                , const tCIDLib::TCard4         c4Count)
{
    ReadRawBuffer(ai4List, sizeof(tCIDLib::TInt4)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::ReadFloat4Array(         tCIDLib::TFloat4* const af4List
                                , const tCIDLib::TCard4         c4Count)
{
    ReadRawBuffer(af4List, sizeof(tCIDLib::TFloat4)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::ReadFloat8Array(         tCIDLib::TFloat8* const af8List
                                , const tCIDLib::TCard4         c4Count)
{
    ReadRawBuffer(af8List, sizeof(tCIDLib::TFloat8)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::WriteBuffer( const   TMemBuf&            mbufSrc
                            , const tCIDLib::TCard4     c4Count
                            ,       tCIDLib::TCard4&    c4Actual)
{
    tCIDLib::TCard4     c4ActualCount = c4Count;

    if (c4Count == kCIDLib::c4MaxCard)
        c4ActualCount = mbufSrc.c4Size();

    // Ask the implementation to read us a buffer.
    _strmiThis().WriteBytes
    (
        (tCIDLib::TCard1*)mbufSrc.pData()
        , c4ActualCount
        , c4Actual
    );
}

tCIDLib::TVoid
TBinaryStream::WriteCard1Array( const   tCIDLib::TCard1* const  ac1List
                                , const tCIDLib::TCard4         c4Count)
{
    WriteRawBuffer(ac1List, c4Count);
}

tCIDLib::TVoid
TBinaryStream::WriteCard2Array( const   tCIDLib::TCard2* const  ac2List
                                , const tCIDLib::TCard4         c4Count)
{
    WriteRawBuffer(ac2List, sizeof(tCIDLib::TCard2)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::WriteCard4Array( const   tCIDLib::TCard4* const  ac4List
                                , const tCIDLib::TCard4         c4Count)
{
    WriteRawBuffer(ac4List, sizeof(tCIDLib::TCard4)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::WriteInt1Array(  const   tCIDLib::TInt1* const   ai1List
                                , const tCIDLib::TCard4         c4Count)
{
    WriteRawBuffer(ai1List, c4Count);
}

tCIDLib::TVoid
TBinaryStream::WriteInt2Array(  const   tCIDLib::TInt2*  const  ai2List
                                , const tCIDLib::TCard4         c4Count)
{
    WriteRawBuffer(ai2List, sizeof(tCIDLib::TInt2)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::WriteInt4Array(  const   tCIDLib::TInt4* const   ai4List
                                , const tCIDLib::TCard4         c4Count)
{
    WriteRawBuffer(ai4List, sizeof(tCIDLib::TInt4)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::WriteFloat4Array(const   tCIDLib::TFloat4* const af4List
                                , const tCIDLib::TCard4         c4Count)
{
    WriteRawBuffer(af4List, sizeof(tCIDLib::TFloat4)*c4Count);
}

tCIDLib::TVoid
TBinaryStream::WriteFloat8Array(const   tCIDLib::TFloat8* const af8List
                                , const tCIDLib::TCard4         c4Count)
{
    WriteRawBuffer(af8List, sizeof(tCIDLib::TFloat8)*c4Count);
}


tCIDLib::TVoid
TBinaryStream::WriteClassInfo(const TClass& clsToWrite)
{
    // Insure that the class name is valid
    if (!clsToWrite.bNameValid())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InvalidName
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }

    // Write out the typename stream marker
    *this << EStreamMarker_TypeName;

    // Now get the length of the type name text
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(clsToWrite.pszClassName());

    #if CID_DEBUG_ON
    if (c4Len > kCIDLib::c1MaxCard)
    {
        facCIDLib.LogMsg
        (
            __FILE__
            , __LINE__
            , L"The class name is too long"
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
        );
    }
    #endif

    //
    //  Write out the length of the name, also in a byte. We know this
    //  is legal cause we checked the value above.
    //
    tCIDLib::TCard1 c1Len = tCIDLib::TCard1(c4Len);
    *this << c1Len;

    //
    //  Convert the class name to ASCII text. Since class names have to
    //  be ASCII anyway, this cuts the storage needs in half.
    //
    tCIDLib::Tsch*  pszAsciiName = TRawStr::pszConvert(clsToWrite.pszClassName());
    THeapJanitor    janName(pszAsciiName);

    //
    //  Now write out the bytes of the name. By not providing an 'actual
    //  written' parameter, we insure an exception if we don't write them
    //  all. We don't store the nul.
    //
    WriteRawBuffer(pszAsciiName, c1Len);

    //
    //  Now write out the 1st byte of the hash value. We know that its
    //  below the modulus, which will fit easily in one byte.
    //
    tCIDLib::TCard1 c1Hash = tCIDLib::TCard1(clsToWrite.hshInternal());
    *this << c1Hash;
}

tCIDLib::TVoid
TBinaryStream::WriteRawBuffer(  const   tCIDLib::TVoid* const   pBufToUse
                                , const tCIDLib::TCard4         c4Count
                                ,       tCIDLib::TCard4&        c4Actual)
{
    // Ask the implementation to write us a buffer.
    _strmiThis().WriteBytes
    (
        (tCIDLib::TCard1*)pBufToUse
        , c4Count
        , c4Actual
    );
}


// ----------------------------------------------------------------------------
//  TBinaryStream: Protected, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TBinaryStream::_AdoptImplObject(TBinStreamImpl* const pstrmiToAdopt)
{
    if (__pstrmiThis)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_ImplAlreadySet
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , clsIsA()
        );
    }
    __pstrmiThis = pstrmiToAdopt;
}

inline TBinStreamImpl& TBinaryStream::_strmiThis()
{
    if (!__pstrmiThis)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoImplementation
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , clsIsA()
        );
    }
    return *__pstrmiThis;
}

inline const TBinStreamImpl& TBinaryStream::_strmiThis() const
{
    if (!__pstrmiThis)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoImplementation
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , clsIsA()
        );
    }
    return *__pstrmiThis;
}
