//
// NAME: CIDLib_BaseLock.Cpp
//
// DESCRIPTION:
//
//  This module implements a very simple class, IBaseLock, that is used for
//  ad hoc synchronization purposes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/25/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TBaseLock, TObject)


// ----------------------------------------------------------------------------
//  Local, static data
//
//  __pcrsLock
//      This is a pointer to a critical section object. It is not actually
//      allocated until someone tries to use the lock. An atomic compare and
//      swap is used to create it and set the pointer.
// ----------------------------------------------------------------------------
static TCriticalSection*    __pcrsLock = 0;


// ----------------------------------------------------------------------------
//   CLASS: TBaseLock
//  PREFIX: lock
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TBaseLock: Contructors and Destructors
// ----------------------------------------------------------------------------

TBaseLock::TBaseLock()
{
    //
    //  See if the lock exists. If not, then we need to gen it up. We use
    //  an atomic swap to do it so there is no chance of a leak, but we might
    //  make one redundant object and have to destroy it if someone has already
    //  got it set while we were trying to get ours destructed.
    //
    if (!__pcrsLock)
    {
        // Not there so create one
        TCriticalSection* pcrsCandidate = new TCriticalSection;

        //
        //  Now do the compare and exchange. If we lose, then delete our
        //  candidate object. If we win, then ours is in there and we are
        //  done. The original value is returned, so if the return is 0, then
        //  we won. Otherwise, we have to toast our candidate.
        //
        //  We use a templatized version here so we don't have to a bunch of
        //  casting.
        //
        if (TRawMem::pCompareAndExchangePtr
        (
            __pcrsLock
            , pcrsCandidate
            , (TCriticalSection*)0))
        {
            delete pcrsCandidate;
        }
    }

    // Ok, we know that the lock is created, so enter the critical section
    __pcrsLock->Enter();
}

TBaseLock::~TBaseLock()
{
    // And exit the critical section
    __pcrsLock->Exit();
}
