//
// NAME: CIDLib_BaseAudio.Cpp
//
// DESCRIPTION:
//
//  TAudio is a simple sound generating class that is used to abstract the
//  way that CIDLib makes sounds. Its default implementation just uses the
//  system speaker.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 06/04/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//

// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TAudio,TObject)


// -----------------------------------------------------------------------------
//   CLASS: TAudio
//  PREFIX: audio
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TAudio: Public, virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TAudio::AudioCue(const tCIDLib::EAudioCues eCue) const
{
    switch (eCue)
    {
        case tCIDLib::EAudioCue_Alert          :
            //
            //  Do a high-low pulse kind of like a british siren
            //
            Beep(880, 50);
            Beep(440, 50);
            break;

        case tCIDLib::EAudioCue_Bell           :
            //
            //  Do a high tone followed by a fast series of slightly lower
            //  oscillating frequencies.
            //
            Beep(1000, 10);
            Beep(990, 50);
            Beep(980, 50);
            break;

        case tCIDLib::EAudioCue_Chime          :
            //
            //  Do a short medium high tone followed by a longer slightly
            //  lower tone.
            //
            Beep(880, 10);
            Beep(850, 50);
            break;

        case tCIDLib::EAudioCue_Error          :
            //
            //  Do two hard pulses with a small pause. Put them at a medium
            //  high frequency so that system speakers will carry them well.
            //
            Beep(500, 100);
            TKrnlThread::Sleep(25);
            Beep(500, 100);
            break;

        case tCIDLib::EAudioCue_HiPulse        :
        case tCIDLib::EAudioCue_Information    :
            //
            //  Do a short pulse at 880
            //
            Beep(880, 50);
            break;

        case tCIDLib::EAudioCue_LowPulse       :
            Beep(440, 75);
            break;

        case tCIDLib::EAudioCue_MedPulse       :
            Beep(660, 60);
            break;

        case tCIDLib::EAudioCue_Rising         :
            Beep(400, 50);
            Beep(440, 50);
            Beep(480, 50);
            Beep(520, 50);
            Beep(580, 50);
            break;

        case tCIDLib::EAudioCue_Sinking        :
            Beep(580, 50);
            Beep(520, 50);
            Beep(480, 50);
            Beep(440, 50);
            Beep(400, 50);
            break;

        case tCIDLib::EAudioCue_Question       :
            //
            //  Do three pulses moving upward in frequency
            //
            Beep(880, 25);
            Beep(900, 25);
            Beep(920, 25);
            break;

        case tCIDLib::EAudioCue_Click         :
            //
            //  Do a very short tone at a high frequency
            //
            Beep(880,10);
            break;

        case tCIDLib::EAudioCue_Warning        :
            //
            //  Do two short pulses at 880 with a small pause between.
            //
            Beep(880, 25);
            TKrnlThread::Sleep(25);
            Beep(880, 25);
            break;


        default                     :
            // Something is wrong, so log a message
            TStringStream   strmTmp(64);
            strmTmp << eCue;
            facCIDLib.LogMsg
            (
                __FILE__
                , __LINE__
                , L"Unknown audio cue value: %(1)"
                , tCIDLib::ESev_Warning
                , tCIDLib::EClass_Internal
                , strmTmp.strData()
            );
            break;
    }
}


tCIDLib::TVoid
TAudio::Beep(   const   tCIDLib::TCard4 c4Frequency
                , const tCIDLib::TCard4 c4Duration) const
{
    TKrnlThread::BeepSpeaker(c4Frequency, c4Duration);
}
