//
// NAME: CIDLib_BaseArray.Hpp
//
// DESCRIPTION: 
//
//  This header defines a template class that is used to create arrays of
//  fundamental data types. This is a pretty rare need since CIDLib provides
//  an environment that discourages such things, but its used some internally
//  and might be needed outside in some rare circumstances.
//
//  ONLY use it for fundamental types, not objects or structures.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 07/22/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TBaseArray
//  PREFIX: ba
// ----------------------------------------------------------------------------
template <class ElemType> class TBaseArray :

    public TObject, public MStreamable, public MDuplicable
{
    public      :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TBaseArray
        (
            const   tCIDLib::TCard4         c4ElemCount
            , const ElemType&               InitVal = (ElemType)0) :

            __bStoreCount(kCIDLib::True)
            , __c4ElemCount(c4ElemCount)
            , __paTData(0)
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;
                *((TClass*)&clsThis) = TClass(L"TBaseArray");
            }

            if (!c4ElemCount)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcBAry_InvalidCount
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_BadParms
                );
            }

            // Allocate the buffer
            __paTData = new ElemType[c4ElemCount];

            // And init the data values
            SetAll(InitVal);
        }

        TBaseArray(const TBaseArray<ElemType>& baToCopy) :

            __bStoreCount(kCIDLib::True)
            , __c4ElemCount(baToCopy.__c4ElemCount)
            , __paTData(0)
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;
                *((TClass*)&clsThis) = TClass(L"TBaseArray");
            }

            __paTData = new ElemType[__c4ElemCount];
            TRawMem::CopyMemBuf
            (
                __paTData
                , baToCopy.__paTData
                , __c4ElemCount * sizeof(ElemType)
            );
        }

        ~TBaseArray()
        {
            delete [] __paTData;
            __paTData = 0;
        }


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TBaseArray<ElemType>& operator=(const TBaseArray<ElemType>& baToAssign)
        {
            if (this == &baToAssign)
                return *this;

            if (__c4ElemCount != baToAssign.__c4ElemCount)
            {
                delete __paTData;
                __c4ElemCount = baToAssign.__c4ElemCount;
                __paTData = new ElemType[__c4ElemCount];
            }

            TRawMem::CopyMemBuf
            (
                __paTData
                , baToAssign.__paTData
                , __c4ElemCount*sizeof(ElemType)
            );
            return *this;
        }

        const ElemType operator[](const tCIDLib::TCard4 c4Index) const
        {
            __CheckIndex(c4Index, __LINE__);
            return __paTData[c4Index];
        }

        ElemType& operator[](const tCIDLib::TCard4 c4Index)
        {
            __CheckIndex(c4Index, __LINE__);
            return __paTData[c4Index];
        }

        tCIDLib::TBoolean operator==(const TBaseArray<ElemType>& baToTest) const
        {
            if (__c4ElemCount != baToTest.__c4ElemCount)
                return kCIDLib::False;

            for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4ElemCount; c4Ind++)
            {
                if (__paTData[c4Ind] != baToTest.__paTData[c4Ind])
                    return kCIDLib::False;
            }
            return kCIDLib::True;
        }

        tCIDLib::TBoolean operator!=(const TBaseArray<ElemType>& baToTest) const
        {
            return !operator==(baToTest);
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bStoreCount() const
        {
            return __bStoreCount;
        }

        tCIDLib::TBoolean bStoreCount(const tCIDLib::TBoolean bNew)
        {
            __bStoreCount = bNew;
            return __bStoreCount;
        }

        tCIDLib::TCard4 c4ElementCount() const
        {
            return __c4ElemCount;
        }

        tCIDLib::TVoid Exchange(const   tCIDLib::TCard4 c4First
                                , const tCIDLib::TCard4 c4Second)
        {
            __CheckIndex(c4First, __LINE__);
            __CheckIndex(c4Second, __LINE__);

            ElemType TTmp = __paTData[c4First];
            __paTData[c4First] = __paTData[c4Second];
            __paTData[c4Second] = TTmp;
        }

        tCIDLib::TVoid Number(const ElemType& TOffset = ElemType(0))
        {
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4ElemCount; c4Index++)
                __paTData[c4Index] = ElemType(c4Index + TOffset);
        }

        tCIDLib::TVoid Reallocate(  const   tCIDLib::TCard4     c4NewSize
                                    , const ElemType            InitVal = (ElemType)0)
        {
            // If the new size is different, then reallocate
            if (__c4ElemCount != c4NewSize)
            {
                delete __paTData;
                __c4ElemCount = c4NewSize;
                __paTData = new ElemType[__c4ElemCount * sizeof(ElemType)];
            }

            // And init the new values
            SetAll(InitVal);
        }

        tCIDLib::TVoid RotateDown(const tCIDLib::TCard4 c4Count)
        {
            tCIDLib::TCard4  c4Actual = c4Count % __c4ElemCount;

            if (!c4Actual || (__c4ElemCount == 1))
                return;

            // Make c4Actual passes through the array
            for (tCIDLib::TCard4 c4Rotate = 0; c4Rotate < c4Actual; c4Rotate++)
            {
                ElemType TSave = __paTData[0];

                for (tCIDLib::TCard4 c4Ind = 1; c4Ind < __c4ElemCount; c4Ind++)
                    __paTData[c4Ind-1] = __paTData[c4Ind];

                __paTData[__c4ElemCount-1] = TSave;
            }
        }

        tCIDLib::TVoid RotateUp(const tCIDLib::TCard4 c4Count)
        {
            tCIDLib::TCard4  c4Actual = c4Count % __c4ElemCount;

            if (!c4Actual || (__c4ElemCount == 1))
                return;

            // Make c4Actual passes through the array
            for (tCIDLib::TCard4 c4Rotate = 0; c4Rotate < c4Actual; c4Rotate++)
            {
                ElemType TSave = __paTData[__c4ElemCount-1];

                tCIDLib::TCard4 c4Ind;
                for (c4Ind = __c4ElemCount-1; c4Ind >= 1; c4Ind--)
                    __paTData[c4Ind] = __paTData[c4Ind-1];

                __paTData[0] = TSave;
            }
        }

        tCIDLib::TVoid SetAll(const ElemType& InitVal)
        {
            if (!InitVal)
            {
                // The init value is 0, so we can just set the whole buffer
                TRawMem::SetMemBuf
                (
                    __paTData
                    , tCIDLib::TCard1(0)
                    , __c4ElemCount
                );
            }
             else if (sizeof(ElemType) == 1)
            {
                // The buffer is 1 byte, so we can set it all at once
                TRawMem::SetMemBuf
                (
                    __paTData
                    , tCIDLib::TCard1(InitVal)
                    , __c4ElemCount
                );
            }
             else
            {
                // Do it the hard way
                tCIDLib::TCard4 c4Ind;
                for (c4Ind = 0; c4Ind < __c4ElemCount; c4Ind++)
                    __paTData[c4Ind] = InitVal;
            }
        }

        tCIDLib::TVoid Sort()
        {
            // Just use the CIDLib quick sort
            CIDSort::QSortArray(__paTData, 0, __c4ElemCount-1);
        }

        tCIDLib::TVoid ZeroAll()
        {
            TRawMem::SetMemBuf
            (
                __paTData
                , tCIDLib::TCard1(0)
                , __c4ElemCount * sizeof(ElemType)
            );
        }


    protected       :
        // --------------------------------------------------------------------
        //  Protected constructors and operators
        // --------------------------------------------------------------------
        TBaseArray() {}


        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo(TTextStream& strmToWriteTo) const
        {
            strmToWriteTo   << L"Type: " << clsIsA()
                            << L", Elements:" << __c4ElemCount;
        }

        tCIDLib::TVoid _StreamFrom(TBinaryStream& strmToReadFrom)
        {
            tCIDLib::TCard4 c4NewCount = __c4ElemCount;

            // Pull the element count out if needed
            if (__bStoreCount)
                strmToReadFrom >> c4NewCount;

            if (__c4ElemCount != c4NewCount)
            {
                __c4ElemCount = c4NewCount;
                delete __paTData;
                __paTData = 0;
                __paTData = new ElemType[__c4ElemCount];
            }

            if (sizeof(ElemType) == 1)
            {
                strmToReadFrom.ReadRawBuffer(__paTData, __c4ElemCount);
            }
             else
            { 
                tCIDLib::TCard4 c4Index;
                for (c4Index = 0; c4Index < __c4ElemCount; c4Index++)
                    strmToReadFrom >> __paTData[c4Index];
            }
        }

        tCIDLib::TVoid _StreamTo(TBinaryStream& strmToWriteTo) const
        {
            if (__bStoreCount)
                strmToWriteTo << __c4ElemCount;

            if (sizeof(__paTData[0]) == 1)
            {
                strmToWriteTo.WriteRawBuffer(__paTData, __c4ElemCount);
            }
             else
            {
                tCIDLib::TCard4 c4Index;
                for (c4Index = 0; c4Index < __c4ElemCount; c4Index++)
                    strmToWriteTo << __paTData[c4Index];
            }
        }


    private         :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckIndex
        (
            const   tCIDLib::TCard4         c4IndexToCheck
            , const tCIDLib::TCard4         c4Line)   const
        {
            if (c4IndexToCheck >= __c4ElemCount)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , c4Line
                    , kCIDErrs::errcBAry_IndexError
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_BadParms
                    , clsThis
                    , TCardinal(__c4ElemCount-1)
                    , TCardinal(c4IndexToCheck)
                );
            }
        }


        // --------------------------------------------------------------------
        //  Private data
        //
        //  __bStoreCount
        //      Indicates whether this array should include its element count
        //      when flattened. General it should, so the default is eTrue;
        //      but, when dealing with 3rd party files and such it is
        //      sometimes necessary not to.
        //
        //  __c4ElemCount
        //      This is the count of elements in this array.
        //
        //  __paTData
        //      This is the allocated array of c4Count values. The count is
        //      provided in the constructor.
        // --------------------------------------------------------------------
        tCIDLib::TBoolean       __bStoreCount;
        tCIDLib::TCard4         __c4ElemCount;
        ElemType*               __paTData;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a bag of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;



        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TBaseArray<ElemType>,TObject)
        DefPolyDup(TBaseArray<ElemType>)
};

template <class ElemType> tCIDLib::TBoolean TBaseArray<ElemType>::__bFirstOfType = kCIDLib::False;
template <class ElemType> const TClass TBaseArray<ElemType>::clsThis;

#pragma pack(pop)
