//
// NAME: CIDLib_Bag.Hpp
//
// DESCRIPTION:
//
//  This is the header for the bag collection template. It implements the TBag
//  class template, which provides unordered storage of its elements by value.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 01/30/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Forward reference the bag and bag cursor classes
// ----------------------------------------------------------------------------
template <class TElem> class TBagCursor;
template <class TElem> class TBag;


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TBagNode
//  PREFIX: node
// ----------------------------------------------------------------------------
template <class TElem> class TBagNode : public TDLstNode
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TBagNode(const TElem& objData) :

            __objData(objData)
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TBagNodeOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        ~TBagNode() {}


    protected   :
        // --------------------------------------------------------------------
        //  The bag class and its cursor are our friends
        // --------------------------------------------------------------------
        friend class TBag<TElem>;
        friend class TBagCursor<TElem>;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TBagNode();

        TBagNode(const TBagNode<TElem>&);

        tCIDLib::TVoid operator=(const TBagNode<TElem>&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __objData
        //      The user data object.
        // --------------------------------------------------------------------
        TElem   __objData;


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TBagNode<TElem>,TDLstNode)
};

template <class TElem> tCIDLib::TBoolean TBagNode<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TBagNode<TElem>::clsThis;



// ----------------------------------------------------------------------------
//   CLASS: TBag
//  PREFIX: bag
// ----------------------------------------------------------------------------
template <class TElem> class TBag : public TCollection<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Nested typedefs for cursor and node types used by a bag.
        // --------------------------------------------------------------------
        typedef class TBagCursor<TElem>     TCursor;
        typedef TBagNode<TElem>             TNode;


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TBag(const tCIDLib::EMTStates eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __llstBag(0)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }

        TBag(   const   tCIDLib::TCard4     c4MaxItems
                , const tCIDLib::EMTStates  eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __llstBag(c4MaxItems)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }

        ~TBag() {}


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Add(const TElem& objNew)
        {
            TLocker lockBag(this);
            __llstBag.AppendNode(new TNode(objNew));
            _c4IncSerialNum();
        }

        tCIDLib::TBoolean bFlushCur()
        {
            TLocker lockBag(this);

            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Save the next node
            TNode* const pnodeNext = (TNode*)__pnodeCur->pnodeNext();

            // Now flush the iterator node
            __llstBag.FlushNode(__pnodeCur);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // Put the next node back into the iterator
            __pnodeCur = pnodeNext;

            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bIsEmpty() const
        {
            TLocker lockBag(this);
            tCIDLib::TBoolean bRet = __llstBag.bIsEmpty();
            return bRet;
        }

        tCIDLib::TBoolean bNext()
        {
            TLocker lockBag(this);
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = (TNode*)__pnodeCur->pnodeNext();
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bResetIter()
        {
            TLocker lockBag(this);
            __pnodeCur = (TNode*)__llstBag.pnodeHead();
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            TLocker lockBag(this);
            return __llstBag.c4ElemCount();
        }

        tCIDLib::TCard4 c4MaxElemCount() const
        {
            TLocker lockBag(this);
            tCIDLib::TCard4 c4Ret = __llstBag.c4MaxElemCount();
            return c4Ret;
        }

        tCIDLib::TVoid Flush()
        {
            TLocker lockBag(this);
            __pnodeCur = 0;
            __llstBag.Flush();
            _c4IncSerialNum();
        }

        TElem& objCur() const
        {
            TLocker lockBag(this);
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }

        TElemCursor<TElem>* pcursNew() const
        {
            TLocker lockBag(this);
            return new TCursor(this);
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid FlushAt(TCursor& cursAt)
        {
            TLocker lockBag(this);

            // Make sure the cursor belongs to this bag
            if (!cursAt.bIsCursoring(*this))
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_NotMyCursor
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_BadParms
                    , cursAt.clsIsA()
                    , clsIsA()
                );
            }

            // Get the node to flush
            TNode* pnodeToFlush = cursAt.pnodeCur();

            // Make sure the cursor's valid
            if (!pnodeToFlush)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , cursAt.clsIsA()
                );
            }

            // Bump up the cursor past this node
            cursAt.bNext();

            // If delete node is the internal iterator node, move it up also
            if (pnodeToFlush == __pnodeCur)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            // Now flush that node
            __llstBag.FlushNode(pnodeToFlush);

            // And bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // But we kept the passed cursor up to date by bumping it up
            cursAt._c4SerialNum(c4SerialNum());
        }


    protected   :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TBagCursor<TElem>;

        friend tCIDLib::TBoolean bCompCollections
        (
            const   TBag<TElem>&    col1
            , const TBag<TElem>&    col2
            , const TObjEq<TElem>&  compToUse
        );

        friend TBag<TElem>* pDupCollection
        (
            const   TBag<TElem>&            colToDup
        );


        // --------------------------------------------------------------------
        //  Hidden constructors
        // --------------------------------------------------------------------
        TBag(const TBag<TElem>& colToCopy) :

            TCollection<TElem>(colToCopy)
            , __llstBag(colToCopy.c4MaxElemCount())
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        const TDLinkedList& _llstBag() const
        {
            return __llstBag;
        }

        TDLinkedList& _llstBag()
        {
            return __llstBag;
        }


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators.
        // --------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TBag<TElem>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString strTmp(L"TBagOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __llstBag
        //      This is a doubly linked list that is the storage for the bag
        //      items. Bags are bi-directional, so we use the doubly linked
        //      list.
        //
        //  __pnodeCur
        //      This is the current node, for the writable style iteration
        //      that is built into the node.
        // --------------------------------------------------------------------
        TDLinkedList            __llstBag;
        TNode*                  __pnodeCur;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a bag of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TBag<TElem>,TCollection<TElem>)
};

template <class TElem> tCIDLib::TBoolean TBag<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TBag<TElem>::clsThis;



// ----------------------------------------------------------------------------
//   CLASS: TBagCursor
//  PREFIX: curs
// ----------------------------------------------------------------------------
template <class TElem> class TBagCursor : public TBiElemCursor<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TBagCursor(const TBag<TElem>* const pcolToCursor) :

            TBiElemCursor<TElem>(pcolToCursor->c4SerialNum())
            , __pllstCursoring(&pcolToCursor->_llstBag())
            , __pcolCursoring(pcolToCursor)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();

            __pnodeCur = (TBagNode<TElem>*)__pllstCursoring->pnodeHead();
        }

        TBagCursor(const TBagCursor<TElem>& cursToCopy) :

            TBiElemCursor<TElem>(cursToCopy)
            , __pllstCursoring(cursToCopy.__pllstCursoring)
            , __pcolCursoring(cursToCopy.__pcolCursoring)
            , __pnodeCur(cursToCopy.__pnodeCur)
        {
            __CheckFirstOfType();
        }

        ~TBagCursor() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TBagCursor<TElem>& operator=(const TBagCursor<TElem>& cursToAssign)
        {
            if (this == &cursToAssign)
                return *this;

            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            TParent::operator=(cursToAssign);
            __pllstCursoring = cursToAssign.__pllstCursoring;
            __pcolCursoring = cursToAssign.__pcolCursoring;
            __pnodeCur = cursToAssign.__pnodeCur;

            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsEmpty() const
        {
            return __pllstCursoring->bIsEmpty();
        }

        tCIDLib::TBoolean bIsValid() const
        {
            // If the serial number is off, its not valid
            if (c4SerialNum() != __pcolCursoring->c4SerialNum())
                return kCIDLib::False;

            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bNext()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
                return kCIDLib::False;
            __pnodeCur = (TBagNode<TElem>*)__pnodeCur->pnodeNext();
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bPrevious()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
                return kCIDLib::False;
            __pnodeCur = (TBagNode<TElem>*)__pnodeCur->pnodePrev();
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bReset()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            // Get the head node of the list
            __pnodeCur = (TBagNode<TElem>*)__pllstCursoring->pnodeHead();

            // Get our serial back in sync with the collection
            _c4SerialNum(__pcolCursoring->c4SerialNum());
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            return __pllstCursoring->c4ElemCount();
        }

        const TElem& objCur() const
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsCursoring(const TBag<TElem>& colToCheck) const
        {
            return (__pcolCursoring == &colToCheck);
        }

        const TBagNode<TElem>* pnodeCur() const
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }

        TBagNode<TElem>* pnodeCur()
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TBag<TElem>;


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TBagCursor();


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TBagCursorOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pcolCursoring
        //      The bag collection we are cursoring.
        //
        //  __pllstCursoring
        //      This is the linked list we are cursoring. It is gotten out of
        //      the bag during construction for faster access.
        //
        //  __pnodeCur
        //      This is the current node that we are on in our iteration. It
        //      is 0 if we are at the end.
        // --------------------------------------------------------------------
        const TBag<TElem>*      __pcolCursoring;
        const TDLinkedList*     __pllstCursoring;
        TBagNode<TElem>*        __pnodeCur;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a bag cursor of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TBagCursor<TElem>,TBiElemCursor<TElem>)
};

template <class TElem> tCIDLib::TBoolean TBagCursor<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TBagCursor<TElem>::clsThis;

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Some template functions to make duplication and copying of collections
//  typesafe.
// ----------------------------------------------------------------------------
template <class TElem> tCIDLib::TBoolean
bCompCollections(   const   TBag<TElem>&    col1
                    , const TBag<TElem>&    col2
                    , const TObjEq<TElem>&  compToUse)
{
    // Check for comparison against self
    if (&col1 == &col2)
        return kCIDLib::True;

    // Lock both of the bags
    TLocker lockBag1(&col1);
    TLocker lockBag2(&col2);

    //
    //  First see if the linked lists behind them have equal element counts,
    //  max element counts, and default delete mode. Note that this does
    //  not compare the elements because the linked knows nothing about how
    //  to compare them.
    //
    if (col1.__llstBag != col2.__llstBag)
        return kCIDLib::False;

    // If the element count of either is 0, then they are both zero so equal
    if (!col1.c4ElemCount())
        return kCIDLib::True;

    //
    //  Ok so we have to compare the elements. So we just create cursors
    //  for both and compare each element. We know that they have the
    //  same number of elements and are locked so this safe.
    //
    TBag<TElem>::TCursor curs1(&col1);
    TBag<TElem>::TCursor curs2(&col2);
    do
    {
        if (!compToUse.bEqual(curs1.objCur(), curs2.objCur()))
            return kCIDLib::False;
    }   while (curs1.bNext() && curs2.bNext());
    return kCIDLib::True;
}

template <class TElem> tCIDLib::TVoid
CopyCollectionNodes(TBag<TElem>& colDest, const TBag<TElem>& colSource)
{
    // Check for copy to self
    if (&colDest == &colSource)
        return;

    // Lock both collections
    TLocker lockNew(&colDest);
    TLocker lockToDup(&colSource);

    if (colSource.c4ElemCount())
    {
        TBag<TElem>::TCursor  cursSource(&colSource);
        do
        {
            colDest.Add(cursSource.objCur());
        }   while (cursSource.bNext());
    }
}

template <class TElem> TBag<TElem>* pDupCollection(const TBag<TElem>& colToDup)
{
    // Lock the source collection
    TLocker lockToDup(&colToDup);

    // Create a new bag with the same max elem count and thread safety
    TBag<TElem>* const pcolNew = new TBag<TElem>(colToDup);
    TJanitor<TBag<TElem> > janCol(pcolNew);

    // If there are any elements in the source, copy them
    if (colToDup.c4ElemCount())
    {
        TBag<TElem>::TCursor  cursDup(&colToDup);
        do
        {
            pcolNew->Add(cursDup.objCur());
        }   while (cursDup.bNext());
    }

    // Return a pointer to the new bag
    janCol.Orphan();
    return pcolNew;
}
