//
// NAME: CIDLib_Area.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_Area.Cpp module. This module implements
//  the TArea class, which is a 2D rectangular area with a signed upper
//  left origin and an unsigned size in the positive direction of each axis
//  (left to right, top to bottom.) I.e. its a 4th quadrant coordinate system.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 05/23/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  Areas are always inclusive, so a size of 0,0 means empty. a size of
//      1,1 is 1 pixel. When converting to/from a system rectangle internally,
//      the caller must indicate what the rectange is or should be.
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TArea
//  PREFIX: area
// ----------------------------------------------------------------------------
class CIDLIBEXP TArea :

    public TObject, public MStreamable, public MDuplicable, public MFormattable
{
    public  :
        // --------------------------------------------------------------------
        //  Constuctors and Destructors
        // --------------------------------------------------------------------
        TArea();

        TArea
        (
            const   tCIDLib::THostRectl&    rectlSrc
            , const tCIDLib::ERectlTypes    eInclusive
        );

        TArea
        (
            const   tCIDLib::THostPoint&    ptULeft
            , const tCIDLib::THostPoint&    ptLRight
        );

        TArea
        (
            const   TPoint&                 pntULeft
            , const TPoint&                 pntLRight
        );

        TArea
        (
            const   tCIDLib::TInt4          x
            , const tCIDLib::TInt4          y
            , const tCIDLib::TCard4         cx
            , const tCIDLib::TCard4         cy
        );

        TArea
        (
            const   TPoint&                 pntULeft
            , const tCIDLib::TCard4         cx
            , const tCIDLib::TCard4         cy
        );

        TArea
        (
            const   TPoint&                 pntULeft
            , const TSize&                  szExtent
        );

        TArea
        (
            const   TArea&                  areaToCopy
        );

        ~TArea();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TArea& operator=
        (
            const   TArea&                  areaToAssign
        );

        friend TArea CIDLIBEXP operator|
        (
            const   TArea&                  area1
            , const TArea&                  area2
        );

        tCIDLib::TVoid operator|=
        (
            const   TArea&                  areaToOR
        );

        friend TArea CIDLIBEXP operator&
        (
            const   TArea&                  area1
            , const TArea&                  area2
        );

        tCIDLib::TVoid operator&=
        (
            const   TArea&                  areaToAND
        );

        tCIDLib::TBoolean operator==
        (
            const   TArea&                  areaToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TArea&                  areaToTest
        )   const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid AdjustOrg 
        (
            const   tCIDLib::TInt4          i4XOfs
            , const tCIDLib::TInt4          i4YOfs
        );

        tCIDLib::TVoid AdjustOrg
        (
            const   TPoint&                 pntOfs
        );

        tCIDLib::TVoid AdjustSides
        (
            const   TPoint&                 pntOfs
        );

        tCIDLib::TVoid AdjustSize
        (
            const   tCIDLib::TInt4          i4XOfs
            , const tCIDLib::TInt4          i4YOfs
        );

        tCIDLib::TVoid AdjustSize
        (
            const   TPoint&                 pntOfs
        );

        tCIDLib::TBoolean bContainsPoint
        (
            const   tCIDLib::TInt4          i4X
            , const tCIDLib::TInt4          i4Y
        )   const;

        tCIDLib::TBoolean bContainsPoint
        (
            const   TPoint&                 pntTest
        )   const;

        tCIDLib::TBoolean bEmpty() const;

        tCIDLib::TBoolean bIntersects
        (
            const   TArea&                  areaToTest
        )   const;

        tCIDLib::TBoolean bPercentFromOrg
        (
            const   TPoint&                 pntToTest
            ,       tCIDLib::TFloat4&       f4XPercent
            ,       tCIDLib::TFloat4&       f4YPercent
        )   const;

        tCIDLib::TBoolean bSameSize
        (
            const   TArea&                  areaToTest
        )   const;

        tCIDLib::TCard4 c4Width() const;

        tCIDLib::TCard4 c4Width
        (
            const   tCIDLib::TCard4         c4NewCX
        );

        tCIDLib::TCard4 c4Height() const;

        tCIDLib::TCard4 c4Height
        (
            const   tCIDLib::TCard4         c4NewCY
        );

        tCIDLib::TVoid CenterIn
        (
            const   TArea&                  areaTarget
        );

        tCIDLib::TVoid Deflate
        (
            const   tCIDLib::TCard4         c4XOfs
            , const tCIDLib::TCard4         c4YOfs
        );

        tCIDLib::TVoid Deflate
        (
            const   tCIDLib::TCard4         c4Ofs
        );

        tCIDLib::TVoid FromAreaAtOrg
        (
            const   TArea&                  areaSource
        );

        tCIDLib::TVoid FromAreaScaled
        (
            const   TArea&                  areaSource
            , const tCIDLib::TFloat4        f4CXScale
            , const tCIDLib::TFloat4        f4CYScale
        );

        tCIDLib::TVoid FromPoints
        (
            const   TPoint&                 pntULeft
            , const TPoint&                 pntLRight
        );

        tCIDLib::TVoid FromPoints
        (
            const   tCIDLib::THostPoint&    ptULeft
            , const tCIDLib::THostPoint&    ptLRight
        );

        tCIDLib::TVoid FromRectl
        (
            const   tCIDLib::THostRectl&    rectlSrc
            , const tCIDLib::ERectlTypes    eInclusive
        );

        tCIDLib::TVoid ForceWithin
        (
            const   TArea&                  areaToFit
            , const tCIDLib::TBoolean       bTotal = kCIDLib::False
        );

        tCIDLib::TInt4 i4Bottom() const;

        tCIDLib::TInt4 i4Bottom
        (
            const   tCIDLib::TInt4          i4NewBottom
        );

        tCIDLib::TInt4 i4Left() const;

        tCIDLib::TInt4 i4Left
        (
            const   tCIDLib::TInt4          i4NewLeft
        );

        tCIDLib::TInt4 i4X() const;

        tCIDLib::TInt4 i4X
        (
            const   tCIDLib::TInt4          i4NewX
        );

        tCIDLib::TInt4 i4Y() const;

        tCIDLib::TInt4 i4Y
        (
            const   tCIDLib::TInt4          i4NewY
        );

        tCIDLib:: TInt4 i4Right() const;

        tCIDLib:: TInt4 i4Right
        (
            const   tCIDLib::TInt4          i4XRight
        );

        tCIDLib::TInt4 i4Top() const;

        tCIDLib::TInt4 i4Top
        (
            const   tCIDLib::TInt4          i4YTop
        );

        tCIDLib::TVoid Inflate
        (
            const   tCIDLib::TCard4         c4XOfs
            , const tCIDLib::TCard4         c4YOfs
        );

        tCIDLib::TVoid Inflate
        (
            const   tCIDLib::TCard4         c4Ofs
        );

        tCIDLib::TVoid JustifyIn
        (
            const   TArea&                  areaToJustifyIn
            , const tCIDLib::EHJustify      eHJustify
            , const tCIDLib::EVJustify      eVJustify
        );

        tCIDLib::TVoid NegateOrg();

        TPoint pntCenter() const;

        TPoint pntOrg() const;

        TPoint pntOrg
        (
            const   TPoint&                 pntNew
        );

        TPoint pntOrg
        (
            const   tCIDLib::TInt4          i4X
            , const tCIDLib::TInt4          i4Y
        );

        tCIDLib::TVoid ScaleSize
        (
            const   tCIDLib::TFloat4        f4CXScale
            , const tCIDLib::TFloat4        f4CYScale
        );

        tCIDLib::TVoid Set
        (
            const   tCIDLib::TInt4          i4X
            , const tCIDLib::TInt4          i4Y
            , const tCIDLib::TCard4         c4CX
            , const tCIDLib::TCard4         c4CY
        );

        tCIDLib::TVoid SetSizes
        (
            const   tCIDLib::TCard4         c4CX
            , const tCIDLib::TCard4         c4CY
        );

        TSize szSize() const;

        tCIDLib::TVoid ToPointArray
        (
                    tCIDLib::THostPoint* const aptTarget
        )   const;

        tCIDLib::TVoid ToCornerPoints
        (
                    tCIDLib::THostPoint&    ptULeft
            ,       tCIDLib::THostPoint&    ptURight
            ,       tCIDLib::THostPoint&    ptLRight
            ,       tCIDLib::THostPoint&    ptLLeft
        )   const;

        tCIDLib::TVoid ToCornerPoints
        (
                    TPoint&                 pntULeft
            ,       TPoint&                 pntURight
            ,       TPoint&                 pntLRight
            ,       TPoint&                 pntLLeft
        )   const;

        tCIDLib::TVoid ToPoints
        (
                    TPoint&                 pntULeft
            ,       TPoint&                 pntLRight
        )   const;

        tCIDLib::TVoid ToPoints
        (
                    tCIDLib::THostPoint&    ptULeft
            ,       tCIDLib::THostPoint&    ptLRight
        )   const;

        tCIDLib::TVoid ToRectl
        (
                    tCIDLib::THostRectl&    rectlTarget
            , const tCIDLib::ERectlTypes    eInclusive
        )   const;

        tCIDLib::TVoid UpdateSizesIfZero
        (
            const   tCIDLib::TCard4         c4CX
            , const tCIDLib::TCard4         c4CY
        );

        tCIDLib::TVoid ZeroAll();

        tCIDLib::TVoid ZeroOrg();

        tCIDLib::TVoid ZeroSizes();


    protected           :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class    TPoint;


        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strDest
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private             :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __AdjustSides
        (
            const   tCIDLib::TInt4          i4CXOfs
            , const tCIDLib::TInt4          i4CYOfs
        );


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  The init order is important here! Keep the origin members first
        //  then the length members!!
        //
        //  __i4X, __i4Y
        //      The origin of the area. It is integral because it could start
        //      behind the origin of the coordinate space it represents.
        //
        //  __c4CX, __c4CY
        //      The length and width of the area.
        // --------------------------------------------------------------------
        tCIDLib::TInt4      __i4X;
        tCIDLib::TInt4      __i4Y;
        tCIDLib::TCard4     __c4CX;
        tCIDLib::TCard4     __c4CY;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TArea,TObject)
        DefPolyDup(TArea)
};

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Constructors and destructors
// ----------------------------------------------------------------------------
inline TArea::~TArea()
{
}

// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TArea::operator!=(const TArea& areaToTest) const
{
    return !operator==(areaToTest);
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TVoid TArea::AdjustSides(const TPoint& pntOfs)
{
    __AdjustSides(pntOfs.__i4X, pntOfs.__i4Y);
}

inline tCIDLib::TBoolean TArea::bContainsPoint(const TPoint& pntTest) const
{
    return bContainsPoint(pntTest.__i4X, pntTest.__i4Y);
}

inline tCIDLib::TBoolean TArea::bEmpty() const
{
    return (!__c4CX || !__c4CY) ? kCIDLib::True : kCIDLib::False;
}

inline tCIDLib::TBoolean TArea::bSameSize(const TArea& areaToTest) const
{
    if ((areaToTest.__c4CX == __c4CX) && (areaToTest.__c4CY == __c4CY))
        return kCIDLib::True;
    return kCIDLib::False;
}

inline tCIDLib::TCard4 TArea::c4Width() const
{
    return __c4CX;
}

inline tCIDLib::TCard4 TArea::c4Width(const tCIDLib::TCard4 iNewCX)
{
    __c4CX = iNewCX;
    return __c4CX;
}

inline tCIDLib::TCard4 TArea::c4Height() const
{
    return __c4CY;
}

inline tCIDLib::TCard4 TArea::c4Height(const tCIDLib::TCard4 iNewCY)
{
    __c4CY = iNewCY;
    return __c4CY;
}

inline tCIDLib::TVoid
TArea::Deflate(const tCIDLib::TCard4 c4XOfs, const tCIDLib::TCard4 c4YOfs)
{
    __AdjustSides(-tCIDLib::TInt4(c4XOfs), -tCIDLib::TInt4(c4YOfs));
}

inline tCIDLib::TVoid TArea::Deflate(const tCIDLib::TCard4 c4Ofs)
{
    __AdjustSides(-tCIDLib::TInt4(c4Ofs), -tCIDLib::TInt4(c4Ofs));
}

inline tCIDLib::TInt4 TArea::i4Left() const
{
    return __i4X;
}

inline tCIDLib::TInt4 TArea::i4Left(const tCIDLib::TInt4 i4NewLeft)
{
    __i4X = i4NewLeft;
    return __i4X;
}

inline tCIDLib::TInt4 TArea::i4Top() const
{
    return __i4Y;
}

inline tCIDLib::TInt4 TArea::i4Top(const tCIDLib::TInt4 i4NewTop)
{
    __i4Y = i4NewTop;
    return __i4Y;
}

inline tCIDLib::TInt4 TArea::i4X() const
{
    return __i4X;
}

inline tCIDLib::TInt4 TArea::i4X(const tCIDLib::TInt4 iNewX)
{
    __i4X = iNewX;
    return __i4X;
}

inline tCIDLib::TInt4 TArea::i4Y() const
{
    return __i4Y;
}

inline tCIDLib::TInt4 TArea::i4Y(const tCIDLib::TInt4 iNewY)
{
    __i4Y = iNewY;
    return __i4Y;
}

inline tCIDLib::TVoid
TArea::Inflate( const tCIDLib::TCard4 c4XOfs, const tCIDLib::TCard4 c4YOfs)
{
    __AdjustSides(c4XOfs, c4YOfs);
}

inline tCIDLib::TVoid TArea::Inflate(const tCIDLib::TCard4 c4Ofs)
{
    __AdjustSides(c4Ofs, c4Ofs);
}

inline tCIDLib::TVoid TArea::NegateOrg()
{
    __i4X *= -1;
    __i4Y *= -1;
}

inline TPoint TArea::pntCenter() const
{
    return TPoint(__i4X+(__c4CX >> 1), __i4Y+(__c4CY >> 1));
}

inline TPoint TArea::pntOrg() const
{
    return TPoint(__i4X, __i4Y);
}

inline TPoint TArea::pntOrg(const TPoint& pntNew)
{
    __i4X = pntNew.i4X();
    __i4Y = pntNew.i4Y();
    return pntNew;
}

inline TPoint
TArea::pntOrg(const tCIDLib::TInt4 i4X, const tCIDLib::TInt4 i4Y)
{
    __i4X = i4X;
    __i4Y = i4Y;
    return TPoint(i4X,i4Y);
}

inline tCIDLib::TVoid TArea::ZeroAll()
{
    __i4X = 0;
    __i4Y = 0;
    __c4CX = 0;
    __c4CY = 0;
}

inline tCIDLib::TVoid TArea::Set(   const   tCIDLib::TInt4  i4X
                                    , const tCIDLib::TInt4  i4Y
                                    , const tCIDLib::TCard4 c4CX
                                    , const tCIDLib::TCard4 c4CY)
{
    __i4X = i4X;
    __i4Y = i4Y;
    __c4CX = c4CX;
    __c4CY = c4CY;
}

inline tCIDLib::TVoid
TArea::SetSizes(const tCIDLib::TCard4 c4CX, const tCIDLib::TCard4 c4CY)
{
    __c4CX = c4CX;
    __c4CY = c4CY;
}

inline TSize TArea::szSize() const
{
    return TSize(__c4CX, __c4CY);
}

inline tCIDLib::TVoid TArea::ZeroOrg()
{
    __i4X = 0;
    __i4Y = 0;
}

inline tCIDLib::TVoid TArea::ZeroSizes()
{
    __c4CX = 0;
    __c4CY = 0;
}
