//
// NAME: CIDLib_Area.Cpp
//
// DESCRIPTION:
//
//	This module implements the TArea class, a 2D rectangular area.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData2(TArea,TObject)


// ----------------------------------------------------------------------------
//  Local static methods
// ----------------------------------------------------------------------------

static tCIDLib::TVoid __CheckOrgValue(const tCIDLib::TCard4 c4ToCheck)
{
    if (c4ToCheck > kCIDLib::i4MaxInt)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_UnderOverflow
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TString(L"origin")
        );
    }
}

static tCIDLib::TVoid __CheckSizeValue(const tCIDLib::TInt4 i4ToCheck)
{
    if (i4ToCheck < 0)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_UnderOverflow
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TString(L"size")
        );
    }
}

static tCIDLib::TVoid
__CheckOffset(const tCIDLib::TCard4 c4Value, const tCIDLib::TInt4 i4Ofs)
{
    tCIDLib::TInt8 i8Test = c4Value;
    i8Test += i4Ofs;

    if ((i8Test < 0) || (i8Test > kCIDLib::c4MaxCard))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_UnderOverflow
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TString(L"size")
        );
    }
}

static tCIDLib::TVoid
__CheckOffset(const tCIDLib::TInt4 i4Value, const tCIDLib::TInt4 i4Ofs)
{
    tCIDLib::TInt8 i8Test = i4Value;
    i8Test += i4Ofs;

    if ((i8Test < kCIDLib::i4MinInt) || (i8Test > kCIDLib::i4MaxInt))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_UnderOverflow
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TString(L"origin")
        );
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TArea
//  PREFIX: area
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TArea: Constructors and destructors
// ----------------------------------------------------------------------------
TArea::TArea() :

    __i4X(0)
    , __i4Y(0)
    , __c4CX(0)
    , __c4CY(0)
{
}

TArea::TArea(   const   tCIDLib::THostRectl&    rectlSrc
                , const tCIDLib::ERectlTypes    eInclusive) :

    __i4X(MinVal(rectlSrc.i4Left, rectlSrc.i4Right))
    , __i4Y(MinVal(rectlSrc.i4Bottom, rectlSrc.i4Top))
    , __c4CX
      (
        rectlSrc.i4Left > rectlSrc.i4Right ?
        rectlSrc.i4Left - rectlSrc.i4Right :
        rectlSrc.i4Right - rectlSrc.i4Left
      )
    , __c4CY
      (
            rectlSrc.i4Top > rectlSrc.i4Bottom ?
            rectlSrc.i4Top - rectlSrc.i4Bottom :
            rectlSrc.i4Bottom - rectlSrc.i4Top
      )
{
    if (eInclusive == tCIDLib::ERectl_Inclusive)
    {
        // Be careful of an empty rectangle
        if (__c4CX)
            __c4CX++;

        if (__c4CY)
            __c4CY++;
    }
}

TArea::TArea(   const   tCIDLib::THostPoint&    ptULeft
                , const tCIDLib::THostPoint&    ptLRt) :

    // Note that this depends upon correct member order!!
    __i4X(MinVal(ptULeft.i4X, ptLRt.i4X))
    , __i4Y(MinVal(ptULeft.i4Y, ptLRt.i4Y))
    , __c4CX(MaxVal(ptULeft.i4X, ptLRt.i4X) - __i4X)
    , __c4CY(MaxVal(ptULeft.i4Y, ptLRt.i4Y) - __i4Y)
{
}

TArea::TArea(const TPoint& pntULeft, const TPoint& pntLRt)
{
    // Get the values out of the points for greater efficiency
    tCIDLib::TInt4 i4X1 = pntULeft.__i4X;
    tCIDLib::TInt4 i4Y1 = pntULeft.__i4Y;
    tCIDLib::TInt4 i4X2 = pntLRt.__i4X;
    tCIDLib::TInt4 i4Y2 = pntLRt.__i4Y;

    __i4X   = MinVal(i4X1, i4X2);
    __c4CX  = i4X1 > i4X2 ? i4X1-i4X2 : i4X2-i4X1;
    __i4Y   = MinVal(i4Y1, i4Y2);
    __c4CY  = i4Y1 > i4Y2 ? i4Y1-i4Y2 : i4Y2-i4Y1;
}

TArea::TArea(   const   tCIDLib::TInt4  x
                , const tCIDLib::TInt4  y
                , const tCIDLib::TCard4 cx
                , const tCIDLib::TCard4 cy) :
    __i4X(x)
    , __i4Y(y)
    , __c4CX(cx)
    , __c4CY(cy)
{
}

TArea::TArea(const TPoint& pntOrg, const TSize& szExtent) :

    __i4X(pntOrg.__i4X)
    , __i4Y(pntOrg.__i4Y)
    , __c4CX(szExtent.c4Width())
    , __c4CY(szExtent.c4Height())
{
}

TArea::TArea(   const   TPoint&         pntOrg
                , const tCIDLib::TCard4 cx
                , const tCIDLib::TCard4 cy) :

    __i4X(pntOrg.__i4X)
    , __i4Y(pntOrg.__i4Y)
    , __c4CX(cx)
    , __c4CY(cy)
{
}

TArea::TArea(const TArea& areaSrc) :

    __i4X(areaSrc.__i4X)
    , __i4Y(areaSrc.__i4Y)
    , __c4CX(areaSrc.__c4CX)
    , __c4CY(areaSrc.__c4CY)
{
}


// ----------------------------------------------------------------------------
//  TArea: Public operators
// ----------------------------------------------------------------------------

TArea& TArea::operator=(const TArea& areaToAssign)
{
    if (this == &areaToAssign)
        return *this;

    __i4X  = areaToAssign.__i4X;
    __i4Y  = areaToAssign.__i4Y;
    __c4CX = areaToAssign.__c4CX;
    __c4CY = areaToAssign.__c4CY;
    return *this;
}


TArea operator|(const TArea& area1, const TArea& area2)
{
    TArea areaNew(area1);
    areaNew |= area2;
    return areaNew;
}

tCIDLib::TVoid TArea::operator|=(const TArea& areaToOR)
{
    // If the same object, then just return
    if (&areaToOR == this)
        return;

    tCIDLib::TInt4   i4L, i4R, i4B, i4T;

    //
    //  Take the smaller of the origin points and the larger of the upper
    //  and right points.
    //
    i4L = MinVal(__i4X, areaToOR.__i4X);
    i4T = MinVal(__i4Y, areaToOR.__i4Y);
    i4R = MaxVal(i4Right(), areaToOR.i4Right());
    i4B = MaxVal(i4Bottom(), areaToOR.i4Bottom());

    __i4X   = i4L;
    __i4Y   = i4T;
    __c4CX  = (i4R-i4L)+1;
    __c4CY  = (i4B-i4T)+1;
}


TArea operator&(const TArea& area1, const TArea& area2)
{
    TArea areaRet(area1);
    areaRet &= area2;
    return areaRet;
}

tCIDLib::TVoid TArea::operator&=(const TArea& areaToAND)
{
    // If the same object, then nothing to do
    if (&areaToAND == this)
        return;

    tCIDLib::TInt4   i4L, i4T, i4R, i4B;

    //
    //  Take the larger of the origin points and the smaller of the upper
    //  and right points.
    //
    i4L = MaxVal(__i4X, areaToAND.__i4X);
    i4T = MaxVal(__i4Y, areaToAND.__i4Y);
    i4R = MinVal(i4Right(), areaToAND.i4Right());
    i4B = MinVal(i4Bottom(), areaToAND.i4Bottom());

    //
    //  See if there is any common area. If the right is less than the
    //  left of the bottom less than the top, then none. So we would return
    //  and empty area.
    //
    if ((i4R <= i4L) || (i4B <= i4Top))
    {
        __i4X = 0;
        __i4Y = 0;
        __c4CX = 0;
        __c4CY = 0;
        return;
    }

    // Store the new stuff
    __i4X   = i4L;
    __i4Y   = i4T;
    __c4CX  = (i4R-i4L)+1;
    __c4CY  = (i4B-i4T)+1;
}


tCIDLib::TBoolean TArea::operator==(const TArea& areaToTest) const
{
    if (__i4X != areaToTest.__i4X) return kCIDLib::False;
    if (__i4Y != areaToTest.__i4Y) return kCIDLib::False;
    if (__c4CX != areaToTest.__c4CX) return kCIDLib::False;
    if (__c4CY != areaToTest.__c4CY) return kCIDLib::False;

    return kCIDLib::True;
}


// ----------------------------------------------------------------------------
//  TArea: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TArea::AdjustOrg(const tCIDLib::TInt4 i4XOfs, const tCIDLib::TInt4 i4YOfs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__i4X, i4XOfs);
    __CheckOffset(__i4Y, i4YOfs);
    #endif

    __i4X += i4XOfs;
    __i4Y += i4YOfs;
}

tCIDLib::TVoid TArea::AdjustOrg(const TPoint& pntOfs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__i4X, pntOfs.__i4X);
    __CheckOffset(__i4Y, pntOfs.__i4Y);
    #endif

    __i4X += pntOfs.__i4X;
    __i4Y += pntOfs.__i4Y;
}


tCIDLib::TVoid
TArea::AdjustSize(const tCIDLib::TInt4 i4CXOfs, const tCIDLib::TInt4 i4CYOfs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__c4CX, i4CXOfs);
    __CheckOffset(__c4CY, i4CYOfs);
    #endif

    __c4CX += i4CXOfs;
    __c4CY += i4CYOfs;

}

tCIDLib::TVoid TArea::AdjustSize(const TPoint& pntOfs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__c4CX, pntOfs.__i4X);
    __CheckOffset(__c4CY, pntOfs.__i4Y);
    #endif

    __c4CX += pntOfs.__i4X;
    __c4CY += pntOfs.__i4Y;
}


tCIDLib::TBoolean
TArea::bContainsPoint(  const   tCIDLib::TInt4  i4X
                        , const tCIDLib::TInt4  i4Y) const
{
    if (i4X < __i4X)
        return kCIDLib::False;
    if (i4Y < __i4Y)
        return kCIDLib::False;
    if (i4X >= tCIDLib::TInt4(__i4X + __c4CX))
        return kCIDLib::False;
    if (i4Y >= tCIDLib::TInt4(__i4Y + __c4CY))
        return kCIDLib::False;
    return kCIDLib::True;
}


tCIDLib::TBoolean TArea::bIntersects(const TArea& areaTest) const
{
    // If either is empty, then cannot intersect
    if (bEmpty() || areaTest.bEmpty())
        return kCIDLib::False;

    // Get the upper extremes of each area
    tCIDLib::TInt4  i4ThisRight    = i4Right();
    tCIDLib::TInt4  i4ThisBottom   = i4Bottom();
    tCIDLib::TInt4  i4TestRight    = areaTest.i4Right();
    tCIDLib::TInt4  i4TestBottom   = areaTest.i4Bottom();

    // Check the sides
    if (areaTest.__i4X > i4ThisRight)
        return kCIDLib::False;
    if (__i4X > i4TestRight)
        return kCIDLib::False;

    if (areaTest.__i4Y > i4ThisBottom)
        return kCIDLib::False;
    if (__i4Y > i4TestBottom)
        return kCIDLib::False;

    // They intersect
    return kCIDLib::True;
}

tCIDLib::TBoolean
TArea::bPercentFromOrg( const   TPoint&             pntToTest
                        ,       tCIDLib::TFloat4&   f4XPercent
                        ,       tCIDLib::TFloat4&   f4YPercent) const
{
    if (!bContainsPoint(pntToTest))
        return kCIDLib::False;

    tCIDLib::TFloat4 f4Relative = tCIDLib::TFloat4(pntToTest.__i4X - __i4X);
    tCIDLib::TFloat4 f4Range = tCIDLib::TFloat4(__c4CX);
    f4XPercent = f4Relative / f4Range;

    f4Relative = tCIDLib::TFloat4(pntToTest.__i4Y - __i4Y);
    f4Range = tCIDLib::TFloat4(__c4CY);
    f4YPercent = f4Relative / f4Range;

    return kCIDLib::True;
}


tCIDLib::TVoid TArea::CenterIn(const TArea& areaTarget)
{
    JustifyIn(areaTarget, tCIDLib::EHJustify_Center, tCIDLib::EVJustify_Center);
}


tCIDLib::TVoid TArea::FromAreaAtOrg(const TArea& areaSource)
{
    __i4X = 0;
    __i4Y = 0;
    __c4CX = areaSource.__c4CX;
    __c4CY = areaSource.__c4CY;
}


tCIDLib::TVoid
TArea::FromAreaScaled(  const   TArea&              areaSource
                        , const tCIDLib::TFloat4    f4CXScale
                        , const tCIDLib::TFloat4    f4CYScale)
{
    // Insure the scaling values are legal
    if ((f4CXScale < 0.0) || (f4CXScale > 1.0)
    ||  (f4CYScale < 0.0) || (f4CYScale > 1.0))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_BadScaling
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TFPoint(f4CXScale, f4CYScale)
        );
    }

    __i4X = areaSource.__i4X;
    __i4Y = areaSource.__i4Y;
    __c4CX = tCIDLib::TCard4(areaSource.__c4CX * f4CXScale);
    __c4CY = tCIDLib::TCard4(areaSource.__c4CY * f4CYScale);
}


tCIDLib::TVoid
TArea::FromPoints(const TPoint& pntULeft, const TPoint& pntLRight)
{
    if (pntULeft.__i4X > pntLRight.__i4X)
    {
        __i4X  = pntLRight.__i4X;
        __c4CX = (pntULeft.__i4X-pntLRight.__i4X) + 1;
    }
     else
    {
        __i4X  = pntULeft.__i4X;
        __c4CX = (pntLRight.__i4X-pntULeft.__i4X) + 1;
    }

    if (pntULeft.__i4Y > pntLRight.__i4Y)
    {
        __i4Y  = pntLRight.__i4Y;
        __c4CY = (pntULeft.__i4Y-pntLRight.__i4Y) + 1;
    }
     else
    {
        __i4Y  = pntULeft.__i4Y;
        __c4CY = (pntLRight.__i4Y-pntULeft.__i4Y) + 1;
    }
}

tCIDLib::TVoid
TArea::FromPoints(  const   tCIDLib::THostPoint&    ptlULeft
                    , const tCIDLib::THostPoint&    ptlLRight)
{
    if (ptlULeft.i4X > ptlLRight.i4X)
    {
        __i4X  = ptlLRight.i4X;
        __c4CX = tCIDLib::TCard4(ptlULeft.i4X-ptlLRight.i4X)+1;
    }
     else
    {
        __i4X  = ptlULeft.i4X;
        __c4CX = tCIDLib::TCard4(ptlLRight.i4X-ptlULeft.i4X)+1;
    }

    if (ptlULeft.i4Y > ptlLRight.i4Y)
    {
        __i4Y  = ptlLRight.i4Y;
        __c4CY = tCIDLib::TCard4(ptlULeft.i4Y-ptlLRight.i4Y)+1;
    }
     else
    {
        __i4Y  = ptlULeft.i4Y;
        __c4CY = tCIDLib::TCard4(ptlLRight.i4Y-ptlULeft.i4Y)+1;
    }
}


tCIDLib::TVoid TArea::FromRectl(const   tCIDLib::THostRectl&    rectlSrc
                                , const tCIDLib::ERectlTypes    eInclusive)
{
    if (rectlSrc.i4Left > rectlSrc.i4Right)
    {
        __i4X = rectlSrc.i4Right;
        __c4CX = (rectlSrc.i4Left-rectlSrc.i4Right);
    }
     else
    {
        __i4X  = rectlSrc.i4Left;
        __c4CX = (rectlSrc.i4Right-rectlSrc.i4Left);
    }

    if (rectlSrc.i4Bottom > rectlSrc.i4Top)
    {
        __i4Y  = rectlSrc.i4Top;
        __c4CY = (rectlSrc.i4Bottom-rectlSrc.i4Top);
    }
     else
    {
        __i4Y  = rectlSrc.i4Bottom;
        __c4CY = (rectlSrc.i4Top-rectlSrc.i4Bottom);
    }

    // If inclusive, bump up the sizes by 1 if not empty
    if (eInclusive == tCIDLib::ERectl_Inclusive)
    {
        if (__c4CX)
            __c4CX++;

        if (__c4CY)
            __c4CY++;
    }
}


tCIDLib::TVoid
TArea::ForceWithin(const TArea& areaToFit, const tCIDLib::TBoolean bTotal)
{
    if (__i4X < areaToFit.__i4X)
        __i4X = areaToFit.__i4X;

    if (__i4Y < areaToFit.__i4Y)
        __i4Y = areaToFit.__i4Y;

    if (i4Right() > areaToFit.i4Right())
    {
        if (__i4X > areaToFit.__i4X)
            __i4X = areaToFit.__i4X;

        if (bTotal)
        {
            if (i4Right() > areaToFit.i4Right())
                __c4CX = areaToFit.__c4CX;
        }
    }

    if (i4Bottom() > areaToFit.i4Bottom())
    {
        if (__i4Y >= areaToFit.__i4Y)
            __i4Y = areaToFit.__i4Y;

        if (bTotal)
        {
            if (i4Bottom() > areaToFit.i4Bottom())
                __c4CY = areaToFit.__c4CY;
        }
    }
}


tCIDLib:: TInt4 TArea::i4Right() const
{
    if (__c4CX)
        return (__i4X+__c4CX)-1;
     else
        return (__i4X);
}

tCIDLib:: TInt4 TArea::i4Right(const tCIDLib::TInt4 i4NewXRight)
{
    // Make sure that it is not less than the origin
    if (i4NewXRight < __i4X)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_BehindOrg
            , L"(Horizontal)"
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(i4NewXRight)
            , TInteger(__i4X)
        );
    }
    __c4CX = (i4NewXRight-__i4X)+1;
    return i4NewXRight;
}

tCIDLib::TInt4 TArea::i4Bottom() const
{
    if (__c4CY)
        return (__i4Y+__c4CY)-1;
     else
        return (__i4Y);
}

tCIDLib::TInt4 TArea::i4Bottom(const tCIDLib::TInt4 i4NewBottom)
{
    // Make sure that it is not less than the origin
    if (i4NewBottom < __i4Y)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_BehindOrg
            , L"(Vertical)"
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(i4NewBottom)
            , TInteger(__i4Y)
        );
    }
    __c4CY = (i4NewBottom-__i4Y)+1;
    return i4NewBottom;
}


tCIDLib::TVoid
TArea::JustifyIn(   const   TArea&              areaTarget
                    , const tCIDLib::EHJustify  eHJustify
                    , const tCIDLib::EVJustify  eVJustify)
{
    switch(eHJustify)
    {
        case tCIDLib::EHJustify_Left    :
            //  Set the x origin of this area to that of the target
            __i4X = areaTarget.__i4X;
            break;

        case tCIDLib::EHJustify_Center  :
            // Set the y to target.center/2 - cx/2
            __i4X = areaTarget.__i4X+(areaTarget.__c4CX >> 1);
            __i4X -= (__c4CX >> 1);
            break;

        case tCIDLib::EHJustify_Right   :
            //  Set the x origin of this area to target.i4Right-cx
            __i4X = areaTarget.i4Right()-__c4CX;
            break;

        default :
            facCIDLib.LogMsg
            (
                __FILE__
                , __LINE__
                , "Unknown horizontal justification"
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_BadParms
            );
            break;
    }

    switch(eVJustify)
    {
        case tCIDLib::EVJustify_Top  :
            //  Set the y origin of this area to that of the target
            __i4Y = areaTarget.__i4Y;
            break;

        case tCIDLib::EVJustify_Center  :
            // Set the y to target.center/2 - cy/2
            __i4Y = areaTarget.__i4Y+(areaTarget.__c4CY >> 1);
            __i4Y -= (__c4CY >> 1);
            break;

        case tCIDLib::EVJustify_Bottom :
            //  Set the y origin of this area to target.i4Bottom-cy
            __i4Y = areaTarget.i4Bottom()-__c4CY;
            break;

        default :
            facCIDLib.LogMsg
            (
                __FILE__
                , __LINE__
                , "Unknown vertical justification"
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_BadParms
            );
            break;
    }
}


tCIDLib::TVoid
TArea::ScaleSize(   const   tCIDLib::TFloat4 f4CXScale
                    , const tCIDLib::TFloat4 f4CYScale)
{
    // Insure the scaling values are legal
    if ((f4CXScale < 0.0) || (f4CXScale > 1.0)
    ||  (f4CYScale < 0.0) || (f4CYScale > 1.0))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_BadScaling
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TFPoint(f4CXScale, f4CYScale)
        );
    }

    tCIDLib::TFloat8    f4Tmp = __c4CX;
    f4Tmp *= f4CXScale;
    if ((f4Tmp < 0.0) || (f4Tmp > kCIDLib::c4MaxCard))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_Range
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TFloat(f4Tmp, 0)
            , TString("size")
        );
    }
    __c4CX = tCIDLib::TCard4(f4Tmp);

    f4Tmp = __c4CY;
    f4Tmp *= f4CYScale;
    if ((f4Tmp < 0.0) || (f4Tmp > kCIDLib::c4MaxCard))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_Range
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TFloat(f4Tmp, 0)
            , TString("size")
        );
    }
    __c4CY = tCIDLib::TCard4(f4Tmp);
}


tCIDLib::TVoid TArea::ToPointArray(tCIDLib::THostPoint* const aptlTarget) const
{
    tCIDLib::TInt4 i4R = i4Right();
    tCIDLib::TInt4 i4B = i4Bottom();

    aptlTarget[0].i4X = __i4X;
    aptlTarget[0].i4Y = i4B;
    aptlTarget[1].i4X = __i4X;
    aptlTarget[1].i4Y = __i4Y;
    aptlTarget[2].i4X = i4R;
    aptlTarget[2].i4Y = __i4Y;
    aptlTarget[3].i4X = i4R;
    aptlTarget[3].i4Y = i4B;
    aptlTarget[4].i4X = __i4X;
    aptlTarget[4].i4Y = i4B;
}


tCIDLib::TVoid TArea::ToCornerPoints(   tCIDLib::THostPoint&    ptlULeft
                                        , tCIDLib::THostPoint&  ptlURight
                                        , tCIDLib::THostPoint&  ptlLRight
                                        , tCIDLib::THostPoint&  ptlLLeft) const
{
    tCIDLib::TInt4 i4R = i4Right();
    tCIDLib::TInt4 i4B = i4Bottom();

    ptlULeft.i4X  = __i4X;
    ptlULeft.i4Y  = __i4Y;

    ptlURight.i4X = i4R;
    ptlURight.i4Y = __i4Y;

    ptlLRight.i4X = i4R;
    ptlLRight.i4Y = i4B;

    ptlLLeft.i4X = __i4X;
    ptlLLeft.i4Y = i4B;
}


tCIDLib::TVoid TArea::ToCornerPoints(   TPoint&     pntULeft
                                        , TPoint&   pntURight
                                        , TPoint&   pntLRight
                                        , TPoint&   pntLLeft) const
{
    tCIDLib::TInt4 i4R = i4Right();
    tCIDLib::TInt4 i4B = i4Bottom();

    pntULeft.__i4X  = __i4X;
    pntULeft.__i4Y  = __i4Y;

    pntURight.__i4X = i4R;
    pntURight.__i4Y = __i4Y;

    pntLRight.__i4X = i4R;
    pntLRight.__i4Y = i4B;

    pntLLeft.__i4X = __i4X;
    pntLLeft.__i4Y = i4B;
}


tCIDLib::TVoid TArea::ToPoints(TPoint& pntULeft, TPoint& pntLRight) const
{
    pntULeft.__i4X = __i4X;
    pntULeft.__i4Y = __i4Y;

    pntLRight.__i4X = i4Right();
    pntLRight.__i4Y = i4Bottom();
}

tCIDLib::TVoid
TArea::ToPoints(tCIDLib::THostPoint&    ptlULeft
                , tCIDLib::THostPoint&  ptlLRight) const
{
    ptlULeft.i4X = __i4X;
    ptlULeft.i4Y = __i4Y;

    ptlLRight.i4X = i4Right();
    ptlLRight.i4Y = i4Bottom();
}


tCIDLib::TVoid TArea::ToRectl(          tCIDLib::THostRectl&    rectlDest
                                , const tCIDLib::ERectlTypes    eInclusive) const
{
    // Create a inclusive rectangle
    rectlDest.i4Left  = __i4X;
    rectlDest.i4Top   = __i4Y;

    rectlDest.i4Right = __i4X+tCIDLib::TInt4(__c4CX);
    rectlDest.i4Bottom= __i4Y+tCIDLib::TInt4(__c4CY);

    //
    //  If the rectangle is inclusive, then bump down the upper right corner
    //  so that it lies on the upper right sides.
    //
    if (eInclusive == tCIDLib::ERectl_Inclusive)
    {
        if (__c4CX)
            rectlDest.i4Right--;
        if (__c4CY)
            rectlDest.i4Bottom--;
    }
}


tCIDLib::TVoid
TArea::UpdateSizesIfZero(const tCIDLib::TCard4 c4CX, const tCIDLib::TCard4 c4CY)
{
    if (!__c4CX)
        __c4CX = c4CX;
    if (!__c4CY)
        __c4CY = c4CY;
}


// ----------------------------------------------------------------------------
//  TArea: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TArea::_FormatTo(TTextStream& strmDest) const
{
    strmDest << __i4X << L"," << __i4Y << L"-" << __c4CX << L"," << __c4CY;
}


tCIDLib::TVoid TArea::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom  >> __i4X
                    >> __i4Y
                    >> __c4CX
                    >> __c4CY;
}

tCIDLib::TVoid TArea::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo   << __i4X
                    << __i4Y
                    << __c4CX
                    << __c4CY;
}


// ----------------------------------------------------------------------------
//  TArea: Private, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __AdjustSides
//
// DESCRIPTION:
//
//  This method will adjust the sides of the area by the given x and y
//  offsets. Negative values will cause the area to become smaller and
//  positive values will cause it to become larger. Both sides are pushed
//  outward or inward by the offset amount.
// ---------------------------------------
//   INPUT: i4XOfs, i4YOfs are the offsets to adjust the sides by
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TArea::__AdjustSides(   const   tCIDLib::TInt4  i4XOfs
                        , const tCIDLib::TInt4  i4YOfs)
{
    tCIDLib::TBoolean  bNegX, bNegY;

    // Get copies of the values that we can abuse
    tCIDLib::TInt4  i4XAdj = i4XOfs;
    tCIDLib::TInt4  i4YAdj = i4YOfs;

    //
    //  If the x offset is negative, then see if the area is large enough to
    //  adjust by this offset. If not, then set the offset to the best we can
    //  do, which is to make it a 0 sized area with a new origin that is at
    //  the middle of the old size. Do the same thing for the y offset also.
    //
    if (i4XAdj < 0)
    {
        bNegX = kCIDLib::True;
        i4XAdj *= -1;
    }
     else
    {
        bNegX = kCIDLib::False;
    }
    if (tCIDLib::TCard4(i4XAdj << 1) >= __c4CX)
        i4XAdj = tCIDLib::TInt4(__c4CX >> 1);

    if (i4YAdj < 0)
    {
        bNegY = kCIDLib::True;
        i4YAdj *= -1;
    }
     else
    {
        bNegY = kCIDLib::False;
    }
    if (tCIDLib::TCard4(i4YAdj << 1) >= __c4CY)
        i4YAdj = tCIDLib::TInt4(__c4CY >> 1);

    //
    //  Now adjust the sides. Adjust the orgin first, then adjust the length
    //  and width. When doing the length and width, be careful to check for
    //  a potential underflow of the __c4CX and __c4CY values. Despite the
    //  checks done above, the offset may have been an odd number, in which
    //  case there could still be an overflow of 1. The division done above
    //  was via a shift operation, which would just truncate the extra odd
    //  bit.
    //
    if (bNegX)
    {
        __i4X += i4XAdj;
        if (__c4CX <= tCIDLib::TCard4(i4XAdj * 2))
            __c4CX = 0;
         else
            __c4CX -= (i4XAdj * 2);
    }
     else
    {
        __i4X  -= i4XAdj;
        __c4CX += (i4XAdj * 2);
    }

    if (bNegY)
    {
        __i4Y += i4YAdj;
        if (__c4CY <= tCIDLib::TCard4(i4YAdj * 2))
            __c4CY = 0;
         else
            __c4CY -= (i4YAdj * 2);
    }
     else
    {
        __i4Y  -= i4YAdj;
        __c4CY += (i4YAdj * 2);
    }
}
