//
// NAME: CIDLib_4By4Matrix.Cpp
//
// DESCRIPTION:
//
//  This module implements the T4By4Matrix class.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 09/10/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"



// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData(T4By4Matrix,TObject)



// -----------------------------------------------------------------------------
//   CLASS: T4By4Matrix
//  PREFIX: mtrx
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  T4By4Matrix: Constructor and Destructors
// -----------------------------------------------------------------------------

T4By4Matrix::T4By4Matrix(   const   tCIDLib::TFloat8&   f81_1
                            , const tCIDLib::TFloat8&   f81_2
                            , const tCIDLib::TFloat8&   f81_3
                            , const tCIDLib::TFloat8&   f82_1
                            , const tCIDLib::TFloat8&   f82_2
                            , const tCIDLib::TFloat8&   f82_3
                            , const tCIDLib::TFloat8&   f83_1
                            , const tCIDLib::TFloat8&   f83_2
                            , const tCIDLib::TFloat8&   f83_3
                            , const tCIDLib::TFloat8&   f8Xt
                            , const tCIDLib::TFloat8&   f8Yt
                            , const tCIDLib::TFloat8&   f8Zt)
{
    TRawMem::SetMemBuf(__af8Matrix, tCIDLib::TCard1(0), sizeof(__af8Matrix));
    __af8Matrix[0][0]     = f81_1;
    __af8Matrix[0][1]     = f81_2;
    __af8Matrix[0][2]     = f81_3;
    __af8Matrix[1][0]     = f82_1;
    __af8Matrix[1][1]     = f82_2;
    __af8Matrix[1][2]     = f82_3;
    __af8Matrix[2][0]     = f83_1;
    __af8Matrix[2][1]     = f83_2;
    __af8Matrix[2][2]     = f83_3;
    __af8Matrix[3][0]     = f8Xt;
    __af8Matrix[3][1]     = f8Yt;
    __af8Matrix[3][2]     = f8Zt;
}

T4By4Matrix::T4By4Matrix(const T4By4Matrix& mtrxSrc)
{
    TRawMem::CopyMemBuf(__af8Matrix, mtrxSrc.__af8Matrix, sizeof(__af8Matrix));
}


// -----------------------------------------------------------------------------
//  T4By4Matrix: Public operators
// -----------------------------------------------------------------------------

const tCIDLib::TFloat8
T4By4Matrix::operator[](const tCIDLib::EMatrixElems eElem) const
{
    return __af8Array[eElem];
}

tCIDLib::TFloat8&
T4By4Matrix::operator[](const tCIDLib::EMatrixElems eElem)
{
    return __af8Array[eElem];
}


T4By4Matrix operator*(const T4By4Matrix& mtrx1, const T4By4Matrix& mtrx2)
{
    T4By4Matrix mtrxRet;

    mtrxRet.Zero();
    for (tCIDLib::TCard4 c4Outer = 0; c4Outer < 4; c4Outer++)
    {
        for (tCIDLib::TCard4 c4Inner = 0; c4Inner < 4; c4Inner++)
        {
            for (tCIDLib::TCard4 c4Element = 0; c4Element < 4; c4Element++)
                mtrxRet.__af8Matrix[c4Outer][c4Inner]
                        += (mtrx1.__af8Matrix[c4Outer][c4Element]
                            * mtrx2.__af8Matrix[c4Element][c4Inner]);
        }
    }
    return mtrxRet;
}

tCIDLib::TVoid T4By4Matrix::operator*=(const T4By4Matrix& mtrxSrc)
{
    // Make a copy of this matrix
    T4By4Matrix mtrxTmp(*this);

    // Zero out this matrix as the new empty matrix to fill in
    Zero();
    for (tCIDLib::TCard4 c4Outer = 0; c4Outer < 4; c4Outer++)
    {
        for (tCIDLib::TCard4 c4Inner = 0; c4Inner < 4; c4Inner++)
        {
            for (tCIDLib::TCard4 c4Element = 0; c4Element < 4; c4Element++)
            {
                __af8Matrix[c4Outer][c4Inner]
                            += (mtrxTmp.__af8Matrix[c4Outer][c4Element]
                                * mtrxSrc.__af8Matrix[c4Element][c4Inner]);
            }
        }
    }
}


tCIDLib::TBoolean
T4By4Matrix::operator==(const T4By4Matrix& mtrxToTest) const
{
    // Compare the values of the objects
    if (TRawMem::eCompareMemBuf
    (
        mtrxToTest.__af8Matrix
        , __af8Matrix
        , sizeof(__af8Matrix)))
    {
        return kCIDLib::False;
    }
    return kCIDLib::True;
}


T4By4Matrix& T4By4Matrix::operator=(const T4By4Matrix& mtrxToAssign)
{
    if (this == &mtrxToAssign)
        return *this;

    TRawMem::CopyMemBuf
    (
        __af8Matrix
        , mtrxToAssign.__af8Matrix
        , sizeof(__af8Matrix)
    );
    return *this;
}


// -----------------------------------------------------------------------------
//  T4By4Matrix: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
T4By4Matrix::AppendRotation(const T3DVector& vecAngles)
{
    // Get a copy of the vector and convert to radians
    T3DVector vecTmp(vecAngles);
    vecTmp.ToRadians();

    // Get the sine and cosine of each value
    tCIDLib::TFloat8 f8SinX, f8SinY, f8SinZ;
    tCIDLib::TFloat8 f8CosX, f8CosY, f8CosZ;
    f8SinX = TMathLib::f8Sine(vecTmp.f8X());
    f8CosX = TMathLib::f8Cosine(vecTmp.f8X());
    f8SinY = TMathLib::f8Sine(vecTmp.f8Y());
    f8CosY = TMathLib::f8Cosine(vecTmp.f8Y());
    f8SinZ = TMathLib::f8Sine(vecTmp.f8Z());
    f8CosZ = TMathLib::f8Cosine(vecTmp.f8Z());

    // Concatenate the X rotation
    T4By4Matrix mtrxTmp;
    mtrxTmp[tCIDLib::EMatrix_22] = f8CosX;
    mtrxTmp[tCIDLib::EMatrix_33] = f8CosX;
    mtrxTmp[tCIDLib::EMatrix_23] = f8SinX;
    mtrxTmp[tCIDLib::EMatrix_32] = -f8SinX;
    *this *= mtrxTmp;

    // Concatenate the Y rotation
    mtrxTmp.SetToIdentity();
    mtrxTmp[tCIDLib::EMatrix_11] = f8CosY;
    mtrxTmp[tCIDLib::EMatrix_33] = f8CosY;
    mtrxTmp[tCIDLib::EMatrix_13] = -f8SinY;
    mtrxTmp[tCIDLib::EMatrix_31] = f8SinY;
    *this *= mtrxTmp;

    // Concatenate the Z rotation
    mtrxTmp.SetToIdentity();
    mtrxTmp[tCIDLib::EMatrix_11] = f8CosZ;
    mtrxTmp[tCIDLib::EMatrix_22] = f8CosZ;
    mtrxTmp[tCIDLib::EMatrix_12] = f8SinZ;
    mtrxTmp[tCIDLib::EMatrix_21] = -f8SinZ;
    *this *= mtrxTmp;
}

tCIDLib::TVoid
T4By4Matrix::AppendRotation(const   tCIDLib::TFloat8&   f8XAngle
                            , const tCIDLib::TFloat8&   f8YAngle
                            , const tCIDLib::TFloat8&   f8ZAngle)
{
    // Convert the angles to radians
    tCIDLib::TFloat8 f8XA = f8XAngle * kCIDLib::f8PI / 180.0;
    tCIDLib::TFloat8 f8YA = f8YAngle * kCIDLib::f8PI / 180.0;
    tCIDLib::TFloat8 f8ZA = f8ZAngle * kCIDLib::f8PI / 180.0;

    // Get the sine and cosine of each value
    tCIDLib::TFloat8 f8SinX, f8SinY, f8SinZ;
    tCIDLib::TFloat8 f8CosX, f8CosY, f8CosZ;
    f8SinX = TMathLib::f8Sine(f8XA);
    f8CosX = TMathLib::f8Cosine(f8XA);
    f8SinY = TMathLib::f8Sine(f8YA);
    f8CosY = TMathLib::f8Cosine(f8YA);
    f8SinZ = TMathLib::f8Sine(f8ZA);
    f8CosZ = TMathLib::f8Cosine(f8ZA);

    // Concatenate the X rotation
    T4By4Matrix mtrxTmp;
    mtrxTmp[tCIDLib::EMatrix_22] = f8CosX;
    mtrxTmp[tCIDLib::EMatrix_33] = f8CosX;
    mtrxTmp[tCIDLib::EMatrix_23] = f8SinX;
    mtrxTmp[tCIDLib::EMatrix_32] = -f8SinX;
    *this *= mtrxTmp;

    // Concatenate the Y rotation
    mtrxTmp.SetToIdentity();
    mtrxTmp[tCIDLib::EMatrix_11] = f8CosY;
    mtrxTmp[tCIDLib::EMatrix_33] = f8CosY;
    mtrxTmp[tCIDLib::EMatrix_13] = -f8SinY;
    mtrxTmp[tCIDLib::EMatrix_31] = f8SinY;
    *this *= mtrxTmp;

    // Concatenate the Z rotation
    mtrxTmp.SetToIdentity();
    mtrxTmp[tCIDLib::EMatrix_11] = f8CosZ;
    mtrxTmp[tCIDLib::EMatrix_22] = f8CosZ;
    mtrxTmp[tCIDLib::EMatrix_12] = f8SinZ;
    mtrxTmp[tCIDLib::EMatrix_21] = -f8SinZ;
    *this *= mtrxTmp;
}


tCIDLib::TVoid T4By4Matrix::AppendScale(const T3DVector& vecScale)
{
    // Create a temporary matrix and set up the scaling
    T4By4Matrix mtrxTmp;
    mtrxTmp[tCIDLib::EMatrix_11] = vecScale.f8X();
    mtrxTmp[tCIDLib::EMatrix_22] = vecScale.f8Y();
    mtrxTmp[tCIDLib::EMatrix_33] = vecScale.f8Z();

    // Concatenate it
    *this *= mtrxTmp;
}

tCIDLib::TVoid
T4By4Matrix::AppendScale(   const   tCIDLib::TFloat8&   f8XScale
                            , const tCIDLib::TFloat8&   f8YScale
                            , const tCIDLib::TFloat8&   f8ZScale)
{
    // Create a temporary matrix and set up the scaling
    T4By4Matrix    mtrxTmp;
    mtrxTmp[tCIDLib::EMatrix_11] = f8XScale;
    mtrxTmp[tCIDLib::EMatrix_22] = f8YScale;
    mtrxTmp[tCIDLib::EMatrix_33] = f8ZScale;

    // Concatenate it
    *this *= mtrxTmp;
}


tCIDLib::TVoid T4By4Matrix::AppendTranslation(const T3DVector& vecTrans)
{
    // Create a temporary matrix and set up the translation
    T4By4Matrix mtrxTmp;
    mtrxTmp[tCIDLib::EMatrix_Xt] = vecTrans.f8X();
    mtrxTmp[tCIDLib::EMatrix_Yt] = vecTrans.f8Y();
    mtrxTmp[tCIDLib::EMatrix_Zt] = vecTrans.f8Z();

    // Concatenate it
    *this *= mtrxTmp;
}

tCIDLib::TVoid
T4By4Matrix::AppendTranslation( const   tCIDLib::TFloat8&   f8XTrans
                                , const tCIDLib::TFloat8&   f8YTrans
                                , const tCIDLib::TFloat8&   f8ZTrans)
{
    // Create a temporary matrix and set up the translation
    T4By4Matrix mtrxTmp;
    mtrxTmp[tCIDLib::EMatrix_Xt] = f8XTrans;
    mtrxTmp[tCIDLib::EMatrix_Yt] = f8YTrans;
    mtrxTmp[tCIDLib::EMatrix_Zt] = f8ZTrans;

    // Concatenate it
    *this *= mtrxTmp;
}


tCIDLib::TVoid T4By4Matrix::SetToIdentity()
{
    TRawMem::SetMemBuf(__af8Matrix, tCIDLib::TCard1(0), sizeof(__af8Matrix));
    __af8Matrix[0][0]   = 1.0;
    __af8Matrix[1][1]   = 1.0;
    __af8Matrix[2][2]   = 1.0;
    __af8Matrix[3][3]   = 1.0;
}


tCIDLib::TVoid T4By4Matrix::SetToTransposed(const T4By4Matrix& mtrxSrc)
{
    // Flip the order of the indices
    for (tCIDLib::TInt4 i4Outer = 0; i4Outer < 4; i4Outer++)
    {
        for (tCIDLib::TInt4 i4Inner = 0; i4Inner < 4; i4Inner++)
        {
            __af8Matrix[i4Outer][i4Inner]
                            = mtrxSrc.__af8Matrix[i4Inner][i4Outer];
        }
    }
}


tCIDLib::TVoid T4By4Matrix::Transform(T3DVector& vecTarget) const
{
    tCIDLib::TFloat8 f8X, f8Y, f8Z;

    f8X =   (vecTarget.__f8XMag * __af8Matrix[0][0])
            + (vecTarget.__f8YMag * __af8Matrix[1][0])
            + (vecTarget.__f8ZMag * __af8Matrix[2][0])
            + __af8Matrix[3][0];

    f8Y =   (vecTarget.__f8XMag * __af8Matrix[0][1])
            + (vecTarget.__f8YMag * __af8Matrix[1][1])
            + (vecTarget.__f8ZMag * __af8Matrix[2][1])
            + __af8Matrix[3][1];

    f8Z =   (vecTarget.__f8XMag * __af8Matrix[0][2])
            + (vecTarget.__f8YMag * __af8Matrix[1][2])
            + (vecTarget.__f8ZMag * __af8Matrix[2][2])
            + __af8Matrix[3][2];

    vecTarget.__f8XMag = f8X;
    vecTarget.__f8YMag = f8Y;
    vecTarget.__f8ZMag = f8Z;
}

tCIDLib::TVoid
T4By4Matrix::Transform(const T3DVector& vecSource, T3DVector& vecTarget) const
{
    vecTarget.__f8XMag =
    (
        (vecSource.__f8XMag * __af8Matrix[0][0])
        + (vecSource.__f8YMag * __af8Matrix[1][0])
        + (vecSource.__f8ZMag * __af8Matrix[2][0])
        + __af8Matrix[3][0]
    );

    vecTarget.__f8YMag = 
    (
        (vecSource.__f8XMag * __af8Matrix[0][1])
        + (vecSource.__f8YMag * __af8Matrix[1][1])
        + (vecSource.__f8ZMag * __af8Matrix[2][1])
        + __af8Matrix[3][1]
    );

    vecTarget.__f8ZMag =
    (
        (vecSource.__f8XMag * __af8Matrix[0][2])
        + (vecSource.__f8YMag * __af8Matrix[1][2])
        + (vecSource.__f8ZMag * __af8Matrix[2][2])
        + __af8Matrix[3][2]
    );
}


tCIDLib::TVoid T4By4Matrix::Transpose()
{
    tCIDLib::TFloat8 af8Tmp[4][4];

    // Copy the current matrix to the temp array
    TRawMem::CopyMemBuf(af8Tmp, __af8Matrix, sizeof(__af8Matrix));

    // Now put them back in the opposite order
    for (tCIDLib::TInt4 i4Outer = 0; i4Outer < 4; i4Outer++)
    {
        for (tCIDLib::TInt4 i4Inner = 0; i4Inner < 4; i4Inner++)
            __af8Matrix[i4Outer][i4Inner] = af8Tmp[i4Inner][i4Outer];
    }
}


tCIDLib::TVoid T4By4Matrix::Zero()
{
    TRawMem::SetMemBuf(__af8Matrix, tCIDLib::TCard1(0), sizeof(__af8Matrix));
}


// -----------------------------------------------------------------------------
//  T4By4Matrix: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid T4By4Matrix::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Save the stream state
    TStreamJanitor janWrite(&strmToWriteTo);

    strmToWriteTo   << TTextStream::Width(10) << TTextStream::Precision(4)
                    << TTextStream::Justify(tCIDLib::EHJustify_Right)
                    << TTextStream::Fill(kCIDLib::chSpace);

    tCIDLib::TCard4 c4Index;
    for (c4Index = 0; c4Index < tCIDLib::EMatrixElems_Count; c4Index++)
    {
        strmToWriteTo << __af8Array[c4Index];
        
        if (c4Index != tCIDLib::EMatrixElems_Max)
            strmToWriteTo << L", ";
    }
}


tCIDLib::TVoid T4By4Matrix::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom.ReadFloat8Array(__af8Array, tCIDLib::EMatrixElems_Count);
}

tCIDLib::TVoid T4By4Matrix::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo.WriteFloat8Array(__af8Array, tCIDLib::EMatrixElems_Count);
}
