//
// NAME: CIDLib_3DVector.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_3DVector.Cpp module. This module
//  implements the T3DVector class, which is a mathematical construct that
//  represent forces, position, velocity, magnetic phenomena, etc... in a
//  3D floating point coordinate space.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: T3DVector
// PREFIX: vec3
// -----------------------------------------------------------------------------
class CIDLIBEXP T3DVector :

    public TObject, public MStreamable, public MDuplicable, public MFormattable
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and destructors
        // ---------------------------------------------------------------------
        T3DVector();

        T3DVector
        (
            const   tCIDLib::TFloat8&       f8XMag
            , const tCIDLib::TFloat8&       f8YMag
            , const tCIDLib::TFloat8&       f8ZMag
        );

        T3DVector
        (
            const   T3DVector&               vecSrc
        );

        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        T3DVector& operator=
        (
            const   T3DVector&              vecToAssign
        );

        friend T3DVector CIDLIBEXP operator+
        (
            const   T3DVector&              vec1
            , const T3DVector&              vec2
        );

        friend T3DVector CIDLIBEXP operator+
        (
            const   T3DVector&              vec1
            , const tCIDLib::TFloat8&       f8Inc
        );

        tCIDLib::TVoid operator+=
        (
            const   T3DVector&              vecSrc
        );

        tCIDLib::TVoid operator+=
        (
            const   tCIDLib::TFloat8&       f8Inc
        );

        friend T3DVector CIDLIBEXP operator-
        (
            const   T3DVector&              vec1
            , const T3DVector&              vec2
        );

        friend T3DVector CIDLIBEXP operator-
        (
            const   T3DVector&              vec1
            , const tCIDLib::TFloat8&       f8Dec
        );

        tCIDLib::TVoid operator-=
        (
            const   T3DVector&              vecSrc
        );

        tCIDLib::TVoid operator-=
        (
            const   tCIDLib::TFloat8&       f8Dec
        );

        friend T3DVector CIDLIBEXP operator*
        (
            const   T3DVector&              vec1
            , const T3DVector&              vec2
        );

        friend T3DVector CIDLIBEXP operator*
        (
            const   T3DVector&              vec1
            , const tCIDLib::TFloat8&       f8Mul
        );

        tCIDLib::TVoid operator*=
        (
            const   T3DVector&              vecSrc
        );

        tCIDLib::TVoid operator*=
        (
            const   tCIDLib::TFloat8&       f8Mul
        );

        friend T3DVector CIDLIBEXP operator/
        (
            const   T3DVector&              vec1
            , const T3DVector&              vecDiv
        );

        friend T3DVector CIDLIBEXP operator/
        (
            const   T3DVector&              vec1
            , const tCIDLib::TFloat8&       f8Div
        );

        tCIDLib::TVoid operator/=
        (
            const   T3DVector&              vecDiv
        );

        tCIDLib::TVoid operator/=
        (
            const   tCIDLib::TFloat8&       f8Div
        );

        tCIDLib::TBoolean operator==
        (
            const   T3DVector&              vecToAssign
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   T3DVector&              vecToAssign
        )   const;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid Abs();

        tCIDLib::TVoid AddScaled
        (
            const   T3DVector&              vecSrc
            , const tCIDLib::TFloat8&       f8Scale
        );

        tCIDLib::TVoid Adjust
        (
            const   tCIDLib::TFloat8&       f8X
            , const tCIDLib::TFloat8&       f8Y
            , const tCIDLib::TFloat8&       f8Z
        );

        tCIDLib::TBoolean bAtOrg() const;

        tCIDLib::TBoolean bWithinLimits
        (
            const   tCIDLib::TFloat8&       f8Limit
        )   const;

        tCIDLib::TFloat8 f8Dot
        (
            const   T3DVector&               vec2
        )   const;

        tCIDLib::TFloat8 f8X() const;

        tCIDLib::TFloat8 f8X
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TFloat8 f8Y() const;

        tCIDLib::TFloat8 f8Y
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TFloat8 f8Z() const;

        tCIDLib::TFloat8 f8Z
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TFloat8 f8Magnitude() const;

        tCIDLib::TFloat8 f8MagSum() const;

        tCIDLib::TVoid ClipToDegrees();

        tCIDLib::TVoid Cross
        (
            const   T3DVector&              vec2
        );

        tCIDLib::TVoid HalfBetween
        (
            const   T3DVector&              vec2
        );

        tCIDLib::TVoid Mix
        (
            const   T3DVector&              vec2
        );

        tCIDLib::TVoid Negate();

        tCIDLib::TVoid Normalize();

        tCIDLib::TVoid Set
        (
            const   tCIDLib::TFloat8&       f8XMag
            , const tCIDLib::TFloat8&       f8YMag
            , const tCIDLib::TFloat8&       f8ZMag
        );

        tCIDLib::TVoid Square();

        tCIDLib::TVoid ToRadians();


        // ---------------------------------------------------------------------
        //  Friend methods
        // ---------------------------------------------------------------------
        friend T3DVector CIDLIBEXP vecCross
        (
            const   T3DVector&              vec1
            , const T3DVector&              vec2
        );

        friend T3DVector CIDLIBEXP vecHalfBetween
        (
            const   T3DVector&              vec1
            , const T3DVector&              vec2
        );

        friend T3DVector CIDLIBEXP vecMix
        (
            const   T3DVector&               vec1
            , const T3DVector&               vec2
        );


    protected           :
        // ---------------------------------------------------------------------
        //  Declare our friends
        // ---------------------------------------------------------------------
        friend class    T4By4Matrix;


        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;



    private :
        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __f8XMag
        //  __f8YMag
        //  __f8ZMag
        //      These are the magnitude values in the x,y, and z directions.
        // ---------------------------------------------------------------------
        tCIDLib::TFloat8        __f8XMag;
        tCIDLib::TFloat8        __f8YMag;
        tCIDLib::TFloat8        __f8ZMag;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(T3DVector,TObject)
        DefPolyDup(T3DVector);
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Public operators
// -----------------------------------------------------------------------------
inline T3DVector& T3DVector::operator=(const T3DVector& vecToAssign)
{
    __f8XMag = vecToAssign.__f8XMag;
    __f8YMag = vecToAssign.__f8YMag;
    __f8ZMag = vecToAssign.__f8ZMag;
    return *this;
}

inline tCIDLib::TVoid T3DVector::operator-=(const tCIDLib::TFloat8& f8Dec)
{
    __f8XMag -= f8Dec;
    __f8YMag -= f8Dec;
    __f8ZMag -= f8Dec;
}

inline tCIDLib::TVoid T3DVector::operator+=(const tCIDLib::TFloat8& f8Inc)
{
    __f8XMag += f8Inc;
    __f8YMag += f8Inc;
    __f8ZMag += f8Inc;
}

inline T3DVector operator*( const   T3DVector&          vec1
                            , const tCIDLib::TFloat8&   f8Scale)
{
    return T3DVector
    (
        vec1.__f8XMag * f8Scale
        , vec1.__f8YMag * f8Scale
        , vec1.__f8ZMag * f8Scale
    );
}

inline T3DVector operator*(const T3DVector& vec1, const T3DVector& vec2)
{
    return T3DVector
    (
        vec1.__f8XMag * vec2.__f8XMag
        , vec1.__f8YMag * vec2.__f8YMag
        , vec1.__f8ZMag * vec2.__f8ZMag
    );
}

inline tCIDLib::TVoid T3DVector::operator*=(const tCIDLib::TFloat8& f8Mul)
{
    __f8XMag *= f8Mul;
    __f8YMag *= f8Mul;
    __f8ZMag *= f8Mul;
}

inline tCIDLib::TVoid T3DVector::operator/=(const tCIDLib::TFloat8& f8Div)
{
    __f8XMag /= f8Div;
    __f8YMag /= f8Div;
    __f8ZMag /= f8Div;
}

inline tCIDLib::TBoolean
T3DVector::operator!=(const T3DVector& vecToAssign) const
{
    return !operator==(vecToAssign);
}


// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TVoid
T3DVector::AddScaled(const T3DVector& vecSrc, const tCIDLib::TFloat8& f8Scale)
{
    __f8XMag += vecSrc.__f8XMag * f8Scale; 
    __f8YMag += vecSrc.__f8YMag * f8Scale; 
    __f8ZMag += vecSrc.__f8ZMag * f8Scale; 
}

inline tCIDLib::TVoid
T3DVector::Adjust(  const   tCIDLib::TFloat8&   f8X
                    , const tCIDLib::TFloat8&   f8Y
                    , const tCIDLib::TFloat8&   f8Z)
{
    __f8XMag += f8X;
    __f8YMag += f8Y;
    __f8ZMag += f8Z;
}

inline tCIDLib::TFloat8 T3DVector::f8X() const
{
    return __f8XMag;
}

inline tCIDLib::TFloat8 T3DVector::f8X(const tCIDLib::TFloat8& f8New)
{
    __f8XMag=f8New;
    return __f8XMag;
}

inline tCIDLib::TFloat8 T3DVector::f8Y() const
{
    return __f8YMag;
}

inline tCIDLib::TFloat8 T3DVector::f8Y(const tCIDLib::TFloat8& f8New)
{
    __f8YMag=f8New;
    return __f8YMag;
}

inline tCIDLib::TFloat8 T3DVector::f8Z() const
{
    return __f8ZMag;
}

inline tCIDLib::TFloat8 T3DVector::f8Z(const tCIDLib::TFloat8& f8New)
{
    __f8ZMag=f8New;
    return __f8ZMag;
}

inline tCIDLib::TFloat8 T3DVector::f8MagSum() const
{
    return __f8XMag + __f8YMag + __f8ZMag;
}

inline tCIDLib::TVoid T3DVector::Negate()
{
    __f8XMag *= -1;
    __f8YMag *= -1;
    __f8ZMag *= -1;
}

inline tCIDLib::TVoid
T3DVector::Set( const   tCIDLib::TFloat8&   f8XMag
                , const tCIDLib::TFloat8&   f8YMag
                , const tCIDLib::TFloat8&   f8ZMag)
{
    __f8XMag = f8XMag;
    __f8YMag = f8YMag;
    __f8ZMag = f8ZMag;
}

inline tCIDLib::TVoid T3DVector::Square()
{
    __f8XMag *= __f8XMag;
    __f8YMag *= __f8YMag;
    __f8ZMag *= __f8ZMag;
}

inline T3DVector vecHalfBetween(const T3DVector& vec1, const T3DVector& vec2)
{
    return T3DVector
    (
        0.5 * (vec1.__f8XMag + vec2.__f8XMag)
        , 0.5  * (vec1.__f8YMag + vec2.__f8YMag)
        , 0.5  * (vec1.__f8ZMag + vec2.__f8ZMag)
    );
}

inline T3DVector vecMix(const T3DVector& vec1, const T3DVector& vec2)
{
    return T3DVector
    (
        vec1.__f8XMag * vec2.__f8YMag
        , vec1.__f8XMag * vec2.__f8ZMag
        , vec1.__f8YMag * vec2.__f8ZMag
    );
}


// -----------------------------------------------------------------------------
//  Typedef the vector to T3DPoint so that it can be more self documenting
//  when used as a point in a 3D floating point space.
// -----------------------------------------------------------------------------
typedef T3DVector   T3DPoint;
