//
// NAME: CIDLib_3DVector.Cpp
//
// DESCRIPTION:
//
//  This module implements the T3DVector class.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 09/10/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData(T3DVector,TObject)




// -----------------------------------------------------------------------------
//   CLASS: T3DVector
//  PREFIX: vec3
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  T3DVector: Constructor and Destructors
// -----------------------------------------------------------------------------

T3DVector::T3DVector() :

    __f8XMag(0.0)
    , __f8YMag(0.0)
    , __f8ZMag(0.0)
{
}

T3DVector::T3DVector(   const   tCIDLib::TFloat8&   f8XMag
                        , const tCIDLib::TFloat8&   f8YMag
                        , const tCIDLib::TFloat8&   f8ZMag) :
    __f8XMag(f8XMag)
    , __f8YMag(f8YMag)
    , __f8ZMag(f8ZMag)
{
}

T3DVector::T3DVector(const T3DVector& vecSrc) :

    __f8XMag(vecSrc.__f8XMag)
    , __f8YMag(vecSrc.__f8YMag)
    , __f8ZMag(vecSrc.__f8ZMag)
{
}



// -----------------------------------------------------------------------------
//  T3DVector: Public operators
// -----------------------------------------------------------------------------

tCIDLib::TBoolean T3DVector::operator==(const T3DVector& vecToTest) const
{
    // Compare the values of the objects
    if (vecToTest.__f8XMag != __f8XMag)
        return kCIDLib::False;

    if (vecToTest.__f8YMag != __f8YMag)
        return kCIDLib::False;

    if (vecToTest.__f8ZMag != __f8ZMag)
        return kCIDLib::False;

    return kCIDLib::True;
}


T3DVector operator+(const T3DVector& vec1, const T3DVector& vec2)
{
    return T3DVector
    (
        vec1.__f8XMag + vec2.__f8XMag
        , vec1.__f8YMag + vec2.__f8YMag
        , vec1.__f8ZMag + vec2.__f8ZMag
    );
}

T3DVector operator+(const T3DVector& vec1, const tCIDLib::TFloat8& f8Inc)
{
    return T3DVector
    (
        vec1.__f8XMag + f8Inc
        , vec1.__f8YMag + f8Inc
        , vec1.__f8ZMag + f8Inc
    );
}

tCIDLib::TVoid T3DVector::operator+=(const T3DVector& vecSrc)
{
    __f8XMag += vecSrc.__f8XMag;
    __f8YMag += vecSrc.__f8YMag;
    __f8ZMag += vecSrc.__f8ZMag;
}


T3DVector operator-(const T3DVector& vec1, const T3DVector& vec2)
{
    return T3DVector
    (
        vec1.__f8XMag - vec2.__f8XMag
        , vec1.__f8YMag - vec2.__f8YMag
        , vec1.__f8ZMag - vec2.__f8ZMag
    );
}

T3DVector operator-(const T3DVector& vec1, const tCIDLib::TFloat8& f8Div)
{
    return T3DVector
    (
        vec1.__f8XMag - f8Div
        , vec1.__f8YMag - f8Div
        , vec1.__f8ZMag - f8Div
    );
}

tCIDLib::TVoid T3DVector::operator-=(const T3DVector& vecSrc)
{
    __f8XMag -= vecSrc.__f8XMag;
    __f8YMag -= vecSrc.__f8YMag;
    __f8ZMag -= vecSrc.__f8ZMag;
}

tCIDLib::TVoid T3DVector::operator*=(const T3DVector& vecSrc)
{
    __f8XMag *= vecSrc.__f8XMag;
    __f8YMag *= vecSrc.__f8YMag;
    __f8ZMag *= vecSrc.__f8ZMag;
}


T3DVector operator/(const T3DVector& vec1, const T3DVector& vec2)
{
    return T3DVector(vec1.__f8XMag / vec2.__f8XMag
                    , vec1.__f8YMag / vec2.__f8YMag
                    , vec1.__f8ZMag / vec2.__f8ZMag);
}

tCIDLib::TVoid T3DVector::operator/=(const T3DVector& vecDiv)
{
    __f8XMag /= vecDiv.__f8XMag;
    __f8YMag /= vecDiv.__f8YMag;
    __f8ZMag /= vecDiv.__f8ZMag;
}

T3DVector operator/(const   T3DVector&          vec1
                    , const tCIDLib::TFloat8&   f8Div)
{
    return T3DVector
    (
        vec1.__f8XMag / f8Div
        , vec1.__f8YMag / f8Div
        , vec1.__f8ZMag / f8Div
    );
}


// -----------------------------------------------------------------------------
//  T3DVector: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid T3DVector::Abs()
{
    __f8XMag = TMathLib::f8Abs(__f8XMag);
    __f8YMag = TMathLib::f8Abs(__f8YMag);
    __f8ZMag = TMathLib::f8Abs(__f8ZMag);
}


tCIDLib::TBoolean T3DVector::bAtOrg() const
{
    if ((__f8XMag == 0.0) && (__f8YMag == 0.0) && (__f8ZMag == 0.0))
        return kCIDLib::True;
     else
        return kCIDLib::False;
}


tCIDLib::TBoolean
T3DVector::bWithinLimits(const tCIDLib::TFloat8& f8Limit) const
{
    if ((__f8XMag > f8Limit)
    ||  (__f8YMag > f8Limit)
    ||  (__f8ZMag > f8Limit)
    ||  (__f8XMag < -f8Limit)
    ||  (__f8YMag < -f8Limit)
    ||  (__f8ZMag < -f8Limit))
    {
        return kCIDLib::False;
    }
    return kCIDLib::True;
}


tCIDLib::TFloat8 T3DVector::f8Dot(const T3DVector& vec2) const
{
    return  (__f8XMag * vec2.__f8XMag)
            + (__f8YMag * vec2.__f8YMag)
            + (__f8ZMag * vec2.__f8ZMag);
}


tCIDLib::TFloat8 T3DVector::f8Magnitude() const
{
    return TMathLib::f8SqrRoot
    (
        (__f8XMag * __f8XMag)
        + (__f8YMag * __f8YMag)
        + (__f8ZMag * __f8ZMag)
    );
}


tCIDLib::TVoid T3DVector::ClipToDegrees()
{
    if (__f8XMag > 360.0)
        __f8XMag = TMathLib::f8Mod(__f8XMag, 360.0);

    if (__f8YMag > 360.0)
        __f8YMag = TMathLib::f8Mod(__f8YMag, 360.0);

    if (__f8ZMag > 360.0)
        __f8ZMag = TMathLib::f8Mod(__f8ZMag, 360.0);
}


tCIDLib::TVoid T3DVector::Normalize()
{
    tCIDLib::TFloat8 f8Mag = TMathLib::f8SqrRoot
    (
        (__f8XMag * __f8XMag)
        + (__f8YMag * __f8YMag)
        + (__f8ZMag * __f8ZMag)
    );

    if (f8Mag)
    {
        __f8XMag /= f8Mag;
        __f8YMag /= f8Mag;
        __f8ZMag /= f8Mag;
    }
}


tCIDLib::TVoid T3DVector::ToRadians()
{
    if (TMathLib::f8Abs(__f8XMag) > 360.0)
        __f8XMag = TMathLib::f8Mod(__f8XMag, 360.0);

    if (TMathLib::f8Abs(__f8YMag) > 360.0)
        __f8YMag = TMathLib::f8Mod(__f8YMag, 360.0);

    if (TMathLib::f8Abs(__f8ZMag) > 360.0)
        __f8ZMag = TMathLib::f8Mod(__f8ZMag, 360.0);

    __f8XMag *= kCIDLib::f8PI / 180.0;
    __f8YMag *= kCIDLib::f8PI / 180.0;
    __f8ZMag *= kCIDLib::f8PI / 180.0;
}


tCIDLib::TVoid T3DVector::Cross(const T3DVector& vec2)
{
    tCIDLib::TFloat8 f8XTmp =   (__f8YMag * vec2.__f8ZMag)
                                - (__f8ZMag * vec2.__f8YMag);
    tCIDLib::TFloat8 f8YTmp =   (__f8ZMag * vec2.__f8XMag)
                                - (__f8XMag * vec2.__f8ZMag);
    tCIDLib::TFloat8 f8ZTmp =   (__f8XMag * vec2.__f8YMag)
                                - (__f8YMag * vec2.__f8XMag);
    __f8XMag = f8XTmp;
    __f8YMag = f8YTmp;
    __f8ZMag = f8ZTmp;
}


tCIDLib::TVoid T3DVector::HalfBetween(const T3DVector& vec2)
{
    __f8XMag = 0.5 * (__f8XMag + vec2.__f8XMag);
    __f8YMag = 0.5 * (__f8YMag + vec2.__f8YMag);
    __f8ZMag = 0.5 * (__f8ZMag + vec2.__f8ZMag);
}

tCIDLib::TVoid T3DVector::Mix(const T3DVector& vec2)

{
    __f8XMag *= vec2.__f8YMag;
    __f8XMag *= vec2.__f8ZMag;
    __f8YMag *= vec2.__f8ZMag;
}       

T3DVector vecCross(const T3DVector& vec1, const T3DVector& vec2)
{
    return T3DVector
    (
        (vec1.__f8YMag * vec2.__f8ZMag) - (vec1.__f8ZMag * vec2.__f8YMag)
        , (vec1.__f8ZMag * vec2.__f8XMag) - (vec1.__f8XMag * vec2.__f8ZMag)
        , (vec1.__f8XMag * vec2.__f8YMag) - (vec1.__f8YMag * vec2.__f8XMag)
    );
}



// -----------------------------------------------------------------------------
//  T3DVector: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid T3DVector::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo   << L"{" << __f8XMag
                    << L"," << __f8YMag
                    << L"," << __f8ZMag
                    << L"}";
}


tCIDLib::TVoid T3DVector::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Get the values out
    strmToReadFrom >> __f8XMag;
    strmToReadFrom >> __f8YMag;
    strmToReadFrom >> __f8ZMag;
}

tCIDLib::TVoid T3DVector::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __f8XMag;
    strmToWriteTo << __f8YMag;
    strmToWriteTo << __f8ZMag;
}
