//
// NAME: CIDLib_2DVector.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_2DVector.Cpp module. This module
//  implements the T2DVector class, which is a mathematical construct that
//  represent forces, velocity, magnetic phenomena, etc... in a 2
//  dimensional floating point coordinate space.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: T2DVector
// PREFIX: vec
// -----------------------------------------------------------------------------
class CIDLIBEXP T2DVector :

    public TObject, public MStreamable, public MDuplicable, public MFormattable
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and destructors
        // ---------------------------------------------------------------------
        T2DVector();

        T2DVector
        (
            const   T2DVector&              vecToCopy
        );

        T2DVector
        (
            const   tCIDLib::TFloat8&       f8XMag
            , const tCIDLib::TFloat8&       f8YMag
        );


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        T2DVector& operator=
        (
            const   T2DVector&              vecToAssign
        );

        friend T2DVector CIDLIBEXP operator+
        (
            const   T2DVector&              vec1
            , const T2DVector&              vec2
        );

        tCIDLib::TVoid operator+=
        (
            const   T2DVector&              vecToAdd
        );

        friend T2DVector CIDLIBEXP operator-
        (
            const   T2DVector&              vec1
            , const T2DVector&              vec2
        );

        tCIDLib::TVoid operator-=
        (
            const   T2DVector&              vecToSub
        );

        friend T2DVector CIDLIBEXP operator*
        (
            const   T2DVector&              vec1
            , const T2DVector&              vec2
        );

        tCIDLib::TVoid operator*=
        (
            const   T2DVector&              vecToMul
        );

        tCIDLib::TVoid operator*=
        (
            const   tCIDLib::TFloat8&       f8Factor
        );

        friend T2DVector CIDLIBEXP operator/
        (
            const   T2DVector&              vec1
            , const T2DVector&              vecDiv
        );

        friend T2DVector CIDLIBEXP operator/
        (
            const   T2DVector&              vec1
            , const tCIDLib::TFloat8&       f8Div
        );

        tCIDLib::TVoid operator/=
        (
            const   T2DVector&              vecDiv
        );

        tCIDLib::TVoid operator/=
        (
            const   tCIDLib::TFloat8&       f8Div
        );

        tCIDLib::TBoolean operator==
        (
            const   T2DVector&              vecToCmopare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   T2DVector&              vecToCompare
        )   const;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bAtOrg() const;

        friend tCIDLib::TFloat8 CIDLIBEXP f8Dot
        (
            const   T2DVector&              vec1
            , const T2DVector&              vec2
        );

        tCIDLib::TFloat8 f8Magnitude() const;

        tCIDLib::TFloat8 f8X() const;

        tCIDLib::TFloat8 f8X 
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TFloat8 f8Y() const;

        tCIDLib::TFloat8 f8Y
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TVoid Negate();

        tCIDLib::TVoid Normalize();

        tCIDLib::TVoid Set
        (
            const   tCIDLib::TFloat8&       f8XMag
            , const tCIDLib::TFloat8&       f8YMag
        );

        tCIDLib::TVoid Square();

        tCIDLib::TVoid ToRadians();

        friend T2DVector CIDLIBEXP vecHalfBetween
        (
            const   T2DVector&              vec1
            , const T2DVector&              vec2
        );


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;

    protected           :
        // ---------------------------------------------------------------------
        //  Protected data members
        //
        //  __f8XMag
        //  __f8YMag
        //      These are the magnitude values in the x and y directions.
        // ---------------------------------------------------------------------
        tCIDLib::TFloat8        __f8XMag;
        tCIDLib::TFloat8        __f8YMag;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(T2DVector,TObject)
        DefPolyDup(T2DVector);
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Public operators
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean
T2DVector::operator!=(const T2DVector& vecToAssign) const
{
    return !operator==(vecToAssign);
}


// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TFloat8 T2DVector::f8X() const
{
    return __f8XMag;
}

inline tCIDLib::TFloat8 T2DVector::f8X(const tCIDLib::TFloat8& f8New)
{
    __f8XMag = f8New;
    return __f8XMag;
}

inline tCIDLib::TFloat8 T2DVector::f8Y() const
{
    return __f8YMag;
}

inline tCIDLib::TFloat8 T2DVector::f8Y(const tCIDLib::TFloat8& f8New)
{
    __f8YMag = f8New;
    return __f8YMag;
}

inline tCIDLib::TVoid T2DVector::Negate()

{
    __f8XMag = -__f8XMag;
    __f8YMag = -__f8YMag;
}

inline tCIDLib::TVoid
T2DVector::Set( const   tCIDLib::TFloat8&   f8XMag
                , const tCIDLib::TFloat8&   f8YMag)
{
    __f8XMag = f8XMag;
    __f8YMag = f8YMag;
}

inline tCIDLib::TVoid T2DVector::Square()
{
    __f8XMag *= __f8XMag;
    __f8YMag *= __f8YMag;
}

// -----------------------------------------------------------------------------
//  Typedef the vector to TFPoint so that it can be more self documenting
//  when used as a point in a floating point space.
// -----------------------------------------------------------------------------
typedef T2DVector   TFPoint;
