//
// NAME: CIDLib_2DVector.Cpp
//
// DESCRIPTION:
//
//  This module implements the T2DVector class.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 09/10/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData2(T2DVector,TObject)




// -----------------------------------------------------------------------------
//   CLASS: T2DVector
//  PREFIX: vec2
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  T2DVector: Constructor and Destructors
// -----------------------------------------------------------------------------

T2DVector::T2DVector() :

    __f8XMag(0.0)
    , __f8YMag(0.0)
{
}

T2DVector::T2DVector(const T2DVector& vecSrc) :

    __f8XMag(vecSrc.__f8XMag)
    , __f8YMag(vecSrc.__f8YMag)
{
}

T2DVector::T2DVector(   const   tCIDLib::TFloat8&   f8XMag
                        , const tCIDLib::TFloat8&   f8YMag) :

    __f8XMag(f8XMag)
    , __f8YMag(f8YMag)
{
}


// -----------------------------------------------------------------------------
//  T2DVector: Public operators
// -----------------------------------------------------------------------------

T2DVector& T2DVector::operator=(const T2DVector& vecToAssign)
{
    if (this == &vecToAssign)
        return *this;

    __f8XMag = vecToAssign.__f8XMag;
    __f8YMag = vecToAssign.__f8YMag;
    return *this;
}

T2DVector operator+(const T2DVector& vec1, const T2DVector& vec2)
{
    return T2DVector
    (
        vec1.__f8XMag + vec2.__f8XMag
        , vec1.__f8YMag + vec2.__f8YMag
    );
}

tCIDLib::TVoid T2DVector::operator+=(const T2DVector& vecSrc)
{
    __f8XMag += vecSrc.__f8XMag;
    __f8YMag += vecSrc.__f8YMag;
}

T2DVector operator-(const T2DVector& vec1, const T2DVector& vec2)
{
    return T2DVector
    (
        vec1.__f8XMag - vec2.__f8XMag
        , vec1.__f8YMag - vec2.__f8YMag
    );
}

tCIDLib::TVoid T2DVector::operator-=(const T2DVector& vecSrc)
{
    __f8XMag -= vecSrc.__f8XMag;
    __f8YMag -= vecSrc.__f8YMag;
}

T2DVector operator*(const T2DVector& vec1, const T2DVector& vec2)
{
    return T2DVector
    (
        vec1.__f8XMag * vec2.__f8XMag
        , vec1.__f8YMag * vec2.__f8YMag
    );
}

tCIDLib::TVoid T2DVector::operator*=(const T2DVector& vecSrc)
{
    __f8XMag *= vecSrc.__f8XMag;
    __f8YMag *= vecSrc.__f8YMag;
}

tCIDLib::TVoid T2DVector::operator*=(const tCIDLib::TFloat8& f8Factor)
{
    __f8XMag *= f8Factor;
    __f8YMag *= f8Factor;
}


T2DVector operator/(const T2DVector& vec1, const T2DVector& vec2)
{
    return T2DVector(vec1.__f8XMag / vec2.__f8XMag
                    , vec1.__f8YMag / vec2.__f8YMag);
}

tCIDLib::TVoid T2DVector::operator/=(const T2DVector& vecDiv)
{
    __f8XMag /= vecDiv.__f8XMag;
    __f8YMag /= vecDiv.__f8YMag;
}

T2DVector operator/(const   T2DVector&          vec1
                    , const tCIDLib::TFloat8&   f8Div)
{
    return T2DVector
    (
        vec1.__f8XMag / f8Div
        , vec1.__f8YMag / f8Div
    );
}

tCIDLib::TBoolean T2DVector::operator==(const T2DVector& vecToTest) const
{
    if (this == &vecToTest)
        return kCIDLib::True;

    // Compare the values of the objects
    if (vecToTest.__f8XMag != __f8XMag)
        return kCIDLib::False;

    if (vecToTest.__f8YMag != __f8YMag)
        return kCIDLib::False;

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  T2DVector: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean T2DVector::bAtOrg() const
{
    if ((__f8XMag == 0.0) && (__f8YMag == 0.0))
        return kCIDLib::True;
     else
        return kCIDLib::False;
}

tCIDLib::TFloat8 f8Dot(const T2DVector& vec1, const T2DVector& vec2)
{
    return  (vec1.__f8XMag * vec2.__f8XMag) + (vec1.__f8YMag * vec2.__f8YMag);
}


tCIDLib::TFloat8 T2DVector::f8Magnitude() const
{
    return TMathLib::f8SqrRoot((__f8XMag * __f8XMag) + (__f8YMag * __f8YMag));
}


tCIDLib::TVoid T2DVector::Normalize()
{
    tCIDLib::TFloat8 f8Mag = TMathLib::f8SqrRoot
    (
        (__f8XMag * __f8XMag) + (__f8YMag * __f8YMag)
    );
    __f8XMag /= f8Mag;
    __f8YMag /= f8Mag;
}


tCIDLib::TVoid T2DVector::ToRadians()
{
    if (TMathLib::f8Abs(__f8XMag) > 360.0)
         __f8XMag = TMathLib::f8Mod(__f8XMag, 360.0);

    if (TMathLib::f8Abs(__f8YMag) > 360.0)
        __f8YMag = TMathLib::f8Mod(__f8YMag, 360.0);

    __f8XMag *= kCIDLib::f8PI / 180.0;
    __f8YMag *= kCIDLib::f8PI / 180.0;
}


T2DVector vecHalfBetween(const T2DVector& vec1, const T2DVector& vec2)
{
    return T2DVector
    (
        0.5 * (vec1.__f8XMag + vec2.__f8XMag)
        , 0.5  * (vec1.__f8YMag + vec2.__f8YMag)
    );
}


// -----------------------------------------------------------------------------
//  T2DVector: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid T2DVector::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo << L"{" << __f8XMag << L"," << __f8YMag << L"}";
}

tCIDLib::TVoid T2DVector::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Get the values out
    strmToReadFrom >> __f8XMag;
    strmToReadFrom >> __f8YMag;
}

tCIDLib::TVoid T2DVector::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Write out the data members
    strmToWriteTo << __f8XMag;
    strmToWriteTo << __f8YMag;
}
