//
//  FILE NAME: CIDLib.Cpp
//
//  DESCRIPTION:
//
//  This is the main module of the facility. This guy provides the DLL
//  init function in order to handle initialization of any modules that
//  require it
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/23/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"
#include    "CIDLib_ModuleInit_.Hpp"


// ----------------------------------------------------------------------------
//  Global data
//
//  facCIDLib
//      This is the facility object for CIDLib.Dll. Its exported via the
//      main header.
// ----------------------------------------------------------------------------
TFacCIDLib  facCIDLib;


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
static tCIDLib::TVoid __InitTermMain
(
    const   tCIDLib::EInitTerm      eInitTerm
    , const tCIDLib::EGlobalStates  eGlobals
    , const TModule&                modInit
    , const tCIDLib::TCard4         c4MaxChars
    ,       tCIDLib::Tch* const     pszFailReason
);



// ----------------------------------------------------------------------------
//  Data for local use
//
//  __afnInitTerm
//      This is the list of init/term functions for this facility.
//
//  __bReleaseEntered
//      This is used to prevent multiple invocations of the release of the
//      main thread. The first thread to get there provides its desired
//      exit code and other threads after that are ignored.
//
//  __c4InitCount
//      The number of entries in the __afnInitTerm array.
//
//  __eProcessExit
//      This is set by the _ReleaseMainThread() during normal exits. The
//      value is that returned by the primary thread upon return from the
//      user code. When the main thread exits from CIDLib_MTHibernate()
//      it will return this value.
//
//  __pkevMainBlock
//      This event semaphore is used to block the main thread until the
//      program exits.
// ----------------------------------------------------------------------------
static tCIDLib::TModuleInitTermRec  __afnInitTerm[] =
{
      { __InitTermMain              , 0 , kCIDMsgs::midInit_Main }
    , { _InitTermLocale             , 0 , kCIDMsgs::midInit_Locale }
    , { _InitTermMetrics            , 0 , kCIDMsgs::midInit_Metrics }
    , { _InitTermTypeRegistry       , 0 , kCIDMsgs::midInit_TypeReg }
    , { _InitTermMod                , 0 , kCIDMsgs::midInit_ModuleSupport }
    , { _InitTermSysInfo            , 0 , kCIDMsgs::midInit_SystemInfo }
    , { _InitTermProcessRegistry    , 0 , kCIDMsgs::midInit_ProcRegistry }
    , { _InitTermThisFac            , 0 , kCIDMsgs::midInit_Facility }
    , { _InitTermThread             , 0 , kCIDMsgs::midInit_Threading }
    , { _InitTermTextStream         , 0 , kCIDMsgs::midInit_TextSteams }
};
static tCIDLib::TBoolean            __bReleaseEntered = kCIDLib::False;
static tCIDLib::TCard4              __c4InitCount = c4ArrayElems(__afnInitTerm);
static tCIDLib::EExitCodes          __eProcessExit;
static TKrnlEvent*                  __pkevMainBlock;



// ----------------------------------------------------------------------------
//  Do some RTTI macros for classes that are all inline (i.e. they don't
//  have a Cpp file to do it in.)
// ----------------------------------------------------------------------------
RTTIData(TCriticalSection,TObject)
RTTIData(TCritSecLock,TObject)
RTTIData(TLocker,TObject)
RTTIData(TLogger,TObject)
RTTIData(TSafeCounter,TObject)
RTTIData(TUniqueName,TObject)


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: CIDLib_Init
//
// DESCRIPTION:
//
//  This is called from the magic startup code. Its called after the main()
//  function has been entered and before, CIDLib_MTHibernate(), below, is
//  called to put the main thread asleep. Global objects exist by now.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid CIDLib_Init()
{
    // Nothing to do right now
    return;
}


//
// FUNCTION/METHOD NAME: CIDLib_MTHibernate
//
// DESCRIPTION:
//
//  This is called by the wmain() macro after it called CIDLib_Init(), above,
//  and starts the user's main thread. It just puts the primary thread to
//  sleep forever. A call to _ReleaseMainThread(), below, will let it go
//  and cause a normal exit.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: The __eProcExit value left for us.
//
tCIDLib::EExitCodes CIDLib_MTHibernate()
{
    // Bblock forever, which is the default value for bWaitFor().
    try
    {
        __pkevMainBlock->WaitFor();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        _KRNLERRPOPUP_
        (
            facCIDLib.pszLoadCIDMsg(kCIDMsgs::midMain_MainBlockFailed)
            , 0
            , kerrToCatch
        )
        return tCIDLib::EExit_FatalError;
    }

    // Return any process exit value that was left for us
    return __eProcessExit;
}


tCIDLib::TVoid __InitTermMain(  const   tCIDLib::EInitTerm      eInitTerm
                                , const tCIDLib::EGlobalStates  eGlobals
                                , const TModule&                modInit
                                , const tCIDLib::TCard4         c4MaxChars
                                ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszReason = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Unknown);

    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //
            //  Load up some primal strings so that we have their translatable
            //  text available if something goes wrong during init. These are
            //  really const pointers, so we have to cast off their const'ness
            //  in order to point them at their loaded values. If the load fails,
            //  then their default values will remain for use.
            //
            pszReason = modInit.pszLoadCIDMsg(kCIDMsgs::midMain_LoadingTitles);
            const tCIDLib::Tch* pszTmp = modInit.pszLoadCIDMsg
            (
                kCIDMsgs::midGen_Title1
            );

            const tCIDLib::Tch* pszNC;
            if (pszTmp)
            {
                pszNC = const_cast<tCIDLib::Tch*>(kCIDLib_::pszTitle1);
                pszNC = pszTmp;
            }

            pszTmp = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Title2);
            if (pszTmp)
            {
                pszNC = const_cast<tCIDLib::Tch*>(kCIDLib_::pszTitle2);
                pszNC = pszTmp;
            }

            // Load up the init module strings
            pszReason = modInit.pszLoadCIDMsg(kCIDMsgs::midMain_LoadInitMods);
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4InitCount; c4Index++)
            {
                __afnInitTerm[c4Index].pszDescription = modInit.pszLoadCIDMsg
                (
                    __afnInitTerm[c4Index].midDescription
                );
            }

            // Create the mutex used to block the main thread
            pszReason = modInit.pszLoadCIDMsg(kCIDMsgs::midMain_CreateMutex);
            __pkevMainBlock = new TKrnlEvent;
            __pkevMainBlock->Create(tCIDLib::EEventState_Reset);
        }
    }

    catch(...)
    {
        TRawStr::CopyStr
        (
            pszFailReason
            , pszReason
            , c4MaxChars
        );
        throw;
    }
}


// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _ReleaseMainThread
//
// DESCRIPTION:
//
//  This is an intrafacility function that is called to let the (always
//  otherwise blocked) main thread go. This will cause an exit of the
//  process. 
// -------------------------------------
//   INPUT: eProcExit is a process exit code to leave for the main thread
//              to return.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid _ReleaseMainThread(const tCIDLib::EExitCodes eProcExit)
{
    if (!__bReleaseEntered)
    {
        // Synchronize this in case two threads come in at once
        TBaseLock lockSerialize;

        if (__bReleaseEntered)
            return;

        __bReleaseEntered = kCIDLib::True;
    }

    //
    //  Store the passed exit code for the main thread to get when it
    //  wakes up.
    //
    __eProcessExit = eProcExit;

    //
    //  Let the main thread go if we got far enough to even get the
    //  block event set up.
    //
    if (__pkevMainBlock->bValid())
        __pkevMainBlock->Trigger();
}


// ----------------------------------------------------------------------------
//  DLL Initialization and program entry point
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: CIDLibInit
//
// DESCRIPTION:
//
//  This is the facility init function. It is called from the host OS'
//  init code and it just turns around and calls the standard init method,
//  passing along the info it needs to do the init operation.
// -------------------------------------
//   INPUT: hmodThis is the module handle of this DLL.
//          eInitTerm is the reason we are getting called.
//          pReserved is not used of course.
//
//  OUTPUT: None
//
//  RETURN: 1 if successful, else 0. If a 0 is returned, then the load of
//              the client app will fail.
//
tCIDLib::TCard4 DLLINITAPI
CIDLibInit( tCIDLib::TModHandle     hmodThis
            , tCIDLib::EInitTerm    eInitTerm
            , tCIDLib::TVoid*       pDummy)
{
    // Do our initialization and termination stuff
    if (!bCIDLib_DoModuleInitTerm
    (
        eInitTerm
        , hmodThis
        , tCIDLib::EModType_Dll
        , __c4InitCount
        , __afnInitTerm
        , kCIDLib_::pszTitle1
        , kCIDLib_::pszTitle2))
    {
        return 0;
    }
    return 1;
}
