//
//  FILE NAME: CIDKernel_Time.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/20/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_Time.Cpp module. This module
//  implements the TKrnlTimeStamp class.
//
//
//  CAVEATS/GOTCHAS:
//
//  1)  The FormatTime() and FormatDate() methods require that the string be at
//      least 8 and 10 chars long respectively.
//
//  2)  This class assumes that its CIDLib sibling class, TTime, will never set
//      the time stamp value to a negative number. If it does, then bad things
//      are likely to happen.
//

#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TKrnlTimeStamp
//  PREFIX: ktms
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlTimeStamp
{
    public  :
        // -------------------------------------------------------------------
        //  Constructors and destructors
        // -------------------------------------------------------------------
        TKrnlTimeStamp();

        TKrnlTimeStamp
        (
			const   tCIDLib::TInt8&         i8Time
        );

        TKrnlTimeStamp
        (
            const   TKrnlTimeStamp&         ktmsToCopy
        );

        ~TKrnlTimeStamp();


        // -------------------------------------------------------------------
        //  Public operators
        // -------------------------------------------------------------------
        TKrnlTimeStamp& operator=
        (
            const   TKrnlTimeStamp&         ktmsToAssign
        );

        TKrnlTimeStamp& operator=
        (
            const   tCIDLib::TInt8&         i8ToAssign
        );


        // --------------------------------------------------------------------
        //  Public, static methods
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean bIsLeapYear
        (
            const   tCIDLib::TCard4         c4Year
        );

        static tCIDLib::TBoolean bIsValidDate
        (
            const   tCIDLib::TCard4         c4Day
            , const tCIDLib::EMonths        eMonth
            , const tCIDLib::TCard4         c4Year
        );

        static tCIDLib::TBoolean bIsValidTime
        (
            const   tCIDLib::TCard4         c4Hours
            , const tCIDLib::TCard4         c4Minutes
            , const tCIDLib::TCard4         c4Seconds
            , const tCIDLib::TCard4         c4Hundredths
        );

        static tCIDLib::TCard4 c4MaxDayForMonth
        (
            const   tCIDLib::EMonths        eMonth
            , const tCIDLib::TCard4         c4Year
        );


        // -------------------------------------------------------------------
        //  Public, non-virtual methods
        // -------------------------------------------------------------------
        tCIDLib::TVoid Add
        (
            const   tCIDLib::TInt8&         i8ToAdd
        );

        tCIDLib::TBoolean bValidate() const;

        tCIDLib::TCard4 c4AsTimeInfo
        (
                    tCIDLib::TCard4&        c4Hour
            ,       tCIDLib::TCard4&        c4Minute
            ,       tCIDLib::TCard4&        c4Second    = NUL_TCard4
        )   const;

        tCIDLib::TCard4 c4JulianDate() const;

        tCIDLib::EWeekDays eAsDateInfo
        (
                    tCIDLib::TCard4&        c4Year
            ,       tCIDLib::EMonths&       eMonth
            ,       tCIDLib::TCard4&        c4Day
        )   const;

        tCIDLib::EWeekDays eDayOfWeek() const;

        tCIDLib::TVoid FromDetails
        (
            const   tCIDLib::TCard4         c4Year
            , const tCIDLib::EMonths        eMonth
            , const tCIDLib::TCard4         c4Day
            , const tCIDLib::TCard4         c4Hour      = 0
            , const tCIDLib::TCard4         c4Minute    = 0
            , const tCIDLib::TCard4         c4Second    = 0
            , const tCIDLib::TCard4         c4MilliSecs = 0
        );

        tCIDLib::TInt8 i8Difference
        (
            const   TKrnlTimeStamp&         ktmsToSubtract
        )   const;

        tCIDLib::TInt8 i8MilliSeconds() const;

        tCIDLib::TInt8 i8Time() const;

        tCIDLib::TVoid SetTo
        (
            const   tCIDLib::TInt8&         i8New
        );

        tCIDLib::TVoid SetToNow();

        tCIDLib::TVoid SetToMidnight();

        tCIDLib::TVoid Subtract
        (
            const   tCIDLib::TInt8&         i8ToSubtract
        );


    private :
        // -------------------------------------------------------------------
        //  Private data
        //
        //  __i8Time
        //      This is the time, which is the most convenient way to
        //      store it internally. Its in 100-nanosecond intervals since
        //      1601.
        // -------------------------------------------------------------------
        tCIDLib::TInt8          __i8Time;
};

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Constructors and destructors
// ----------------------------------------------------------------------------
inline TKrnlTimeStamp::TKrnlTimeStamp() :

    __i8Time(0)
{
}

inline TKrnlTimeStamp::TKrnlTimeStamp(const tCIDLib::TInt8& i8Time) :

    __i8Time(i8Time)
{
    // Make sure its positive
    if (__i8Time < 0)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNegativeResult);
}

inline TKrnlTimeStamp::TKrnlTimeStamp(const TKrnlTimeStamp& ktmsToCopy) :

    __i8Time(ktmsToCopy.__i8Time)
{
}

inline TKrnlTimeStamp::~TKrnlTimeStamp()
{
}


// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------
inline TKrnlTimeStamp&
TKrnlTimeStamp::operator=(const TKrnlTimeStamp& ktmsToAssign)
{
    __i8Time = ktmsToAssign.__i8Time;
    return *this;
}

inline TKrnlTimeStamp&
TKrnlTimeStamp::operator=(const tCIDLib::TInt8& i8ToAssign)
{
    __i8Time = i8ToAssign;
    return *this;
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TInt8
TKrnlTimeStamp::i8Difference(const TKrnlTimeStamp& ktmsToSubtract) const
{
    return __i8Time - ktmsToSubtract.__i8Time;
}

inline tCIDLib::TInt8 TKrnlTimeStamp::i8Time() const
{
    return __i8Time;
}

inline tCIDLib::TVoid TKrnlTimeStamp::SetTo(const tCIDLib::TInt8& i8New)
{
    __i8Time = i8New;
    if (__i8Time < 0)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNegativeResult);
}
