//
//  FILE NAME: CIDKernel_Time.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/20/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the TKrnlTimeStamp class. It represents a time
//  in 100 nano-second intervals from the base time of 01/01/1601.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"



// -----------------------------------------------------------------------------
//  Local constant values
//
//  __c4CntDays
//      The count of days that have accumulated by the time each month starts.
//
//  __c4MonDays
//      The number of days in each month, in a non-leap year.
// -----------------------------------------------------------------------------
static const tCIDLib::TCard4 __c4CntDays[12] =
{
        0,  31,  59,  90, 120, 151
    , 181, 212, 243, 273, 304, 334
};

static const tCIDLib::TCard4 __c4MonDays[12] =
{
       31,  28,  31,  30,  31,  30
    ,  31,  31,  30,  31,  30,  31
};




// ----------------------------------------------------------------------------
//   CLASS: TKrnlTimeStamp
//  PREFIX: ktms
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlTimeStamp: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TKrnlTimeStamp::bIsLeapYear(const tCIDLib::TCard4 c4Year)
{
    if (!(c4Year % 4) && ((c4Year % 100) || !(c4Year % 1000)))
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean
TKrnlTimeStamp::bIsValidDate(   const   tCIDLib::TCard4     c4Day
                                , const tCIDLib::EMonths    eMonth
                                , const tCIDLib::TCard4     c4Year)
{
    if ((c4Year < kCIDLib::c4MinYear) || (c4Year > kCIDLib::c4MaxYear))
        return kCIDLib::False;

    if ((eMonth < tCIDLib::EMonths_Min) || (eMonth > tCIDLib::EMonths_Max))
        return kCIDLib::False;

    if ((c4Day < 1) || (c4Day > c4MaxDayForMonth(eMonth, c4Year)))
        return kCIDLib::False;

    return kCIDLib::True;
}

tCIDLib::TBoolean
TKrnlTimeStamp::bIsValidTime(   const   tCIDLib::TCard4 c4Hours
                                , const tCIDLib::TCard4 c4Minutes
                                , const tCIDLib::TCard4 c4Seconds
                                , const tCIDLib::TCard4 c4Hundredths)
{
    if (c4Hours> 23)
        return kCIDLib::False;

    if (c4Minutes> 59)
        return kCIDLib::False;

    if (c4Seconds> 59)
        return kCIDLib::False;

    if (c4Hundredths > 99)
        return kCIDLib::False;

    return kCIDLib::True;
}


tCIDLib::TCard4
TKrnlTimeStamp::c4MaxDayForMonth(   const   tCIDLib::EMonths    eMonth
                                    , const tCIDLib::TCard4     c4Year)
{
    if (bIsLeapYear(c4Year) && (eMonth == tCIDLib::EMonth_February))
            return 29;
    return __c4MonDays[eMonth];
}



// ----------------------------------------------------------------------------
//  TKrnlTimeStamp: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlTimeStamp::Add(const tCIDLib::TInt8& i8ToAdd)
{
    __i8Time += i8ToAdd;
}


tCIDLib::TBoolean TKrnlTimeStamp::bValidate() const
{
    FILETIME    FlTime;
    FlTime.dwLowDateTime = TRawBits::c4Low32From64(__i8Time);
    FlTime.dwHighDateTime = TRawBits::c4High32From64(__i8Time);

    SYSTEMTIME  SysTime;
    if (!FileTimeToSystemTime(&FlTime, &SysTime))
        return kCIDLib::False;

    return kCIDLib::True;
}


tCIDLib::EWeekDays
TKrnlTimeStamp::eAsDateInfo(tCIDLib::TCard4&        c4Year
                            , tCIDLib::EMonths&     eMonth
                            , tCIDLib::TCard4&      c4Day) const
{
    // Get our time into the file time
    FILETIME    FlTime;
    FlTime.dwLowDateTime = TRawBits::c4Low32From64(__i8Time);
    FlTime.dwHighDateTime = TRawBits::c4High32From64(__i8Time);

    SYSTEMTIME  SysTime;
    if (!FileTimeToSystemTime(&FlTime, &SysTime))
        TKrnlError::ThrowKrnlError();

    c4Year      = SysTime.wYear;
    eMonth      = tCIDLib::EMonths(SysTime.wMonth-1);
    c4Day       = SysTime.wDay;
 
    return tCIDLib::EWeekDays(SysTime.wDayOfWeek);
}


tCIDLib::TCard4
TKrnlTimeStamp::c4AsTimeInfo(   tCIDLib::TCard4&    c4Hour
                                , tCIDLib::TCard4&  c4Minute
                                , tCIDLib::TCard4&  c4Second) const
{
    // Get our time into the file time
    FILETIME    FlTime;
    FlTime.dwLowDateTime = TRawBits::c4Low32From64(__i8Time);
    FlTime.dwHighDateTime = TRawBits::c4High32From64(__i8Time);

    SYSTEMTIME  SysTime;
    if (!FileTimeToSystemTime(&FlTime, &SysTime))
        TKrnlError::ThrowKrnlError();

    c4Hour      = SysTime.wHour;
    c4Minute    = SysTime.wMinute;
    c4Second    = SysTime.wSecond;
    return SysTime.wMilliseconds;
}


tCIDLib::TCard4 TKrnlTimeStamp::c4JulianDate() const
{
    // Get our time into the file time
    FILETIME    FlTime;
    FlTime.dwLowDateTime = TRawBits::c4Low32From64(__i8Time);
    FlTime.dwHighDateTime = TRawBits::c4High32From64(__i8Time);

    SYSTEMTIME  SysTime;
    if (!FileTimeToSystemTime(&FlTime, &SysTime))
        return kCIDLib::False;

    tCIDLib::TCard4     c4Year  = SysTime.wYear;
    tCIDLib::EMonths    eMonth  = tCIDLib::EMonths(SysTime.wMonth-1);

    // Get the rough value from the array   
    tCIDLib::TCard4 c4Julian = __c4CntDays[eMonth];

    // If a leap year and past feb, then add one for leap day
    if (bIsLeapYear(c4Year) && (eMonth > tCIDLib::EMonth_February))
        c4Julian++;

    return c4Julian;
}


tCIDLib::EWeekDays TKrnlTimeStamp::eDayOfWeek() const
{
    FILETIME    FlTime;
    FlTime.dwLowDateTime = TRawBits::c4Low32From64(__i8Time);
    FlTime.dwHighDateTime = TRawBits::c4High32From64(__i8Time);

    SYSTEMTIME  SysTime;
    FileTimeToSystemTime(&FlTime, &SysTime);

    if (SysTime.wDayOfWeek > tCIDLib::EWeekDays_Max)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcBadDayOfWeek);

    return tCIDLib::EWeekDays(SysTime.wDayOfWeek);
}


tCIDLib::TVoid
TKrnlTimeStamp::FromDetails(const   tCIDLib::TCard4     c4Year
                            , const tCIDLib::EMonths    eMonth
                            , const tCIDLib::TCard4     c4Day
                            , const tCIDLib::TCard4     c4Hour
                            , const tCIDLib::TCard4     c4Minute
                            , const tCIDLib::TCard4     c4Second
                            , const tCIDLib::TCard4     c4MilliSecs)
{
    SYSTEMTIME  SysTime;

    SysTime.wYear       = WORD(c4Year);
    SysTime.wMonth      = WORD(eMonth+1);
    SysTime.wDay        = WORD(c4Day);
    SysTime.wHour       = WORD(c4Hour);
    SysTime.wMinute     = WORD(c4Minute);
    SysTime.wSecond     = WORD(c4Second);
    SysTime.wMilliseconds=WORD(c4MilliSecs);

    FILETIME    FlTime;
    if (!SystemTimeToFileTime(&SysTime, &FlTime))
        TKrnlError::ThrowKrnlError();

    __i8Time = TRawBits::i8From32(FlTime.dwLowDateTime, FlTime.dwHighDateTime);
}


tCIDLib::TInt8 TKrnlTimeStamp::i8MilliSeconds() const
{
    if (__i8Time < 10000)
        return 0;

    return __i8Time / 10000;
}


tCIDLib::TVoid TKrnlTimeStamp::SetToNow()
{
    SYSTEMTIME  SysTime;
    GetLocalTime(&SysTime);

    FILETIME FlTime;
    if (!SystemTimeToFileTime(&SysTime, &FlTime))
        TKrnlError::ThrowKrnlError();

    __i8Time = TRawBits::i8From32(FlTime.dwLowDateTime, FlTime.dwHighDateTime);
}

tCIDLib::TVoid TKrnlTimeStamp::SetToMidnight()
{
    // Get the current system time
    SYSTEMTIME  SysTime;
    GetLocalTime(&SysTime);

    // Zero out all of the time related fields
    SysTime.wHour           = 0;
    SysTime.wMinute         = 0;
    SysTime.wSecond         = 0;
    SysTime.wMilliseconds   = 0;

    // Convert to file time
    FILETIME    FlTime;
    if (!SystemTimeToFileTime(&SysTime, &FlTime))
        TKrnlError::ThrowKrnlError();

    // And get that back into our value
    __i8Time = TRawBits::i8From32(FlTime.dwLowDateTime, FlTime.dwHighDateTime);
}


tCIDLib::TVoid TKrnlTimeStamp::Subtract(const tCIDLib::TInt8& i8ToSubtract)
{
    __i8Time -= i8ToSubtract;

    if (__i8Time < 0)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNegativeResult);
}
