//
//  FILE NAME: CIDKernel_Process.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/23/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the TKrnlThread class.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"
#include    <process.h>


// ----------------------------------------------------------------------------
//   CLASS: TKrnlThread
//  PREFIX: kthrd
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlThread: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlThread::TKrnlThread() :

    __hthrThis(kCIDLib::hthrInvalid)
    , __tidThis(kCIDLib::tidInvalid)
{
}

TKrnlThread::~TKrnlThread()
{
    // Close the handle
    if (__hthrThis != kCIDLib::hthrInvalid)
    {
        if (!::CloseHandle(__hthrThis))
        {
            #if CID_DEBUG_ON
            TKrnlError kerrToPopup;
            _ERRPOPUP_(kMessages::pszThr_CloseHandle, kerrToPopup.errcId())
            #endif
        }
    }
}


// ----------------------------------------------------------------------------
//  TKrnlThread: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlThread::BeepSpeaker(   const   tCIDLib::TCard4 c4Frequency
                            , const tCIDLib::TCard4 c4MilliSeconds)
{
    Beep(c4Frequency, c4MilliSeconds);
}


tCIDLib::EPrioLevels TKrnlThread::ePriorityOf(const TKrnlThread& kthrToQuery)
{
    if (!kthrToQuery.__hthrThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    tCIDLib::TInt4  i4Level = GetThreadPriority(kthrToQuery.__hthrThis);

    if (i4Level == THREAD_PRIORITY_ERROR_RETURN)
        TKrnlError::ThrowKrnlError();

    // Convert the system value ot our enum type
    tCIDLib::EPrioLevels eLevel = tCIDLib::EPrioLevels(i4Level + 2);

    // Make sure its valid
    if (!bIsValidEnum(eLevel))
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidPriority);

    return eLevel;
}


tCIDLib::TOSErrCode TKrnlThread::errcGetLast()
{
    return ::GetLastError();
}


tCIDLib::TVoid TKrnlThread::Exit(const tCIDLib::EExitCodes eExitCode)
{
    _endthreadex(eExitCode);
}


tCIDLib::TThreadHandle TKrnlThread::hthrCaller()
{
    return GetCurrentThread();
}


tCIDLib::TVoid TKrnlThread::SetLastError(const tCIDLib::TOSErrCode errcToSet)
{
    ::SetLastError(errcToSet);
}


tCIDLib::TVoid TKrnlThread::Sleep(const tCIDLib::TCard4 c4MilliSeconds)
{
    SleepEx(c4MilliSeconds, 0);
}


tCIDLib::TThreadId TKrnlThread::tidCaller()
{
    return GetCurrentThreadId();
}


// ----------------------------------------------------------------------------
//  TKrnlThread: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlThread::AdoptCaller()
{
    if (__hthrThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadySet);

    // Store the stuff
    __hthrThis = TKrnlThread::hthrCaller();
    __tidThis = TKrnlThread::tidCaller();
}


tCIDLib::TVoid
TKrnlThread::AdoptHandle(   const   tCIDLib::TThreadHandle  hToAdopt
                            , const tCIDLib::TThreadId      tidAdopted)
{
    if (__hthrThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadySet);

    // Store the stuff
    __hthrThis = hToAdopt;
    __tidThis = tidAdopted;
}


tCIDLib::TBoolean TKrnlThread::bIsRunning() const
{
    // If the handle is set to an invalid value, then obviously not
    if (__hthrThis == kCIDLib::hthrInvalid)
        return kCIDLib::False;

    // Seems like a valid handle so test it
    tCIDLib::TCard4 c4Res = WaitForSingleObject(__hthrThis, 0);

    if (c4Res == WAIT_FAILED)
        TKrnlError::ThrowKrnlError();

    if (c4Res == WAIT_TIMEOUT)
        return kCIDLib::True;

    return kCIDLib::False;
}


tCIDLib::TVoid
TKrnlThread::BeginThread(   const   TKrnlThread::TThrFuncPtr pfnFunc
                            , const tCIDLib::TCard4          c4StackSize
                            ,       tCIDLib::TVoid* const    pData
                            , const tCIDLib::EThreadFlags    eFlags)
{
    if (__hthrThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyRunning);

    tCIDLib::TThreadHandle  hthrTmp = 0;
    tCIDLib::TThreadId      tidTmp = 0;

    //
    //  Start the thread. Note that we start it blocked so that we have
    //  time to store away its info before it could ask for it.
    //
    typedef unsigned (__stdcall* THostThreadFunc)(void*);
    hthrTmp = (tCIDLib::TVoid*)_beginthreadex
    (
        0
        , c4StackSize
        , THostThreadFunc(pfnFunc)
        , pData
        , eFlags
        , (tCIDLib::TUInt*)&tidTmp
    );

    if (tidTmp == tCIDLib::TThreadId(-1))
        TKrnlError::ThrowKrnlError();

    // It worked so store away the handle and id info
    __tidThis = tidTmp;
    __hthrThis = hthrTmp;

    // Now release the thread
     if (::ResumeThread(hthrTmp) == 0xFFFFFFFF)
        TKrnlError::ThrowKrnlError();
}


tCIDLib::EPrioLevels TKrnlThread::ePriority() const
{
    // Just call the static one and pass ourself as the thread
    return TKrnlThread::ePriorityOf(*this);
}


tCIDLib::TVoid TKrnlThread::SetPriority(const tCIDLib::EPrioLevels eLevel)
{
    if (!SetThreadPriority(__hthrThis, eLevel - 2))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::EExitCodes TKrnlThread::eQueryExitCode() const
{
    tCIDLib::TCard4 c4Code;
    if (!GetExitCodeThread(__hthrThis, &c4Code))
        TKrnlError::ThrowKrnlError();

    if (c4Code == STILL_ACTIVE)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcStillRunning);

    return tCIDLib::EExitCodes(c4Code);
}


tCIDLib::EExitCodes
TKrnlThread::eWaitForDeath(const tCIDLib::TCard4 c4MilliSeconds) const
{
    if (__hthrThis == kCIDLib::hthrInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    tCIDLib::TCard4 c4Res = WaitForSingleObject(__hthrThis, c4MilliSeconds);

    if ((c4Res == WAIT_FAILED)
    ||  (c4Res == WAIT_TIMEOUT)
    ||  (c4Res == WAIT_ABANDONED))
    {
        TKrnlError::ThrowKrnlError();
    }
    return eQueryExitCode();
}
