//
//  FILE NAME: CIDKernel_SystemInfo.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/06/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the TKrnlSysInfo class. This is a simple
//  class that encapsulates the retrieval of general system information.
//  It will precache info that cannot change for the run of the host
//  process, for faster return. Otherwise, it just gets when asked.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"



// ----------------------------------------------------------------------------
//  Intra-facility methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
_InitTermSysInfo(   const   tCIDLib::EInitTerm      eInitTerm
                    , const tCIDLib::EGlobalStates  eGlobals
                    , const tCIDLib::TModHandle     hmodThis
                    , const tCIDLib::TCard4         c4MaxChars
                    ,       tCIDLib::Tch* const     pszFailReason)
{
    // We only have pre-constructor init
    if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
    &&  (eGlobals == tCIDLib::EGlobalState_Before))
    {
        tCIDLib::TCard4 c4Tmp;

        // Query the OS version info
        OSVERSIONINFO   OSVer;
        OSVer.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        if (!GetVersionEx(&OSVer))
        {
            TRawStr::CopyStr
            (
                pszFailReason
                , kMessages::pszSysI_QueryOSVer
                , c4MaxChars
            );
            TKrnlError::ThrowKrnlError();
        }

        // We only support the NT platform
        if (OSVer.dwPlatformId != VER_PLATFORM_WIN32_NT)
        {
            TRawStr::CopyStr
            (
                pszFailReason
                , kMessages::pszSysI_OnlyNT
                , c4MaxChars
            );
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcWrongOS);
        }

        // We survived, so copy over the stuff we want
        TKrnlSysInfo::__c4OSBuildNum    = OSVer.dwBuildNumber;
        TKrnlSysInfo::__c4OSMajVersion  = OSVer.dwMajorVersion;
        TKrnlSysInfo::__c4OSMinVersion  = OSVer.dwMinorVersion;

        //
        //  Query the basic system info from the OS and copy out the stuff
        //  we want into the class' members.
        //
        SYSTEM_INFO SystemInfo = {0};
        GetSystemInfo(&SystemInfo);

        //
        //  Get the count of CPUs, and the type of CPU. It will have defaulted
        //  to 0, which is 'unkown' if we don't set it.
        //
        TKrnlSysInfo::__c4CPUCount = SystemInfo.dwNumberOfProcessors;
        if (SystemInfo.dwProcessorType == PROCESSOR_INTEL_386)
            TKrnlSysInfo::__eCPUType = tCIDLib::ECPUType_386DX;
        else if (SystemInfo.dwProcessorType == PROCESSOR_INTEL_486)
            TKrnlSysInfo::__eCPUType = tCIDLib::ECPUType_486DX;
        else if (SystemInfo.dwProcessorType == PROCESSOR_INTEL_PENTIUM)
            TKrnlSysInfo::__eCPUType = tCIDLib::ECPUType_Pentium;
        else if (SystemInfo.dwProcessorType == PROCESSOR_MIPS_R4000)
            TKrnlSysInfo::__eCPUType = tCIDLib::ECPUType_R4000;
        else if (SystemInfo.dwProcessorType == PROCESSOR_ALPHA_21064)
            TKrnlSysInfo::__eCPUType = tCIDLib::ECPUType_Alpha;

        //
        //  Confirm that our page size constant matches that of the system,
        //  otherwise we will be in bad shape.
        //
        if (SystemInfo.dwPageSize != kCIDLib::c4MemPageSize)
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcHostMemPageSz);

        //
        //  Query the memory information and get the stuff out that we need
        //  to the class members.
        //
        MEMORYSTATUS MemStatus = {0};
        MemStatus.dwLength = sizeof(MEMORYSTATUS);
        GlobalMemoryStatus(&MemStatus);

        // Total physical memory will not change so we can save it
        TKrnlSysInfo::__c4TotalPhysicalMem = MemStatus.dwTotalPhys;

        // Find the name of the workstation that we are running in.
        c4Tmp = c4MaxBufChars(TKrnlSysInfo::__szNodeName);
        if (!GetComputerName(TKrnlSysInfo::__szNodeName, &c4Tmp))
        {
            TRawStr::CopyStr
            (
                pszFailReason
                , kMessages::pszSysI_QueryCompName
                , c4MaxChars
            );
            TKrnlError::ThrowKrnlError();
        }
        TKrnlSysInfo::__szNodeName[c4Tmp] = 0;

        //
        //  Query the module name for the 0 module, which will get us the
        //  main process Exe module's file name.
        //
        const tCIDLib::TCard4 c4Len = c4MaxBufChars(TKrnlSysInfo::__szProcessName);
        tCIDLib::Tch szTmp[c4Len+1];
        if (!GetModuleFileName(0, szTmp, c4Len))
        {
            TRawStr::CopyStr
            (
                pszFailReason
                , kMessages::pszSysI_QueryExeName
                , c4MaxChars
            );
        }

        //
        //  We need to strip out the path part of it. So we find the last slash
        //  and copy everything from there on out to the member string.
        //
        const tCIDLib::Tch* pszRoot = TRawStr::pszFindLastChar
        (
            szTmp
            , kCIDLib::chPathSeparator
        );

        if (!pszRoot)
        {
            TRawStr::CopyStr
            (
                TKrnlSysInfo::__szProcessName
                , szTmp
                , c4MaxBufChars(TKrnlSysInfo::__szProcessName)
            );
        }
        else
        {
            TRawStr::CopyStr
            (
                TKrnlSysInfo::__szProcessName
                , pszRoot+1
                , c4MaxBufChars(TKrnlSysInfo::__szProcessName)
            );
        }

        // Get a pointer to the command line parms
        const tCIDLib::Tch* pszCmd = TKrnlSysInfo::pszCommandLine();

        // Parse it to a list of strings
        tCIDLib::TSInt  iCount;
        tCIDLib::Tch**  apszTmp = CommandLineToArgvW(pszCmd, &iCount);

        // Check for too many parms
        if (iCount >= kCIDLib::c4MaxCmdLineParms)
        {
            TRawStr::CopyStr
            (
                pszFailReason
                , kMessages::pszSysI_TooManyParms
                , c4MaxChars
            );
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcTooMany);
        }

        // Store the argument count
        TKrnlSysInfo::__c4ArgCnt = iCount;

        // Load up any command line args we have
        for (tCIDLib::TCard4 c4Index = 0; c4Index < TKrnlSysInfo::__c4ArgCnt; c4Index++)
        {
            TKrnlSysInfo::__apszArgList[c4Index] =
                                    TRawStr::pszReplicate(apszTmp[c4Index]);
        }
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TKrnlSysInfo
//  PREFIX: ksysi
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlSysInfo: Private, static data
// ----------------------------------------------------------------------------
tCIDLib::Tch*           TKrnlSysInfo::__apszArgList[kCIDLib::c4MaxCmdLineParms];
tCIDLib::TCard4         TKrnlSysInfo::__c4ArgCnt;
tCIDLib::TCard4         TKrnlSysInfo::__c4CPUCount;
tCIDLib::TCard4         TKrnlSysInfo::__c4TotalPhysicalMem;
tCIDLib::ECPUTypes      TKrnlSysInfo::__eCPUType;
tCIDLib::TCard4         TKrnlSysInfo::__c4OSBuildNum;
tCIDLib::TCard4         TKrnlSysInfo::__c4OSMajVersion;
tCIDLib::TCard4         TKrnlSysInfo::__c4OSMinVersion;
tCIDLib::TZStr64        TKrnlSysInfo::__szNodeName;
tCIDLib::TZStr128       TKrnlSysInfo::__szProcessName;

//
//  We need this to be set from ground zero so that any code that might
//  access it will be correct.
//
#if CIDLIB_LITTLEENDIAN
tCIDLib::EEndianModes   TKrnlSysInfo::__eEndianMode = tCIDLib::EEndianMode_Little;
#else
tCIDLib::EEndianModes   TKrnlSysInfo::__eEndianMode = tCIDLib::EEndianMode_Big;
#endif



// ----------------------------------------------------------------------------
//  TKrnlSysInfo: Constructors and Destructors
// ----------------------------------------------------------------------------
TKrnlSysInfo::TKrnlSysInfo()
{
}

TKrnlSysInfo::~TKrnlSysInfo()
{
}


// ----------------------------------------------------------------------------
//  TKrnlSysInfo: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TCard4 TKrnlSysInfo::c4SystemColor(const tCIDLib::ESysColors eColor)
{
    //
    //  <TBD> Move these to the new shell and get them from there. For
    //  now we just return canned values that cannot be changed by the
    //  user.
    //
    //  The encoding is 0xXXBBGGRR, where XX is 0.
    //
    if (eColor == tCIDLib::ESysClr_3DLight)
        return 0xE0E0E0;
    else if (eColor == tCIDLib::ESysClr_3DDark)
        return 0x7F7F7F;
    else if (eColor == tCIDLib::ESysClr_ActiveBorder)
        return 0x00FFFF;
    else if (eColor == tCIDLib::ESysClr_ActiveTitleBgn)
        return 0xA0000;
    else if (eColor == tCIDLib::ESysClr_ActiveTitleText)
        return 0xFFFFFF;
    else if (eColor == tCIDLib::ESysClr_AppWorkspace)
        return 0x7F7F7F;
    else if (eColor == tCIDLib::ESysClr_Background)
        return 0xFFFFFF;
    else if (eColor == tCIDLib::ESysClr_BorderContrast)
        return 0x7F7F7F;
    else if (eColor == tCIDLib::ESysClr_ButtonBgn)
        return 0xC0C0C0;
    else if (eColor == tCIDLib::ESysClr_ButtonText)
        return 0x0;
    else if (eColor == tCIDLib::ESysClr_DialogBgn)
        return 0xC0C0C0;
    else if (eColor == tCIDLib::ESysClr_EntryFieldBgn)
        return 0xFFFFFF;
    else if (eColor == tCIDLib::ESysClr_EntryFieldText)
        return 0x0;
    else if (eColor == tCIDLib::ESysClr_GrayText)
        return 0x7F7F7F;
    else if (eColor == tCIDLib::ESysClr_HiliteBgn)
        return 0xA0000;
    else if (eColor == tCIDLib::ESysClr_HiliteText)
        return 0xFFFFFF;
    else if (eColor == tCIDLib::ESysClr_IconText)
        return 0x0;
    else if (eColor == tCIDLib::ESysClr_InactiveBorder)
        return 0xC0C0C0;
    else if (eColor == tCIDLib::ESysClr_InactiveTitleBgn)
        return 0x7F7F7F;
    else if (eColor == tCIDLib::ESysClr_InactiveTitleText)
        return 0xFFFFFF;
    else if (eColor == tCIDLib::ESysClr_MenuBgn)
        return 0xC0C0C0;
    else if (eColor == tCIDLib::ESysClr_MenuText)
        return 0x0;
    else if (eColor == tCIDLib::ESysClr_ScrollBar)
        return 0x7F7F7F;
    else if (eColor == tCIDLib::ESysClr_StaticText)
        return 0xA00000;
    else if (eColor == tCIDLib::ESysClr_TipsBgn)
        return 0x59DBE6;
    else if (eColor == tCIDLib::ESysClr_TipsText)
        return 0x0;
    else if (eColor == tCIDLib::ESysClr_Window)
        return 0xFFFFFF;
    else if (eColor == tCIDLib::ESysClr_WindowFrame)
        return 0x0;
    else if (eColor == tCIDLib::ESysClr_WindowText)
        return 0x0;
    else
    {
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcBadSysColor);
    }

    // Make the compiler happy
    return 0x0;
}

tCIDLib::TProcessHandle TKrnlSysInfo::hprocThis()
{
    return GetCurrentProcess();
}


tCIDLib::TThreadHandle TKrnlSysInfo::hthrCurrent()
{
    return GetCurrentThread();
}


tCIDLib::TVoid
TKrnlSysInfo::LoadOSMsg(const   tCIDLib::TOSErrCode errcId
                        ,       tCIDLib::Tch* const pszBuffer
                        ,       tCIDLib::TCard4&    c4MaxChars)
{
    c4MaxChars = ::FormatMessageW
    (
        FORMAT_MESSAGE_IGNORE_INSERTS | FORMAT_MESSAGE_FROM_SYSTEM
        , 0
        , errcId
        , MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT)
        , pszBuffer
        , c4MaxChars
        , 0
    );

    if (!c4MaxChars)
        TKrnlError::ThrowKrnlError();

    //
    //  Get rid of any trailing new line that the system messages tend
    //  to have.
    //
    while ( (pszBuffer[c4MaxChars-1] == kCIDLib::chLF)
    ||      (pszBuffer[c4MaxChars-1] == kCIDLib::chCR))
    {
        if (!c4MaxChars)
            break;

        pszBuffer[c4MaxChars-1] = kCIDLib::chNull;
        c4MaxChars--;
    }
}

tCIDLib::TProcessId TKrnlSysInfo::pidThis()
{
    return GetCurrentProcessId();
}


const tCIDLib::Tch* TKrnlSysInfo::pszCommandLine()
{
    return ::GetCommandLineW();
}


const tCIDLib::Tch* TKrnlSysInfo::pszNodeName()
{
    return __szNodeName;
}


const tCIDLib::Tch* TKrnlSysInfo::pszProcessName()
{
    return __szProcessName;
}

tCIDLib::TVoid
TKrnlSysInfo::QueryUserName(        tCIDLib::Tch* const pszBuffer
                            , const tCIDLib::TCard4     c4MaxChars)
{
    tCIDLib::TCard4 c4Chars = c4MaxChars;

    if (!::GetUserNameW(pszBuffer, &c4Chars))
        TKrnlError::ThrowKrnlError();

    pszBuffer[c4Chars] = 0;
}


const tCIDLib::Tch* TKrnlSysInfo::pszCmdLineArg(const tCIDLib::TCard4 c4Index)
{
    if (c4Index >= __c4ArgCnt)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcIndexError);
    return __apszArgList[c4Index];
}


tCIDLib::TVoid TKrnlSysInfo::RemoveCmdLineArg(const tCIDLib::TCard4 c4Index)
{
    if (!c4Index || (c4Index >= __c4ArgCnt))
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcIndexError);

    // Delete the string in this element
    delete __apszArgList[c4Index];

    // If this is the only element, then we are done
    if (__c4ArgCnt == 1)
    {
        __c4ArgCnt = 0;
        return;
    }

    // If this is the last element, we are done
    if (c4Index == __c4ArgCnt-1)
    {
        __c4ArgCnt--;
        return;
    }

    // Move the elements down
    for (tCIDLib::TCard4 c4MoveIndex = c4Index; c4MoveIndex < __c4ArgCnt-1; c4MoveIndex++)
        __apszArgList[c4MoveIndex] = __apszArgList[c4MoveIndex+1];

    __c4ArgCnt--;
}


tCIDLib::TThreadId TKrnlSysInfo::tidCurrent()
{
    return ::GetCurrentThreadId();
}
