//
//  FILE NAME: CIDKernel_SysException.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 10/18/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This file implements the TKrnlSysExcept class, which is a mechanism for
//  encapsulating code inside a system exception handler.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//  Local, static methods
// ----------------------------------------------------------------------------

//
//  Converts the system exception code into one of our system exception
//  enums.
//
static tCIDLib::ESysExcepts
__eConvertSysCode(  const   tCIDLib::TCard4 c4ExceptionCode
                    , const tCIDLib::TCard4 c4Param)
{
    tCIDLib::ESysExcepts eRet = tCIDLib::ESysExcept_Unknown;
    switch(c4ExceptionCode)
    {
        case EXCEPTION_ACCESS_VIOLATION :
            if (c4Param)
                eRet = tCIDLib::ESysExcept_WriteAccess;
            else
                eRet = tCIDLib::ESysExcept_ReadAccess;
            break;

        case EXCEPTION_BREAKPOINT :
            eRet = tCIDLib::ESysExcept_BreakPoint;
            break;

        case EXCEPTION_DATATYPE_MISALIGNMENT :
            eRet = tCIDLib::ESysExcept_DataAlignment;
            break;

        case EXCEPTION_FLT_DENORMAL_OPERAND :
            eRet = tCIDLib::ESysExcept_Float_Denormal;
            break;

        case EXCEPTION_FLT_DIVIDE_BY_ZERO :
            eRet = tCIDLib::ESysExcept_Float_DivByZero;
            break;

        case EXCEPTION_FLT_INEXACT_RESULT :
            eRet = tCIDLib::ESysExcept_Float_InexactResult;
            break;

        case EXCEPTION_FLT_INVALID_OPERATION :
            eRet = tCIDLib::ESysExcept_Float_InvalidOp;
            break;

        case EXCEPTION_FLT_OVERFLOW :
            eRet = tCIDLib::ESysExcept_Float_Overflow;
            break;

        case EXCEPTION_FLT_STACK_CHECK :
            eRet = tCIDLib::ESysExcept_Float_Stack;
            break;

        case EXCEPTION_FLT_UNDERFLOW :
            eRet = tCIDLib::ESysExcept_Float_Underflow;
            break;

        case EXCEPTION_IN_PAGE_ERROR :
            eRet = tCIDLib::ESysExcept_PageLoadFailed;
            break;

        case EXCEPTION_INT_DIVIDE_BY_ZERO :
            eRet = tCIDLib::ESysExcept_Int_DivByZero;
            break;

        case EXCEPTION_INT_OVERFLOW :
            eRet = tCIDLib::ESysExcept_Int_Overflow;
            break;

        case EXCEPTION_STACK_OVERFLOW :
            eRet = tCIDLib::ESysExcept_StackOverflow;
            break;

        default :
            break;
    };
    return eRet;
}

static tCIDLib::TInt4 __i4Filter(TKrnlSysExcept* const          pksexcThis
                                 , const _EXCEPTION_POINTERS*   pExceptInfo)
{
    //
    //  Pull out the information that we store in the object for the
    //  _Handler() method to see.
    //
    pksexcThis->__eExceptCode = __eConvertSysCode
    (
        pExceptInfo->ExceptionRecord->ExceptionCode
        , pExceptInfo->ExceptionRecord->ExceptionInformation[0]
    );

    pksexcThis->__eRecoverable = pExceptInfo->ExceptionRecord->ExceptionFlags ? 
                                    tCIDLib::ERecover_Recoverable :
                                    tCIDLib::ERecover_Unrecoverable;

    pksexcThis->__pAddr = pExceptInfo->ExceptionRecord->ExceptionAddress;

    //
    //  Ask the derived class what it wants to do about this exception
    //  and convert that to the result needed by the system.
    //
    tCIDLib::ESysExceptRes eRes = pksexcThis->_eRecoverTest();

    if (eRes == tCIDLib::ESysExceptRes_Dismiss)
        return EXCEPTION_CONTINUE_EXECUTION;
    else if (eRes == tCIDLib::ESysExceptRes_Propogate)
        return EXCEPTION_CONTINUE_SEARCH;
    else
        return EXCEPTION_EXECUTE_HANDLER;
}


// ----------------------------------------------------------------------------
//   CLASS: TKrnlSysExcept
//  PREFIX: ksexc
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlSysExcept: Constructors and Destructor
// ----------------------------------------------------------------------------

TKrnlSysExcept::TKrnlSysExcept()
{
}

TKrnlSysExcept::~TKrnlSysExcept()
{
}


// ----------------------------------------------------------------------------
//  TKrnlSysExcept: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlSysExcept::DoWork()
{
    __try
    {
        _DoWork();
    }

    __except(__i4Filter(this, GetExceptionInformation()))
    {
        _Handler();
    }
}
