//
// NAME: CIDKernel_StackDump.Cpp
//
// DESCRIPTION:
//
//	This module provides the stack dump functions that are called to get
//  a dump of the call sequence to a place where an error has occured.
//  The methods here are actually statics of TKrnlThread, but they are
//  done here because they are really standalone and very platform
//  specific.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/02/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  This guy uses some raw RTL string functions because some of the
//      APIs we use here don't understand UNICode strings. So we have to
//      manipulate regular strings.
//
//  2)  Also, raw file operations are used here to avoid making use of
//      CIDLib functionality as much as possible.
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"
#include    <imagehlp.h>


// ----------------------------------------------------------------------------
//  Local const data
//
//  __pszNewLine
//      The newline used by the file write functions below. We write the
//      output as non-UNICODE text.
// ----------------------------------------------------------------------------
static const tCIDLib::Tsch* const   __pszNewLine  = "\r\n";


// ----------------------------------------------------------------------------
//  Local static data
//
//  __bSymsLoaded
//      This flag is used to know whether we have already loaded symbols
//      or not. This allows us to only load them once.
//
//  __hflDump
//      This is the handle of the file being dumped to. Its set and then
//      used by all of the APIs. The dump operations are protected so we
//      don't have to worry about thread's stepping on each other's toes.
//
//  __CriticalSec
//      This is the critical section structure that we used to insure that
//      only one thread enters the dump function at a time. Its initilized
//      during DLL init.
//
//  __pszTargetDir
//      The directory into which dump files are placed. This is settable via
//      the CIDLIB_ERRDUMPDIR environment variable. If not set, then they
//      will go into the current directory.
// ----------------------------------------------------------------------------
static tCIDLib::TBoolean    __bSymsLoaded = kCIDLib::False;
static HANDLE               __hflDump;
static CRITICAL_SECTION     __CriticalSec = {0};
static const tCIDLib::Tch*  __pszTargetDir = L"";


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
static tCIDLib::TVoid __FmtCard4
(
    tCIDLib::TCard4         c4Val
    , tCIDLib::Tch* const   pszBuf
    , tCIDLib::ERadices     eRadix = tCIDLib::ERadix_Hex
);

static tCIDLib::TVoid __WriteLine
(
    const   tCIDLib::Tsch* const pszStr
);

static tCIDLib::TVoid __WriteLine
(
    const   tCIDLib::Tch* const pszStr
);

static tCIDLib::TVoid __WriteStr
(
    const   tCIDLib::Tsch* const pszStr
);

static tCIDLib::TVoid __WriteStr
(
    const   tCIDLib::Tch* const pszStr
);


// ----------------------------------------------------------------------------
//  Intra-facility functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid
_InitTermStackDump( const   tCIDLib::EInitTerm      eInitTerm
                    , const tCIDLib::EGlobalStates  eGlobals
                    , const tCIDLib::TModHandle     hmodThis
                    , const tCIDLib::TCard4         c4MaxChars
                    ,       tCIDLib::Tch* const     pszFailReason)
{
    if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
    &&  (eGlobals == tCIDLib::EGlobalState_Before))
    {
        // Init the critical section used to insure only one thread hits at once
        InitializeCriticalSection(&__CriticalSec);

        //
        //  See if there is an environment variable for the target output
        //  directory for dump files.
        //
        tCIDLib::Tch szTmp[512];
        if (::GetEnvironmentVariableW(L"CIDERRDUMPDIR", szTmp, 512))
            __pszTargetDir = TRawStr::pszReplicate(szTmp);
    }
     else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
          &&  (eGlobals == tCIDLib::EGlobalState_After))
    {
        DeleteCriticalSection(&__CriticalSec);

        if (__bSymsLoaded)
        {
            // Free the symbols
            SymCleanup(GetCurrentProcess());
        }
    }
}



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

static tCIDLib::TVoid __FmtCard4(   tCIDLib::TCard4         c4Val
                                    , tCIDLib::Tch* const   pszBuf
                                    , tCIDLib::ERadices     eRadix)
{
    static tCIDLib::Tch chChars[] =
    {
         L'0', L'1', L'2', L'3', L'4', L'5', L'6', L'7'
        ,L'8', L'9', L'A', L'B', L'C', L'D', L'E', L'F'
    };

    tCIDLib::TBoolean   bLeading0 = kCIDLib::True;
    tCIDLib::TCard4     c4Divisor, c4Tmp;
    tCIDLib::TInt4      i4Ind, i4StrInd;

    //
    //  Put a 0 in the first byte just in case the value is zero. Otherwise,
    //  the code below would leave an empty string.
    //
    pszBuf[0] = L'0';
    pszBuf[1] = 0;

    if (eRadix == tCIDLib::ERadix_Hex)
    {
        c4Divisor   = 0x10000000;
        i4Ind       = 7;
    }
     else
    {
        //
        //  Divisor starts at the billions since largest number is
        //  around 4 billion.
        //
        c4Divisor   = 1000000000;
        i4Ind       = 9;
    }

    i4StrInd = 0;
    for (; i4Ind >= 0; i4Ind--)
    {
        c4Tmp = c4Val / c4Divisor;

        // If non-zero make sure that leading zero is cleared
        if (c4Tmp)
            bLeading0 = kCIDLib::False;

        if (!bLeading0 || (eRadix == tCIDLib::ERadix_Hex))
        {
            pszBuf[i4StrInd] = chChars[c4Tmp];
            i4StrInd++;
        }

        // Bump down the value by the modulus
        c4Val -= (c4Tmp * c4Divisor);

        if (eRadix == tCIDLib::ERadix_Hex)
            c4Divisor >>= 4;
         else
            c4Divisor /= 10;
    }

    if (i4StrInd)
        pszBuf[i4StrInd] = 0;
}


static tCIDLib::TVoid __CreateFile()
{
    //
    //  Build the name of the file that we are going to output to. Its the
    //  target directory, plus the process name and the process id. We
    //  build the name itself into a separate directory because, if we cannot
    //  create the file in the target dir, we create in the current directory.
    //
    const tCIDLib::TCard4 c4BufSz = 1024;
    tCIDLib::Tch szFullName[c4BufSz+1];

    szFullName[0] = 0;
    if (__pszTargetDir[0])
    {
        TRawStr::CopyStr(szFullName, __pszTargetDir, c4BufSz);
        if (szFullName[TRawStr::c4StrLen(szFullName)-1] != kCIDLib::chPathSeparator)
            TRawStr::CatStr(szFullName, kCIDLib::szPathSeparator, c4BufSz);
    }

    tCIDLib::Tch szPID[64+1];
    TRawStr::FormatVal(TKrnlSysInfo::pidThis(), szPID, 64);

    tCIDLib::Tch szFileName[c4BufSz+1];
    TRawStr::CopyCatStr
    (
        szFileName
        , c4BufSz
        , TKrnlSysInfo::pszProcessName()
        , L"_"
        , szPID
        , L".ErrorInf"
    );

    // Copy the name onto the directory
    TRawStr::CatStr(szFullName, szFileName, c4BufSz);

    //
    //  Ok, we are doing pretty well so far, so lets open or create
    //  the file and seek to the end in order to append beyond any
    //  current content.
    //
    __hflDump = CreateFile
    (
        szFullName
        , GENERIC_WRITE
        , 0
        , 0
        , OPEN_ALWAYS
        , FILE_ATTRIBUTE_NORMAL
          | FILE_FLAG_SEQUENTIAL_SCAN
        , 0
    );

    if (__hflDump == INVALID_HANDLE_VALUE)
    {
        // If its not because the path is bad, then throw the error
        tCIDLib::TOSErrCode errcRes = TKrnlThread::errcGetLast();
        if (errcRes != ERROR_PATH_NOT_FOUND)
            throw errcRes;

        // Just open the file in the current directory
        __hflDump = CreateFile
        (
            szFileName
            , GENERIC_WRITE
            , 0
            , 0
            , OPEN_ALWAYS
            , FILE_ATTRIBUTE_NORMAL
              | FILE_FLAG_SEQUENTIAL_SCAN
            , 0
        );

        if (__hflDump == INVALID_HANDLE_VALUE)
            throw TKrnlThread::errcGetLast();
    }

    SetFilePointer(__hflDump, 0, 0, FILE_END);
}


static tCIDLib::TVoid __LogBasicInfo(const tCIDLib::Tch* const pszThreadName)
{
    // Dump the process and thread name
    __WriteStr("   Process: ");
    __WriteLine(TKrnlSysInfo::pszProcessName());
    __WriteStr("    Thread: ");
    __WriteLine(pszThreadName);
}


static tCIDLib::TVoid
__LogExceptionCode(const EXCEPTION_RECORD* const pExceptRecord)
{
    __WriteStr(" Excp Type: ");

    tCIDLib::Tsch* pszText;
    switch(pExceptRecord->ExceptionCode)
    {
        case EXCEPTION_ACCESS_VIOLATION :
            pszText = "Access Violation";
            break;

        case EXCEPTION_DATATYPE_MISALIGNMENT :
            pszText = "Data Misalignment";
            break;

        case EXCEPTION_BREAKPOINT :
            pszText = "Breakpoint";
            break;

        case EXCEPTION_SINGLE_STEP :
            pszText = "Single Step";
            break;

        case EXCEPTION_ARRAY_BOUNDS_EXCEEDED :
            pszText = "Array Bounds";
            break;

        case EXCEPTION_FLT_DENORMAL_OPERAND :
            pszText = "Denormal Operand";
            break;

        case EXCEPTION_FLT_DIVIDE_BY_ZERO :
            pszText = "FP Divide by Zero";
            break;

        case EXCEPTION_FLT_INEXACT_RESULT :
            pszText = "Inexact Result";
            break;

        case EXCEPTION_FLT_INVALID_OPERATION :
            pszText = "Invalid FP Operation";
            break;

        case EXCEPTION_FLT_OVERFLOW :
            pszText = "FP Overflow";
            break;

        case EXCEPTION_FLT_STACK_CHECK :
            pszText = "FP Stack Check";
            break;

        case EXCEPTION_FLT_UNDERFLOW :
            pszText = "FP Underflow";
            break;

        case EXCEPTION_INT_DIVIDE_BY_ZERO :
            pszText = "Int Divide by Zero";
            break;

        case EXCEPTION_INT_OVERFLOW :
            pszText = "Int Overflow";
            break;

        case EXCEPTION_PRIV_INSTRUCTION :
            pszText = "Priviledged Instruction";
            break;

        case EXCEPTION_IN_PAGE_ERROR :
            pszText = "Page Error";
            break;

        case EXCEPTION_ILLEGAL_INSTRUCTION :
            pszText = "Illegal Instruction";
            break;

        case EXCEPTION_NONCONTINUABLE_EXCEPTION :
            pszText = "Non-Continuable";
            break;

        case EXCEPTION_STACK_OVERFLOW :
            pszText = "Stack Overflow";
            break;

        case EXCEPTION_INVALID_DISPOSITION :
            pszText = "Invalid Disposition";
            break;

        case EXCEPTION_GUARD_PAGE :
            pszText = "Guard Page";
            break;

        case EXCEPTION_INVALID_HANDLE :
            pszText = "Invalid Handle";
            break;

        case CONTROL_C_EXIT :
            pszText = "Ctrl-C Exit";
            break;

        default :
            pszText = "Unknown";
    }
    __WriteStr(pszText);

    // Dump out the extra info if this is a access violation
    if ((pExceptRecord->ExceptionCode == EXCEPTION_ACCESS_VIOLATION)
    &&  (pExceptRecord->NumberParameters >= 1))
    {
        if (pExceptRecord->ExceptionInformation[0])
            __WriteLine("  (Write access)");
        else
            __WriteLine("  (Read access)");
    }
     else
    {
        __WriteLine("");
    }

    // Dump out the exception address
    __WriteStr("   Address: ");
    tCIDLib::TZStr32 szTmp;
    __FmtCard4((tCIDLib::TCard4)pExceptRecord->ExceptionAddress, szTmp);
    __WriteLine(szTmp);
}


static tCIDLib::TVoid __LogRegisters(const CONTEXT& ContextInfo)
{
    tCIDLib::TZStr64 szTmp;

    if (ContextInfo.ContextFlags & CONTEXT_INTEGER)
    {
        __WriteStr("   EAX:");
        __FmtCard4(ContextInfo.Eax, szTmp);
        __WriteStr(szTmp);

        __WriteStr("   EBX:");
        __FmtCard4(ContextInfo.Ebx, szTmp);
        __WriteStr(szTmp);

        __WriteStr("   ECX:");
        __FmtCard4(ContextInfo.Ecx, szTmp);
        __WriteStr(szTmp);

        __WriteStr("   EDX:");
        __FmtCard4(ContextInfo.Edx, szTmp);
        __WriteLine(szTmp);
    }

    if (ContextInfo.ContextFlags & CONTEXT_SEGMENTS)
    {
        __WriteStr("    DS:");
        __FmtCard4(ContextInfo.SegDs, szTmp);
        __WriteStr(szTmp);

        __WriteStr("    ES:");
        __FmtCard4(ContextInfo.SegEs, szTmp);
        __WriteStr(szTmp);

        __WriteStr("    FS:");
        __FmtCard4(ContextInfo.SegFs, szTmp);
        __WriteStr(szTmp);

        __WriteStr("    GS:");
        __FmtCard4(ContextInfo.SegGs, szTmp);
        __WriteLine(szTmp);
    }

    if (ContextInfo.ContextFlags & CONTEXT_SEGMENTS)
    {
        __WriteStr("    SS:");
        __FmtCard4(ContextInfo.SegSs, szTmp);
        __WriteStr(szTmp);

        __WriteStr("   ESP:");
        __FmtCard4(ContextInfo.Esp, szTmp);
        __WriteStr(szTmp);

        __WriteStr("    CS:");
        __FmtCard4(ContextInfo.SegCs, szTmp);
        __WriteStr(szTmp);

        __WriteStr("   ESP:");
        __FmtCard4(ContextInfo.Esp, szTmp);
        __WriteLine(szTmp);

        __WriteStr("   EBP:");
        __FmtCard4(ContextInfo.Ebp, szTmp);
        __WriteStr(szTmp);

        __WriteStr(" Flags:");
        __FmtCard4(ContextInfo.EFlags, szTmp);
        __WriteLine(szTmp);
    }
}


static tCIDLib::TVoid __StackDump(const CONTEXT& ContextInfo)
{
    if (!__bSymsLoaded)
    {
        //
        //  Initialize the symbol support. We pass 0 for the user path so that
        //  it can be set via the environment (_NT_SYMBOL_PATH=xxx)
        //
        if (!SymInitialize
        (
            GetCurrentProcess()
            , 0
            , TRUE))
        {
            TKrnlError kerrToPopup;
            _ERRPOPUP_(kMessages::pszStDump_InitSyms, kerrToPopup.errcId())
            return;
        }

        // Inidcate that symbols are already loaded
        __bSymsLoaded = kCIDLib::True;
    }

    //
    //  Set up some stuff that can be done once before we enter the
    //  walking loop.
    //
    tCIDLib::TCard1*        aSymBuffer = 0;
    const tCIDLib::TCard4   c4SymBufSize = 1024;
    IMAGEHLP_MODULE         ModuleInfo = {0};
    tCIDLib::Tsch           szTmpStr[1024];

    ModuleInfo.SizeOfStruct = sizeof(IMAGEHLP_MODULE);
    try
    {
        // Allocate the symbol buffer
        aSymBuffer = new tCIDLib::TCard1[c4SymBufSize];
        IMAGEHLP_SYMBOL* pSymInfo = (IMAGEHLP_SYMBOL*)aSymBuffer;

        //
        //  Build the initial stack frame info using the register
        //  context info we got. This will be where we start tracing.
        //
        STACKFRAME StackInfo        = {0};
        StackInfo.AddrPC.Offset     = ContextInfo.Eip;
        StackInfo.AddrPC.Mode       = AddrModeFlat;
        StackInfo.AddrStack.Offset  = ContextInfo.Esp;
        StackInfo.AddrStack.Mode    = AddrModeFlat;
        StackInfo.AddrFrame.Offset  = ContextInfo.Ebp;
        StackInfo.AddrFrame.Mode    = AddrModeFlat;

        //
        //  Now start the stack trace loop. We just loop around until
        //  StackWalk does not find any more frames. Create an empty
        //  context record for the stack walk call.
        //
        CONTEXT Context = {0};
        while (1)
        {
            if (!StackWalk
            (
                IMAGE_FILE_MACHINE_I386
                , TKrnlSysInfo::hprocThis()
                , 0
                , &StackInfo
                , &Context
                , ReadProcessMemory
                , SymFunctionTableAccess
                , SymGetModuleBase
                , 0))
            {
                break;
            }

            //
            //  Get the info for the module in which this symbol
            //  lives.
            //
            if (!SymGetModuleInfo
            (
                TKrnlSysInfo::hprocThis()
                , StackInfo.AddrPC.Offset
                , &ModuleInfo))
            {
                TKrnlError::ThrowKrnlError();
            }

            // Start building up the string for this line
            strcpy(szTmpStr, "    Called From: ");
            strcat(szTmpStr, ModuleInfo.ModuleName);
            strcat(szTmpStr, ", ");

            //
            //  Get the symbol for the address of this stack frame
            //  entry.
            //
            pSymInfo->SizeOfStruct = sizeof(IMAGEHLP_SYMBOL);
            pSymInfo->MaxNameLength = c4SymBufSize - sizeof(IMAGEHLP_SYMBOL);
            tCIDLib::TCard4 c4Displacement = 0;

            if (SymGetSymFromAddr
            (
                TKrnlSysInfo::hprocThis()
                , StackInfo.AddrPC.Offset
                , &c4Displacement
                , pSymInfo))
            {
                strcat(szTmpStr, pSymInfo->Name);
            }
             else
            {
                strcat(szTmpStr, "<no symbolic info>");
            }

            __WriteLine(szTmpStr);
        }
    }

    catch(const TKrnlError& kerrFailure)
    {
        _ERRPOPUP_(kMessages::pszStDump_TraceError, kerrFailure.errcId())
    }

    // Clean up what needs cleaning
    delete aSymBuffer;
}


static tCIDLib::TVoid __WriteStr(const tCIDLib::Tsch* const pszStr)
{
    tCIDLib::TCard4 c4Actual;
    WriteFile(__hflDump, pszStr, strlen(pszStr), &c4Actual, 0);
}

static tCIDLib::TVoid __WriteStr(const tCIDLib::Tch* const pszStr)
{
    tCIDLib::Tsch* pszConverted = TRawStr::pszConvert(pszStr);
    tCIDLib::TCard4 c4Actual;
    WriteFile(__hflDump, pszConverted, strlen(pszConverted), &c4Actual, 0);
    delete pszConverted;
}

static tCIDLib::TVoid __WriteLine(const tCIDLib::Tsch* const pszStr)
{
    tCIDLib::TCard4 c4Actual;
    WriteFile(__hflDump, pszStr, strlen(pszStr), &c4Actual, 0);
    WriteFile(__hflDump, __pszNewLine, strlen(__pszNewLine), &c4Actual, 0);
}

static tCIDLib::TVoid __WriteLine(const tCIDLib::Tch* const pszStr)
{
    tCIDLib::TCard4 c4Actual;
    tCIDLib::Tsch* pszConverted = TRawStr::pszConvert(pszStr);
    WriteFile(__hflDump, pszConverted, strlen(pszConverted), &c4Actual, 0);
    delete pszConverted;
    WriteFile(__hflDump, __pszNewLine, strlen(__pszNewLine), &c4Actual, 0);
}



// ----------------------------------------------------------------------------
//  TKrnlThread: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlThread::DumpException( const   TKrnlThread&            thrCaller
                            , const tCIDLib::Tch* const     pszThreadName
                            , const tCIDLib::TVoid* const   pExceptData)
{
    // Serialize access to the dump
    EnterCriticalSection(&__CriticalSec);

    //
    //  Look at the passed exception data as a system specific excpetion
    //  information structure. We don't want to make this structure
    //  public, so its passed as a void pointer.
    //
    const EXCEPTION_POINTERS* pExcept = (EXCEPTION_POINTERS*)pExceptData;

    // Create the file that we dump to
    __CreateFile();
    if (!__hflDump)
    {
        LeaveCriticalSection(&__CriticalSec);
        return;
    }

    // Write the header for an exception
    __WriteLine("--------------------------------------");
    __WriteLine("Exception in Process: ");

    // Log out the common info that we do for all dumps
    __LogBasicInfo(pszThreadName);

    //
    //  Dump out the type of exception that occured. We get the exception
    //  code and just use a lookup table to format it.
    //
    __LogExceptionCode(pExcept->ExceptionRecord);

    // Dump the registers
    __WriteLine("\r\nRegister Dump:");
    __LogRegisters(*pExcept->ContextRecord);

    // Dump the stack, passing along the register context we got
    __WriteLine("\r\nStack Dump:");
    __StackDump(*pExcept->ContextRecord);

    __WriteLine("\r\n\n");
    CloseHandle(__hflDump);
    __hflDump = 0;

    LeaveCriticalSection(&__CriticalSec);
}


tCIDLib::TVoid
TKrnlThread::DumpRuntimeError(  const   TKrnlThread&        thrCaller
                                , const tCIDLib::Tch* const pszThreadName
                                , const tCIDLib::Tch* const pszFacility
                                , const tCIDLib::Tch* const pszError
                                , const tCIDLib::Tch* const pszAuxText
                                , const tCIDLib::TErrCode   errcId
                                , const tCIDLib::TErrCode   errcKrnlId
                                , const tCIDLib::TOSErrCode errcHostId
                                , const tCIDLib::Tch* const pszFile
                                , const tCIDLib::TCard4     c4LineNumber)
{
    EnterCriticalSection(&__CriticalSec);

    try
    {
        tCIDLib::Tch szTmp[64];

        // Create the file and store the handle locally for use
        __CreateFile();

        // Write out the runtime error header
        __WriteLine("--------------------------------------");
        __WriteLine("Runtime Error: ");

        // Log the common stuff that's logged for all dumps
        __LogBasicInfo(pszThreadName);
        if (!__hflDump)
        {
            LeaveCriticalSection(&__CriticalSec);
            return;
        }

        __WriteLine("\r\nError Information:");
        __WriteStr("        Facility: ");
        __WriteLine(pszFacility);
        __WriteStr("           Error: ");
        __WriteLine(pszError);

        if (pszAuxText)
        {
            __WriteStr("  Aux Error Text: ");
            __WriteLine(pszAuxText);
        }

        __WriteStr("        Error Id: ");
        __FmtCard4(errcId, szTmp, tCIDLib::ERadix_Dec);
        __WriteLine(szTmp);
        __WriteStr(" Kernel Error Id: ");
        __FmtCard4(errcKrnlId, szTmp, tCIDLib::ERadix_Dec);
        __WriteLine(szTmp);
        __WriteStr("   Host Error Id: ");
        __FmtCard4(errcHostId, szTmp, tCIDLib::ERadix_Dec);
        __WriteLine(szTmp);
        __WriteStr("            File: ");
        __WriteStr(pszFile);
        __WriteStr(".");
        __FmtCard4(c4LineNumber, szTmp, tCIDLib::ERadix_Dec);
        __WriteLine(szTmp);

        //
        //  Now do the stack dump. Since we don't get a context record for
        //  runtime errors, we query one to pass in.
        //
        CONTEXT Context = {0};
        Context.ContextFlags = CONTEXT_FULL;
        if (!GetThreadContext(TKrnlSysInfo::hthrCurrent(), &Context))
        {
            LeaveCriticalSection(&__CriticalSec);
            return;
        }
        __WriteLine("\r\nStack Dump:");
        __StackDump(Context);

        __WriteLine("\r\n\n");
    }

    catch(...)
    {
    }

    if (__hflDump)
    {
        CloseHandle(__hflDump);
        __hflDump = 0;
    }

    LeaveCriticalSection(&__CriticalSec);
}
