//
//  FILE NAME: CIDKernel_SharedMemBuf.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/04/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_SharedMemBuf.Cpp module, which
//  implements an simple abstraction for a shared memory buffer. Since
//  a shared memory buffer under NT requires more than just the pointer
//  to the memory (i.e. it requires a handle), and some other platform
//  might require even more, this makes the outside world transparent to
//  those differences.
//
//  Also implemented here is TKrnlTypedSharedBuf, which is a simple template
//  class for use inside the kernel where a shared buffer contains a structure.
//  This template makes life simple and typesafe by presenting the memory
//  buffer's raw buffer as the structure type. Note that CIDLib has its own
//  version of this class for public use, which is more complex and supports
//  streaming and all of that.
//
//  CAVEATS/GOTCHAS:
//

#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TKrnlSharedMemBuf
//  PREFIX: ksmb
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlSharedMemBuf
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKrnlSharedMemBuf();

        TKrnlSharedMemBuf
        (
            const   tCIDLib::Tch* const     pszName
            , const tCIDLib::TCard4         c4Size
            , const tCIDLib::EMemAccFlags   eAccess
            , const tCIDLib::ECreateActions eCreateFlags = tCIDLib::ECreateAct_OpenOrCreate
            ,       tCIDLib::TBoolean&      bCreated = NUL_TBoolean
        );

        TKrnlSharedMemBuf
        (
            const   TKrnlSharedMemBuf&      ksmbToCopy
        );

        ~TKrnlSharedMemBuf();

        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator==
        (
            const   TKrnlSharedMemBuf&      ksmbToCompare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TKrnlSharedMemBuf&      ksmbToCompare
        )   const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Alloc
        (
            const   tCIDLib::Tch* const     pszName
            , const tCIDLib::TCard4         c4Size
            , const tCIDLib::EMemAccFlags   eAccess
            , const tCIDLib::ECreateActions eCreateFlags = tCIDLib::ECreateAct_OpenOrCreate
            ,       tCIDLib::TBoolean&      bCreated = NUL_TBoolean
        );

        tCIDLib::TCard4 c4Size() const;

        tCIDLib::EMemAccFlags eAccess() const;

        tCIDLib::TVoid Free();

        tCIDLib::TVoid* pData();

        const tCIDLib::TVoid* pData() const;

        const tCIDLib::Tch* pszName() const;

        tCIDLib::TVoid Zero();


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TKrnlSharedMemBuf&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4Size
        //      This is the maximum size of the buffer allocated, though it
        //      might not be all committed at any one time.
        //
        //  __eAccess
        //      The access rights used to create the buffer. This is required
        //      for duplication in the copy constructor, but is also a
        //      convenient thing to know.
        //
        //  __hmemBacking
        //      This is the handle to the backing memory for this buffer.
        //
        //  __pBuffer
        //      This is a pointer to the buffer.
        //
        //  __pszName
        //      This the name of the buffer, which is allocated as needed
        //      to hold the given name.
        // --------------------------------------------------------------------
        tCIDLib::TCard4         __c4Size;
        tCIDLib::EMemAccFlags   __eAccess;
        tCIDLib::TMemHandle     __hmemBacking;
        tCIDLib::TVoid*         __pBuffer;
        tCIDLib::Tch*           __pszName;
};


// ----------------------------------------------------------------------------
//   CLASS: TKrnlTypedSharedBuf
//  PREFIX: ktsb
// ----------------------------------------------------------------------------
template <class T> class TKrnlTypedSharedBuf : public TKrnlSharedMemBuf
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKrnlTypedSharedBuf() : TKrnlSharedMemBuf()
        {
        }

        TKrnlTypedSharedBuf
        (
            const   tCIDLib::Tch* const     pszName
            , const tCIDLib::EMemAccFlags   eAccess
            , const tCIDLib::ECreateActions eCreateFlags = tCIDLib::ECreateAct_OpenOrCreate
            ,       tCIDLib::TBoolean&      bCreated = NUL_TBoolean) :

            TKrnlSharedMemBuf(pszName, sizeof(T), eAccess, eCreateFlags, bCreated)
        {
        }

        ~TKrnlTypedSharedBuf()
        {
        }

        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        T* operator->()
        {
            return (T*)pData();
        }

        const T* operator->() const
        {
            return (const T*)pData();
        }

        const T& operator()() const
        {
            return *(const T*)pData();
        }

        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Alloc
        (
            const   tCIDLib::Tch* const     pszName
            , const tCIDLib::EMemAccFlags   eAccess
            , const tCIDLib::ECreateActions eCreateFlags = tCIDLib::ECreateAct_OpenOrCreate
            ,       tCIDLib::TBoolean&      bCreated = NUL_TBoolean)
        {
            TKrnlSharedMemBuf::Alloc
            (
                pszName
                , sizeof(T)
                , eAccess
                , eCreateFlags
                , bCreated
            );
        }


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TKrnlTypedSharedBuf(const TKrnlTypedSharedBuf<T>&);

        tCIDLib::TVoid operator=(const TKrnlTypedSharedBuf<T>&);
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  TKrnlSharedMemBuf: Public operators
// ----------------------------------------------------------------------------

inline tCIDLib::TBoolean
TKrnlSharedMemBuf::operator!=(const TKrnlSharedMemBuf& ksmbToCompare) const
{
    return !operator==(ksmbToCompare);
}


// ----------------------------------------------------------------------------
//  TKrnlSharedMemBuf: Public, non-virtual methods
// ----------------------------------------------------------------------------

inline tCIDLib::TCard4 TKrnlSharedMemBuf::c4Size() const
{
    return __c4Size;
}

inline tCIDLib::EMemAccFlags TKrnlSharedMemBuf::eAccess() const
{
    return __eAccess;
}

inline tCIDLib::TVoid* TKrnlSharedMemBuf::pData()
{
    return __pBuffer;
}

inline const tCIDLib::TVoid* TKrnlSharedMemBuf::pData() const
{
    return __pBuffer;
}

inline const tCIDLib::Tch* TKrnlSharedMemBuf::pszName() const
{
    return __pszName;
}

inline tCIDLib::TVoid TKrnlSharedMemBuf::Zero()
{
    TRawMem::SetMemBuf(__pBuffer, tCIDLib::TCard1(0), __c4Size);
}
