//
//  FILE NAME: CIDKernel_SharedMemBuf.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/04/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the the TKrnlSharedMemBuf, which is a wrapper
//  to abstract the concept of a shared memory buffer. Shared memory
//  buffers can have more baggage than just the pointer to the buffer
//  allocated, so this class abstracts that concept.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//   CLASS: TKrnlSharedMemBuf
//  PREFIX: ksmb
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlSharedMemBuf: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlSharedMemBuf::TKrnlSharedMemBuf() :

    __c4Size(0)
    , __eAccess(tCIDLib::EMemAcc_ReadOnly)
    , __hmemBacking(kCIDLib::hmemInvalid)
    , __pBuffer(0)
    , __pszName(0)
{
}

TKrnlSharedMemBuf::TKrnlSharedMemBuf(const  tCIDLib::Tch* const     pszName
                                    , const tCIDLib::TCard4         c4Size
                                    , const tCIDLib::EMemAccFlags   eAccess
                                    , const tCIDLib::ECreateActions eCreateFlags
                                    ,       tCIDLib::TBoolean&      bCreated) :

    __c4Size(0)
    , __eAccess(tCIDLib::EMemAcc_ReadOnly)
    , __hmemBacking(kCIDLib::hmemInvalid)
    , __pBuffer(0)
    , __pszName(0)
{
    Alloc(pszName, c4Size, eAccess, eCreateFlags, bCreated);
}

TKrnlSharedMemBuf::TKrnlSharedMemBuf(const TKrnlSharedMemBuf& ksmbToCopy) :

    __eAccess(ksmbToCopy.__eAccess)
    , __hmemBacking(kCIDLib::hmemInvalid)
    , __pBuffer(0)
    , __pszName(0)
{
    if (!DuplicateHandle
    (
        TKrnlSysInfo::hprocThis()
        , ksmbToCopy.__hmemBacking
        , TKrnlSysInfo::hprocThis()
        , &__hmemBacking
        , 0, 0
        , DUPLICATE_SAME_ACCESS))
    {
        TKrnlError::ThrowKrnlError();
    }

    tCIDLib::TCard4 c4FileAccess = 0;
    if (__eAccess == tCIDLib::EMemAcc_ReadOnly)
        c4FileAccess = FILE_MAP_READ;
    else if (__eAccess == tCIDLib::EMemAcc_ReadWrite)
        c4FileAccess = FILE_MAP_WRITE;

    // Create a mapping for this new handle
    tCIDLib::TVoid* pBuf = MapViewOfFile(__hmemBacking, c4FileAccess, 0, 0, 0);

    if (!pBuf)
    {
        CloseHandle(__hmemBacking);
        __hmemBacking = kCIDLib::hmemInvalid;
        TKrnlError::ThrowKrnlError();
    }

    // It worked so store away the pointer
    __pBuffer = pBuf;
}

TKrnlSharedMemBuf::~TKrnlSharedMemBuf()
{
    Free();
}


// ----------------------------------------------------------------------------
//  TKrnlSharedMemBuf: Public operators
// ----------------------------------------------------------------------------

tCIDLib::TBoolean
TKrnlSharedMemBuf::operator==(const TKrnlSharedMemBuf& ksmbToCompare) const
{
    if (TRawStr::eCompareStr(__pszName, ksmbToCompare.__pszName))
        return kCIDLib::False;

    if (__eAccess != ksmbToCompare.__eAccess)
        return kCIDLib::False;

    return kCIDLib::True;
}


// ----------------------------------------------------------------------------
//  TKrnlSharedMemBuf: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlSharedMemBuf::Alloc(   const  tCIDLib::Tch* const      pszName
                            , const tCIDLib::TCard4         c4Size
                            , const tCIDLib::EMemAccFlags   eAccess
                            , const tCIDLib::ECreateActions eCreateFlags
                            ,       tCIDLib::TBoolean&      bCreated)
{
    if (__pBuffer)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyAllocated);

    if (!pszName)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNullName);

    // Replicate the name of the buffer
    __pszName = TRawStr::pszReplicate(pszName);

    // Store the size and access flags
    __c4Size = c4Size;
    __eAccess = eAccess;

    //
    //  Check for invalid create modes for a shared memory buffer and
    //  return an error.
    //
    if ((eCreateFlags == tCIDLib::ECreateAct_ReplaceIfExists)
    ||  (eCreateFlags == tCIDLib::ECreateAct_TruncateIfExists)
    ||  (eCreateFlags == tCIDLib::ECreateAct_None))
    {
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidParameter);
    }

    // Convert our access flags to the appropriate ones
    tCIDLib::TCard4 c4Access = SEC_COMMIT;

    if (__eAccess == tCIDLib::EMemAcc_ReadOnly)
        c4Access |= PAGE_READONLY;
    else if (__eAccess == tCIDLib::EMemAcc_ReadWrite)
        c4Access |= PAGE_READWRITE;
    else
    {
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidAccess);
    }

    //
    //  Ok, so now create a file access set of flags that match the
    //  memory access flags we just did above. We need one set for
    //  some calls and one set for others.
    //
    //  We already checked for invalid access, so we don't have to
    //  check again.
    //
    tCIDLib::TCard4 c4FileAccess = 0;
    if (__eAccess == tCIDLib::EMemAcc_ReadOnly)
        c4FileAccess = FILE_MAP_READ;
    else if (__eAccess == tCIDLib::EMemAcc_ReadWrite)
        c4FileAccess = FILE_MAP_WRITE;

    //
    //  Try to create the handle and buffer pointer. Init them first
    //  to zero so we can clean up on exception.
    //  
    tCIDLib::TBoolean   bCreatedBuf = kCIDLib::False;
    tCIDLib::THandle    hFl = 0;
    tCIDLib::TVoid*     pBuf = 0;
    try
    {
        hFl = OpenFileMapping(c4FileAccess, 0, __pszName);
    
        if (!hFl)
        {
            // If the create action is 'open if exists' we are done now
            if (eCreateFlags == tCIDLib::ECreateAct_OpenIfExists)
                TKrnlError::ThrowKrnlError();

            // Ok, so lets try to create it
            hFl = CreateFileMapping
            (
                tCIDLib::THandle(0xFFFFFFFF)
                , 0
                , c4Access
                , 0
                , __c4Size
                , pszName
            );

            // If this one fails, then we just give up
            if (!hFl)
                TKrnlError::ThrowKrnlError();

            bCreatedBuf = kCIDLib::True;
        }
         else
        {
            //
            //  If the flags indicate to fail if it exists, then we need
            //  to close it down and return an already exists error.
            //
            if (eCreateFlags == tCIDLib::ECreateAct_FailIfExists)
                TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyExists);
        }

        // Now create the mapping
        pBuf = MapViewOfFile(hFl, c4FileAccess, 0, 0, __c4Size);

        if (!pBuf)
            TKrnlError::ThrowKrnlError();
    }

    catch(...)
    {
        if (pBuf)
            UnmapViewOfFile(pBuf);

        if (hFl)
            CloseHandle(hFl);

        __c4Size = 0;
        __eAccess = tCIDLib::EMemAcc_ReadOnly;

        throw;
    }

    // Store the created parm, if they wanted it
    if (&bCreated)
        bCreated = bCreatedBuf;

    // Store the handle and buffer pointer
    __hmemBacking = hFl;
    __pBuffer = pBuf;
}


tCIDLib::TVoid TKrnlSharedMemBuf::Free()
{
    // If the name got allocated, then free it
    delete __pszName;
    __pszName = 0;

    // If we got the memory allocated, then free it
    if (__pBuffer)
    {
        if (!UnmapViewOfFile(__pBuffer))
            TKrnlError::ThrowKrnlError();

        if (!CloseHandle(__hmemBacking))
            TKrnlError::ThrowKrnlError();

        __pBuffer = 0;
        __hmemBacking = kCIDLib::hmemInvalid;
    }
}
