//
//  FILE NAME: CIDKernel_Semaphore.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/12/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_Semaphore.Cpp module. This module
//  implements the TKrnlSemaphore class, which is a standard counting
//  semaphore.
//
//  CAVEATS/GOTCHAS:
//
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"



// ----------------------------------------------------------------------------
//   CLASS: TKrnlSemaphore
//  PREFIX: ksem
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlSemaphore: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlSemaphore::TKrnlSemaphore(const tCIDLib::TCard4 c4MaxCount) :

    __c4MaxCount(0)
    , __hsemThis(kCIDLib::hsemInvalid)
    , __pszName(0)
{
}

TKrnlSemaphore::TKrnlSemaphore( const   tCIDLib::Tch* const pszName
                                , const tCIDLib::TCard4     c4MaxCount) :
    __c4MaxCount(0)
    , __hsemThis(kCIDLib::hsemInvalid)
    , __pszName(0)
{
    if (pszName)
        __pszName = TRawStr::pszReplicate(pszName);
}

TKrnlSemaphore::~TKrnlSemaphore()
{
    if (__pszName)
    {
        delete __pszName;
        __pszName = 0;
    }

    if (__hsemThis != kCIDLib::hsemInvalid)
        Close();
}

// ----------------------------------------------------------------------------
//  TKrnlSemaphore: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlSemaphore::Close()
{
    if (__hsemThis != kCIDLib::hsemInvalid)
    {
        tCIDLib::TSemHandle hsemTmp = __hsemThis;
        __hsemThis = kCIDLib::hsemInvalid;

		if (!CloseHandle(hsemTmp))
            TKrnlError::ThrowKrnlError();
	}
}


tCIDLib::TVoid TKrnlSemaphore::Create(const tCIDLib::TCard4 c4InitCount)
{
    if (__hsemThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (__pszName)
    {
        __CreateNamed(c4InitCount, kCIDLib::True);
        return;
    }

    tCIDLib::TSemHandle hsemTmp = CreateSemaphore
    (
        0
        , c4InitCount
        , __c4MaxCount
        , 0
    );

    if (!hsemTmp)
        TKrnlError::ThrowKrnlError();

    // We survived, so store the handle
    __hsemThis = hsemTmp;
}

tCIDLib::TVoid
TKrnlSemaphore::CreateOrOpen(const tCIDLib::TCard4 c4InitCount)
{
    if (__hsemThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (!__pszName)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNullName);

    __CreateNamed(c4InitCount, kCIDLib::False);
}


tCIDLib::TVoid TKrnlSemaphore::Duplicate(const TKrnlSemaphore& ksemToDup)
{
    // Clean up the name
    delete __pszName;
    __pszName = 0;

    // Close it if open
    if (__hsemThis != kCIDLib::hsemInvalid)
        Close();

    // Copy over the info
    __c4MaxCount = ksemToDup.__c4MaxCount;

    // Recreate the name
    if (ksemToDup.__pszName)
        __pszName = TRawStr::pszReplicate(ksemToDup.__pszName);

    // Duplicate the handle
    tCIDLib::TSemHandle hsemTmp;
    if (!DuplicateHandle
    (
        ksemToDup.__hsemThis
        , TKrnlSysInfo::hprocThis()
        , TKrnlSysInfo::hprocThis()
        , &hsemTmp
        , 0
        , 0
        , DUPLICATE_SAME_ACCESS))
    {
        TKrnlError::ThrowKrnlError();
    }

    __hsemThis = hsemTmp;
}


tCIDLib::TVoid TKrnlSemaphore::Enter(const tCIDLib::TCard4 c4MilliSecs)
{
    if (__hsemThis == kCIDLib::hsemInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    tCIDLib::TCard4 c4Reason = WaitForSingleObject(__hsemThis, c4MilliSecs);

    if (c4Reason == WAIT_FAILED)
        TKrnlError::ThrowKrnlError();

    if (c4Reason == WAIT_TIMEOUT)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcTimeout);
}

tCIDLib::TVoid TKrnlSemaphore::Exit()
{
    if (__hsemThis == kCIDLib::hsemInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    if (!ReleaseSemaphore(__hsemThis, 1, 0))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid TKrnlSemaphore::Open()
{
    if (__hsemThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (!__pszName)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNullName);

    tCIDLib::TSemHandle hsemTmp = OpenEvent
    (
        EVENT_ALL_ACCESS | SYNCHRONIZE
        , 0
        , __pszName
    );

    if (!hsemTmp)
        TKrnlError::ThrowKrnlError();

    // We survived, so store the handle
    __hsemThis = hsemTmp;
}


tCIDLib::TVoid TKrnlSemaphore::SetName(const tCIDLib::Tch* const pszToSet)
{
    if (__hsemThis != kCIDLib::hsemInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    delete __pszName;
    __pszName = 0;
    __pszName = TRawStr::pszReplicate(pszToSet);
}



// ----------------------------------------------------------------------------
//  TKrnlSemaphore: Private, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __CreateNamed
//
// DESCRIPTION:
//
//  These guys will create a semahpore, using the different rules that named
//  resources require.
// -------------------------------------
//   INPUT: c4InitCount is the initial count value to give to the semahpore.
//          bFailIfExists indicates whether this is a create operation
//              or a create w/ optional open.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TKrnlSemaphore::__CreateNamed(  const   tCIDLib::TCard4     c4InitCount
                                , const tCIDLib::TBoolean   bFailIfExists)
{
    tCIDLib::TSemHandle hsemTmp;

    //
    //  First try to open it. If that succeeds, then see if we need to
    //  fail if it exists. If so, then close it and return an error.
    //
    hsemTmp = OpenSemaphore(EVENT_ALL_ACCESS, FALSE, __pszName);

    if (hsemTmp)
    {
        if (bFailIfExists)
        {
            CloseHandle(hsemTmp);
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyExists);
        }
        __hsemThis = hsemTmp;
        return;
    }

    // The open did not work, so try to create it
    hsemTmp = CreateSemaphore(0, c4InitCount, __c4MaxCount, __pszName);

    if (!hsemTmp)
        TKrnlError::ThrowKrnlError();

    __hsemThis = hsemTmp;
}
