//
//  FILE NAME: CIDKernel_ResourceName.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/01/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_ResourceName.Cpp module. This module
//  implements the TKrnlRscName class. This class abstracts the names of
//  named resources, so that user code can create portable resource names.
//  A resource name, as defined by CIDLib, consists of 3 separate parts.
//
//  There is a company name, a subsystem name, and a resource name. These
//  form a hierarchy that prevents name clashes. They are stored separatedly
//  inside the TKrnlRscName object, and the full (platform specific) name
//  is created upon demand (via a call to BuildFullName().)
//
//  A process id can be optionally provided to create process unique names.
//
//  CAVEATS/GOTCHAS:
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TKrnlRscName
//  PREFIX: krscn
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlRscName
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKrnlRscName();

        TKrnlRscName
        (
            const   tCIDLib::Tch* const     pszCompany
            , const tCIDLib::Tch* const     pszSubsytsem
            , const tCIDLib::Tch* const     pszResource
            , const tCIDLib::TProcessId     pidToFormat = kCIDLib::pidInvalid
        );

        TKrnlRscName
        (
            const   TKrnlRscName&           krscnToCopy
        );

        ~TKrnlRscName();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TKrnlRscName& operator=
        (
            const   TKrnlRscName&           krscnToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   TKrnlRscName&           krscnToCompare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TKrnlRscName&           krscnToCompare
        )   const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bValid() const;

        tCIDLib::TVoid BuildFullName
        (
                    tCIDLib::Tch* const     pszToFill
            , const tCIDLib::TCard4         c4MaxChars
            , const tCIDLib::ENamedRscTypes eType
        )   const;

        tCIDLib::TProcessId pidOfName() const;

        const tCIDLib::Tch* pszCompanyName() const;

        const tCIDLib::Tch* pszSubsystemName() const;

        const tCIDLib::Tch* pszResourceName() const;

        tCIDLib::TVoid SetName
        (
            const   tCIDLib::Tch* const     pszCompany
            , const tCIDLib::Tch* const     pszSubsytsem
            , const tCIDLib::Tch* const     pszResource
            , const tCIDLib::TProcessId     pidToFormat = kCIDLib::pidInvalid
        );


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pszCompany
        //  __pszSubsystem
        //  __pszResource
        //      These are the three name parts. They are filled in during
        //      construction and are never changed.
        //
        //  __pidOfName
        //      This is the optional process id to format into the string.
        //      It defaults to kCIDLib::pidInvalid, which means it won't
        //      be used.
        // --------------------------------------------------------------------
        tCIDLib::Tch*           __pszCompany;
        tCIDLib::Tch*           __pszSubsystem;
        tCIDLib::Tch*           __pszResource;
        tCIDLib::TProcessId     __pidOfName;
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------

inline tCIDLib::TBoolean
TKrnlRscName::operator!=(const TKrnlRscName& krsnToCompare) const
{
    return !operator==(krsnToCompare);
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------

inline tCIDLib::TBoolean TKrnlRscName::bValid() const
{
    // If any substring is empty, its not valid
    if (!__pszCompany[0] || !__pszSubsystem[0] || !__pszResource[0])
        return kCIDLib::False;
    return kCIDLib::True;
}

inline tCIDLib::TProcessId TKrnlRscName::pidOfName() const
{
    return __pidOfName;
}

inline const tCIDLib::Tch* TKrnlRscName::pszCompanyName() const
{
    return __pszCompany;
}

inline const tCIDLib::Tch* TKrnlRscName::pszSubsystemName() const
{
    return __pszSubsystem;
}

inline const tCIDLib::Tch* TKrnlRscName::pszResourceName() const
{
    return __pszResource;
}
