//
//  FILE NAME: CIDKernel_ResourceName.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/01/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the the TKrnlRscName class, which is an
//  abstraction for the names of shareable resources.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//   CLASS: TKrnlRscName
//  PREFIX: krscn
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlRscName: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlRscName::TKrnlRscName() :

    __pidOfName(kCIDLib::pidInvalid)
    , __pszCompany(0)
    , __pszSubsystem(0)
    , __pszResource(0)
{
    __pszCompany = TRawStr::pszReplicate(kCIDLib::pszEmptyZStr);
    __pszSubsystem = TRawStr::pszReplicate(kCIDLib::pszEmptyZStr);
    __pszResource = TRawStr::pszReplicate(kCIDLib::pszEmptyZStr);
}

TKrnlRscName::TKrnlRscName( const   tCIDLib::Tch* const pszCompany
                            , const tCIDLib::Tch* const pszSubsystem
                            , const tCIDLib::Tch* const pszResource
                            , const tCIDLib::TProcessId pidOfName) :
    __pidOfName(pidOfName)
    , __pszCompany(0)
    , __pszSubsystem(0)
    , __pszResource(0)
{
    __pszCompany = TRawStr::pszReplicate(pszCompany);
    __pszSubsystem = TRawStr::pszReplicate(pszSubsystem);
    __pszResource = TRawStr::pszReplicate(pszResource);
}

TKrnlRscName::TKrnlRscName(const TKrnlRscName& krscnToCopy) :

    __pidOfName(krscnToCopy.__pidOfName)
    , __pszCompany(0)
    , __pszSubsystem(0)
    , __pszResource(0)
{
    __pszCompany = TRawStr::pszReplicate(krscnToCopy.__pszCompany);
    __pszSubsystem = TRawStr::pszReplicate(krscnToCopy.__pszSubsystem);
    __pszResource = TRawStr::pszReplicate(krscnToCopy.__pszResource);
}

TKrnlRscName::~TKrnlRscName()
{
    delete __pszCompany;
    __pszCompany = 0;
    delete __pszSubsystem;
    __pszCompany = 0;
    delete __pszResource;
    __pszResource = 0;
}


// ----------------------------------------------------------------------------
//  TKrnlRscName: Public operators
// ----------------------------------------------------------------------------

TKrnlRscName& TKrnlRscName::operator=(const TKrnlRscName& krscnToAssign)
{
    if (this == &krscnToAssign)
        return *this;

    __pidOfName = krscnToAssign.__pidOfName;

    // Clean up our current strings
    delete __pszCompany;
    __pszCompany = 0;
    delete __pszSubsystem;
    __pszSubsystem = 0;
    delete __pszResource;
    __pszResource = 0;

    // Now replicate the source strings
    __pszCompany = TRawStr::pszReplicate(krscnToAssign.__pszCompany);
    __pszSubsystem = TRawStr::pszReplicate(krscnToAssign.__pszSubsystem);
    __pszResource = TRawStr::pszReplicate(krscnToAssign.__pszResource);

    return *this;
}

tCIDLib::TBoolean
TKrnlRscName::operator==(const TKrnlRscName& krscnToCompare) const
{
    if (this == &krscnToCompare)
        return kCIDLib::True;

    // Compare the PID first since its fast
    if (__pidOfName != krscnToCompare.__pidOfName)
        return kCIDLib::False;

    // Then compare in the order that they are most likely to differe
    if (TRawStr::eCompareStr(__pszResource, krscnToCompare.__pszResource))
        return kCIDLib::False;

    if (TRawStr::eCompareStr(__pszSubsystem, krscnToCompare.__pszSubsystem))
        return kCIDLib::False;

    if (TRawStr::eCompareStr(__pszCompany, krscnToCompare.__pszCompany))
        return kCIDLib::False;

    return kCIDLib::True;
}


// ----------------------------------------------------------------------------
//  TKrnlRscName: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlRscName::BuildFullName(        tCIDLib::Tch* const     pszToFill
                            , const tCIDLib::TCard4         c4MaxChars
                            , const tCIDLib::ENamedRscTypes eType)  const
{
    // If its not valid, then return an empty string
    if (!bValid())
    {
        pszToFill[0] = kCIDLib::chNull;
        return;
    }

    //
    //  Start by formatting in the prefix that we force into all names. Under
    //  OS/2 there is a prefix defined, but Win32 does not have one. So we
    //  insure that there is no clash among resources of different types.
    //
    if (eType == tCIDLib::ENamedRsc_Event)
    {
        TRawStr::CopyStr(pszToFill, kMessages::pszGen_Event, c4MaxChars);
    }
     else if (eType == tCIDLib::ENamedRsc_Memory)
    {
        TRawStr::CopyStr(pszToFill, kMessages::pszGen_Memory, c4MaxChars);
    }
     else if (eType == tCIDLib::ENamedRsc_Mutex)
    {
        TRawStr::CopyStr(pszToFill, kMessages::pszGen_Mutex, c4MaxChars);
    }
     else if (eType == tCIDLib::ENamedRsc_Semaphore)
    {
        TRawStr::CopyStr(pszToFill, kMessages::pszGen_Semaphore, c4MaxChars);
    }
     else
    {
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcBadNamedResourceType);
    }

    TRawStr::CatStr(pszToFill, L".", c4MaxChars);

    if (TRawStr::c4StrLen(pszToFill) == c4MaxChars)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInsufficientBuffer);

    //
    //  If there is a process id to format in, then we need to format it
    //  to a hex value and cat it on.
    //
    if (__pidOfName != kCIDLib::pidInvalid)
    {
        tCIDLib::TZStr64 szTmp;
        TRawStr::FormatVal(__pidOfName, szTmp, 64, tCIDLib::ERadix_Hex);
        TRawStr::CatStr(pszToFill, szTmp, c4MaxChars);
        TRawStr::CatStr(pszToFill, L".", c4MaxChars);
        if (TRawStr::c4StrLen(pszToFill) == c4MaxChars)
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcInsufficientBuffer);
    }

    // Now put on the three name parts
    TRawStr::CatStr(pszToFill, __pszCompany, c4MaxChars);
    TRawStr::CatStr(pszToFill, L".", c4MaxChars);
    if (TRawStr::c4StrLen(pszToFill) == c4MaxChars)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInsufficientBuffer);

    TRawStr::CatStr(pszToFill, __pszSubsystem, c4MaxChars);
    TRawStr::CatStr(pszToFill, L".", c4MaxChars);
    if (TRawStr::c4StrLen(pszToFill) == c4MaxChars)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInsufficientBuffer);

    TRawStr::CatStr(pszToFill, __pszResource, c4MaxChars);
    if (TRawStr::c4StrLen(pszToFill) == c4MaxChars)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInsufficientBuffer);
}


tCIDLib::TVoid
TKrnlRscName::SetName(  const   tCIDLib::Tch* const     pszCompany
                        , const tCIDLib::Tch* const     pszSubsystem
                        , const tCIDLib::Tch* const     pszResource
                        , const tCIDLib::TProcessId     pidToFormat)
{
    // Free the existing strings
    delete __pszCompany;
    __pszCompany = 0;
    delete __pszSubsystem;
    __pszSubsystem = 0;
    delete __pszResource;
    __pszResource = 0;

    // Now replicate the source strings
    __pszCompany = TRawStr::pszReplicate(pszCompany);
    __pszSubsystem = TRawStr::pszReplicate(pszSubsystem);
    __pszResource = TRawStr::pszReplicate(pszResource);
}
