//
//  FILE NAME: CIDKernel_Registry.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/13/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the TKrnlRegistry class, which encapsulates
//  access to the system registry information.
//
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Include our private header
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __c4ConvertAccess
//                       __c4ConvertFlags
//
// DESCRIPTION:
//
//  These methods convert our generic enums for registry access and
//  registry options flags to the platform specific bit patterns.
// ---------------------------------------
//   INPUT: eXXXX are the CIDLib enum flags to convert.
//
//  OUTPUT: None
//
//  RETURN: The platform flags
//
static tCIDLib::TCard4
__c4ConvertAccess(const tCIDLib::ERegAccessModes eAccessMode)
{
    tCIDLib::TCard4 c4Ret = 0;

    if (eAccessMode & tCIDLib::ERegAccess_QueryValue)
        c4Ret |= KEY_QUERY_VALUE;

    if (eAccessMode & tCIDLib::ERegAccess_SetValue)
        c4Ret |= KEY_SET_VALUE;

    if (eAccessMode & tCIDLib::ERegAccess_CreateSub)
        c4Ret |= KEY_CREATE_SUB_KEY;

    if (eAccessMode & tCIDLib::ERegAccess_EnumSubKeys)
        c4Ret |= KEY_ENUMERATE_SUB_KEYS;

    if (eAccessMode & tCIDLib::ERegAccess_Read)
        c4Ret |= KEY_READ;

    if (eAccessMode & tCIDLib::ERegAccess_Notify)
        c4Ret |= KEY_NOTIFY;

    if (eAccessMode & tCIDLib::ERegAccess_CreateLink)
        c4Ret |= KEY_CREATE_LINK;

    if (eAccessMode & tCIDLib::ERegAccess_Write)
        c4Ret |= KEY_WRITE;

    if (eAccessMode & tCIDLib::ERegAccess_All)
        c4Ret |= KEY_ALL_ACCESS;

    return c4Ret;
}

static tCIDLib::TCard4 __c4ConvertFlags(const tCIDLib::ERegFlags eFlags)
{
    if (eFlags == tCIDLib::ERegFlag_NonVolatile)
        return REG_OPTION_NON_VOLATILE;
    else if (eFlags == tCIDLib::ERegFlag_Volatile)
        return REG_OPTION_VOLATILE;

    // This is bad, so return a bad value to cause an error
    return 0xFFFF;
}



// ----------------------------------------------------------------------------
//  CLASS: TKrnlRegIterator
// PREFIX: kregi
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlRegIterator: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlRegIterator::TKrnlRegIterator(const tCIDLib::TRegKeyHandle hkeyToIterate) :

    __c4Current(0)
    , __hrkeyIter(hkeyToIterate)
{
}

TKrnlRegIterator::~TKrnlRegIterator()
{
}

// ----------------------------------------------------------------------------
//  TKrnlRegIterator: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean
TKrnlRegIterator::bFindNext(        tCIDLib::Tch* const pszKeyName
                            , const tCIDLib::TCard4     c4BufChars
                            , tCIDLib::TInt8&           i8LastWrite)
{
    tCIDLib::TCard4 c4Chars = c4BufChars+1;
    FILETIME        UpdateTime;

    // Check for a subkey at the current index
    tCIDLib::TInt4 i4Ret = RegEnumKeyEx
    (
        __hrkeyIter
        , __c4Current
        , pszKeyName
        , &c4Chars
        , 0
        , 0
        , 0
        , &UpdateTime
    );

    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);

    // Convert the time to the integer scheme, if it was desired
    if (&i8LastWrite)
    {
        i8LastWrite = TRawBits::i8From32
        (
            UpdateTime.dwLowDateTime
            , UpdateTime.dwHighDateTime
        );
    }

    // Now bump up the counter
    __c4Current++;

    return kCIDLib::True;
}

tCIDLib::TVoid TKrnlRegIterator::Reset()
{
    __c4Current = 0;
}




// ----------------------------------------------------------------------------
//   CLASS: TKrnlRegistry
//  PREFIX: kreg
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlRegistry: Public, static methods
// ----------------------------------------------------------------------------
const tCIDLib::TRegKeyHandle TKrnlRegistry::hrkeyRoot = HKEY_CLASSES_ROOT;
const tCIDLib::TRegKeyHandle TKrnlRegistry::hrkeyCurrentUser = HKEY_CURRENT_USER;
const tCIDLib::TRegKeyHandle TKrnlRegistry::hrkeyLocalMachine = HKEY_LOCAL_MACHINE;
const tCIDLib::TRegKeyHandle TKrnlRegistry::hrkeyUsers = HKEY_USERS;


// ----------------------------------------------------------------------------
//  TKrnlRegistry: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlRegistry::TKrnlRegistry()
{
}

TKrnlRegistry::~TKrnlRegistry()
{
}


// ----------------------------------------------------------------------------
//  TKrnlRegistry: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlRegistry::CloseKey(const tCIDLib::TRegKeyHandle hrkeyToClose)
{
    tCIDLib::TInt4 i4Ret = RegCloseKey(hrkeyToClose);
    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);
}


tCIDLib::TRegKeyHandle
TKrnlRegistry::hregCreateKey(const  tCIDLib::TRegKeyHandle  hrkeyParent
                            , const tCIDLib::Tch* const     pszSubKey
                            , const tCIDLib::Tch* const     pszClass
                            , const tCIDLib::ERegFlags      eFlags
                            , const tCIDLib::ERegAccessModes eAccessMode
                            ,       tCIDLib::TBoolean&      bCreated)
{
    //
    //  Convert the flags and access modes to the native scheme
    //
    tCIDLib::TCard4 c4Access    = __c4ConvertAccess(eAccessMode);
    tCIDLib::TCard4 c4Flags     = __c4ConvertFlags(eFlags);

    tCIDLib::TCard4         c4Disposition;
    tCIDLib::TRegKeyHandle  hrkeyTmp;
    tCIDLib::TInt4  i4Ret = RegCreateKeyEx
    (
        hrkeyParent
        , pszSubKey
        , 0
        , (tCIDLib::Tch*)pszClass
        , c4Flags
        , c4Access
        , 0
        , &hrkeyTmp
        , &c4Disposition
    );

    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);

    // Set the bCreated flag according to the disposition
    if (&bCreated)
    {
        if (c4Disposition == REG_CREATED_NEW_KEY)
            bCreated = kCIDLib::True;
        else
            bCreated = kCIDLib::False;
    }
    return hrkeyTmp;
}


tCIDLib::TRegKeyHandle
TKrnlRegistry::hregOpenKey( const   tCIDLib::TRegKeyHandle  hrkeyParent
                            , const tCIDLib::Tch* const     pszSubKey
                            , const tCIDLib::ERegAccessModes eAccessMode)
{
    // Convert the flags and access modes to the native scheme
    tCIDLib::TCard4 c4Access = __c4ConvertAccess(eAccessMode);

    tCIDLib::TRegKeyHandle hrkeyTmp;
    tCIDLib::TInt4  i4Ret = RegOpenKeyEx
    (
        hrkeyParent
        , pszSubKey
        , 0
        , c4Access
        , &hrkeyTmp
    );

    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);

    return hrkeyTmp;
}


tCIDLib::TVoid
TKrnlRegistry::DeleteValue( const   tCIDLib::TRegKeyHandle  hrkeyTarget
                            , const tCIDLib::Tch* const     pszValueName)
{
    tCIDLib::TInt4 i4Ret = RegDeleteValue(hrkeyTarget, pszValueName);
    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);
}

tCIDLib::TVoid
TKrnlRegistry::DeleteKey(   const   tCIDLib::TRegKeyHandle  hrkeyToDelete
                            , const tCIDLib::Tch* const     pszSubKeyName)
{
    tCIDLib::TInt4 i4Ret = RegDeleteKey(hrkeyToDelete, pszSubKeyName);
    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);
}


tCIDLib::TVoid
TKrnlRegistry::FlushKey(const tCIDLib::TRegKeyHandle hrkeyToFlush)
{
    tCIDLib::TInt4 i4Ret = RegFlushKey(hrkeyToFlush);
    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);
}


tCIDLib::TVoid
TKrnlRegistry::SetKey(  const   tCIDLib::TRegKeyHandle  hrkeyTarget
                        , const tCIDLib::Tch* const     pszValueName
                        , const tCIDLib::Tch* const     pszData)
{
    tCIDLib::TInt4 i4Ret = RegSetValueEx
    (
        hrkeyTarget
        , pszValueName
        , 0
        , REG_SZ
        , (const tCIDLib::TCard1*)pszData
        , TRawStr::c4StrLen(pszData)+1
    );

    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);
}

tCIDLib::TVoid
TKrnlRegistry::SetKey(  const   tCIDLib::TRegKeyHandle  hrkeyTarget
                        , const tCIDLib::Tch* const     pszValueName
                        , const tCIDLib::TCard4         c4Data)
{
    tCIDLib::TInt4 i4Ret = RegSetValueEx
    (
        hrkeyTarget
        , pszValueName
        , 0
        , REG_DWORD
        , (const tCIDLib::TCard1*)&c4Data
        , sizeof(c4Data)
    );

    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);
}

tCIDLib::TVoid
TKrnlRegistry::SetKey(  const   tCIDLib::TRegKeyHandle  hrkeyTarget
                        , const tCIDLib::Tch* const     pszValueName
                        , const tCIDLib::TVoid* const   pBuffer
                        , const tCIDLib::TCard4         c4DataSz)
{
    tCIDLib::TInt4 i4Ret = RegSetValueEx
    (
        hrkeyTarget
        , pszValueName
        , 0
        , REG_BINARY
        , (const tCIDLib::TCard1*)pBuffer
        , c4DataSz
    );

    if (i4Ret != ERROR_SUCCESS)
        TKrnlError::ThrowKrnlError(i4Ret);
}
