//
//  FILE NAME: CIDKernel_RawString.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/10/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module provides our 'blessed' mechanisms for dealing with raw strings.
//  This is not done much at all outside of the kernel but we make them
//  available to the outside world.
//
//  We greatly deal with the BS of dealing with non UNICode vs short
//  char strings by overloading many of the functions, letting C++ pick the
//  right version. This drastically cuts down on grunt work.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"
#include    <math.h>


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eXlatSortRes
//
// DESCRIPTION:
//
//  Used locally to translate raw comparision result values into our more
//  restrictive ESortComps enum.
// -------------------------------------
//   INPUT: sRes is the result of the OS comparison call.
//
//  OUTPUT: None
//
//  RETURN: The corresponding enum value
//
static tCIDLib::ESortComps __eXlatSortRes(const tCIDLib::TSInt sRes)
{
    if (sRes < 0)
        return tCIDLib::ESort_FirstLess;
    else if (sRes > 0)
        return tCIDLib::ESort_FirstGreater;
    return tCIDLib::ESort_Equal;
}


// ----------------------------------------------------------------------------
//  Public functions
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TRawStr::bIsAlpha(const tCIDLib::Tch chToTest)
{
    if (iswalpha(chToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsAlpha(const tCIDLib::Tsch schToTest)
{
    if (isalpha(schToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsAlphaNum(const tCIDLib::Tch chToTest)
{
    if (iswalnum(chToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsAlphaNum(const tCIDLib::Tsch schToTest)
{
    if (isalnum(schToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsDigit(const tCIDLib::Tch chToTest)
{
    if (iswdigit(chToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsDigit(const tCIDLib::Tsch schToTest)
{
    if (isdigit(schToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsHexDigit(const tCIDLib::Tch chToTest)
{
    if (iswxdigit(chToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsHexDigit(const tCIDLib::Tsch schToTest)
{
    if (isxdigit(schToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsPunct(const tCIDLib::Tch chToTest)
{
    if (iswpunct(chToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsPunct(const tCIDLib::Tsch schToTest)
{
    if (ispunct(schToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsLower(const tCIDLib::Tch chToTest)
{
    if (iswlower(chToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsLower(const tCIDLib::Tsch schToTest)
{
    if (islower(schToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsSpace(const tCIDLib::Tch chToTest)
{
    if (iswspace(chToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsSpace(const tCIDLib::Tsch schToTest)
{
    if (isspace(schToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsUpper(const tCIDLib::Tch chToTest)
{
    if (iswupper(chToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TRawStr::bIsUpper(const tCIDLib::Tsch schToTest)
{
    if (isupper(schToTest))
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TCard4 TRawStr::c4StrLen(const tCIDLib::Tch* const pszSrc)
{
    return lstrlen(pszSrc);
}

tCIDLib::TCard4 TRawStr::c4StrLen(const tCIDLib::Tsch* const pszSrc)
{
    return strlen(pszSrc);
}


tCIDLib::Tch TRawStr::chConvert(const tCIDLib::Tsch schToConvert)
{
    tCIDLib::Tch chRet;
    mbtowc(&chRet, &schToConvert, 1);
    return chRet;
}

tCIDLib::Tsch TRawStr::schConvert(const tCIDLib::Tch chToConvert)
{
    tCIDLib::Tsch schRet;
    wctomb(&schRet, chToConvert);
    return schRet;
}


tCIDLib::Tch TRawStr::chLower(const tCIDLib::Tch chToLower)
{
    return towlower(chToLower);
}

tCIDLib::Tsch TRawStr::chLower(const tCIDLib::Tsch chToLower)
{
    return tolower(chToLower);
}

tCIDLib::Tch TRawStr::chUpper(const tCIDLib::Tch chToUpper)
{
    return towupper(chToUpper);
}

tCIDLib::Tsch TRawStr::chUpper(const tCIDLib::Tsch chToUpper)
{
    return toupper(chToUpper);
}


tCIDLib::TVoid
TRawStr::CatStr(        tCIDLib::Tch* const     pszTarget
                , const tCIDLib::Tch* const     pszSrc
                , const tCIDLib::TCard4         c4MaxChars)
{
    if (!pszSrc)
        return;

    // Get the current length of the target
    tCIDLib::TCard4 c4TargetLen = lstrlen(pszTarget);

    // If the string is already full, then nothing to do
    if (c4TargetLen >= c4MaxChars)
        return;

    // And cat the source string
    wcsncat(pszTarget, pszSrc, c4MaxChars - c4TargetLen);
    pszTarget[c4MaxChars] = kCIDLib::chNull;
}

tCIDLib::TVoid
TRawStr::CatStr(        tCIDLib::Tch* const     pszTarget
                , const tCIDLib::Tsch* const    pszSrc
                , const tCIDLib::TCard4         c4MaxChars)
{
    if (!pszSrc)
        return;

    //
    //  Convert the source into a temp buffer that we can cat to the
    //  target. Put a raw janitor on it to make sure its cleaned up.
    //
    tCIDLib::Tch* pszTmp = 0;
    pszTmp = pszConvert(pszSrc);
    THeapJanitor janTmp(pszTmp);

    // And now call our other version
    CatStr(pszTarget, pszTmp, c4MaxChars);
}


tCIDLib::TVoid
TRawStr::CopyCatStr(        tCIDLib::Tch* const     pszTarget
                    , const tCIDLib::TCard4         c4MaxChars
                    , const tCIDLib::Tch* const     pszCopy
                    , const tCIDLib::Tch* const     pszCat)
{
    // Empty the target regardless of what happens
    pszTarget[0] = kCIDLib::chNull;

    //
    //  Do a very efficient asm loop to do the copy and cat with the
    //  minimum overhead.
    //
    tCIDLib::TCard4     c4Index = 0;
    tCIDLib::Tch*       pszOut = pszTarget;
    const tCIDLib::Tch* pszIn;

    if (pszCopy)
    {
        // Copy over the copy string
        pszIn = pszCopy;
        while (c4Index < c4MaxChars)
        {
            if (!*pszIn)
                break;

            *pszOut = *pszIn;
            pszOut++;
            pszIn++;
            c4Index++;
        }
    }

    // And now cat over the cat string, if not already full
    if (pszCat)
    {
        pszIn = pszCat;
        while (c4Index < c4MaxChars)
        {
            if (!*pszIn)
                break;

            *pszOut = *pszIn;
            pszOut++;
            pszIn++;
            c4Index++;
        }
    }

    // Cap it off
    pszTarget[c4Index] = kCIDLib::chNull;
}

tCIDLib::TVoid
TRawStr::CopyCatStr(        tCIDLib::Tch* const     pszTarget
                    , const tCIDLib::TCard4         c4MaxChars
                    , const tCIDLib::Tch* const     pszCopy
                    , const tCIDLib::Tch* const     pszCat1
                    , const tCIDLib::Tch* const     pszCat2)
{
    // Empty the target regardless of what happens
    pszTarget[0] = kCIDLib::chNull;

    //
    //  Do a very efficient asm loop to do the copy and cat with the
    //  minimum overhead.
    //
    tCIDLib::TCard4     c4Index = 0;
    tCIDLib::Tch*       pszOut = pszTarget;
    const tCIDLib::Tch* pszIn;

    if (pszCopy)
    {
        // Copy over the copy string
        pszIn = pszCopy;
        while (c4Index < c4MaxChars)
        {
            if (!*pszIn)
                break;

            *pszOut = *pszIn;
            pszOut++;
            pszIn++;
            c4Index++;
        }
    }

    // And now cat over the first cat string, if not already full
    if (pszCat1)
    {
        pszIn = pszCat1;
        while (c4Index < c4MaxChars)
        {
            if (!*pszIn)
                break;

            *pszOut = *pszIn;
            pszOut++;
            pszIn++;
            c4Index++;
        }
    }

    // And now cat over the second cat string, if not already full
    if (pszCat2)
    {
        pszIn = pszCat2;
        while (c4Index < c4MaxChars)
        {
            if (!*pszIn)
                break;

            *pszOut = *pszIn;
            pszOut++;
            pszIn++;
            c4Index++;
        }
    }

    // Cap it off
    pszTarget[c4Index] = kCIDLib::chNull;
}

tCIDLib::TVoid
TRawStr::CopyCatStr(        tCIDLib::Tch* const     pszTarget
                    , const tCIDLib::TCard4         c4MaxChars
                    , const tCIDLib::Tch* const     pszCopy
                    , const tCIDLib::Tch* const     pszCat1
                    , const tCIDLib::Tch* const     pszCat2
                    , const tCIDLib::Tch* const     pszCat3)
{
    // Empty the target regardless of what happens
    pszTarget[0] = kCIDLib::chNull;

    //
    //  Do a very efficient asm loop to do the copy and cat with the
    //  minimum overhead.
    //
    tCIDLib::TCard4     c4Index = 0;
    tCIDLib::Tch*       pszOut = pszTarget;
    const tCIDLib::Tch* pszIn;

    if (pszCopy)
    {
        // Copy over the copy string
        pszIn = pszCopy;
        while (c4Index < c4MaxChars)
        {
            *pszOut = *pszIn;
            if (!*pszOut)
                break;

            pszOut++;
            pszIn++;
            c4Index++;
        }
    }

    // And now cat over the first cat string, if not already full
    if (pszCat1)
    {
        pszIn = pszCat1;
        while (c4Index < c4MaxChars)
        {
            *pszOut = *pszIn;
            if (!*pszOut)
                break;

            pszOut++;
            pszIn++;
            c4Index++;
        }
    }

    // And now cat over the second cat string, if not already full
    if (pszCat2)
    {
        pszIn = pszCat2;
        while (c4Index < c4MaxChars)
        {
            *pszOut = *pszIn;
            if (!*pszOut)
                break;

            pszOut++;
            pszIn++;
            c4Index++;
        }
    }

    // And now cat over the third cat string, if not already full
    if (pszCat3)
    {
        pszIn = pszCat3;
        while (c4Index < c4MaxChars)
        {
            *pszOut = *pszIn;
            if (!*pszOut)
                break;

            pszOut++;
            pszIn++;
            c4Index++;
        }
    }

    // Cap it off
    pszTarget[c4Index] = kCIDLib::chNull;
}


tCIDLib::TVoid
TRawStr::CopyStr(           tCIDLib::Tch* const pszTarget
                    , const tCIDLib::Tch* const pszSrc
                    , const tCIDLib::TCard4     c4MaxChars)
{
    // If a nul source, then obviously nothing to do
    if (!pszSrc)
        return;

    // Get the source length. If zero, then nothing to do
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszSrc);
    if (!c4Len)
    {
        pszTarget[0] = kCIDLib::chNull;
        return;
    }

    //
    //  Get the actual length that we can copy, which is the shorter of
    //  the source length and the max chars.
    //
    const tCIDLib::TCard4 c4Actual = c4Len > c4MaxChars ? c4MaxChars : c4Len;

    //
    //  See if the buffers overlap. If so, then we have to take a
    //  different approach. Otherwise, we can just do a regular
    //  bounded copy.
    //
    if ((pszSrc > pszTarget) && (pszSrc <= pszTarget+c4MaxChars))
        memmove(pszTarget, pszSrc, c4Actual * kCIDLib::c4CharBytes);
     else
        memcpy(pszTarget, pszSrc,  c4Actual * kCIDLib::c4CharBytes);

    // Cap it off at the actual bytes copied
    pszTarget[c4Actual] = kCIDLib::chNull;
}

tCIDLib::TVoid
TRawStr::CopyStr(       tCIDLib::Tch* const     pszTarget
                , const tCIDLib::Tsch* const    pszSrc
                , const tCIDLib::TCard4         c4MaxChars)
{
    if (!pszSrc)
        return;

    mbstowcs(pszTarget, pszSrc, c4MaxChars);
    pszTarget[c4MaxChars] = kCIDLib::chNull;
}


tCIDLib::ESortComps
TRawStr::eCompareStr(   const   tCIDLib::Tch* const pszStr1
                        , const tCIDLib::Tch* const pszStr2)
{
    return __eXlatSortRes(wcscoll(pszStr1, pszStr2));
}

tCIDLib::ESortComps
TRawStr::eCompareStr(   const   tCIDLib::Tsch* const    pszStr1
                        , const tCIDLib::Tsch* const    pszStr2)
{
    return __eXlatSortRes(strcoll(pszStr1, pszStr2));
}

tCIDLib::ESortComps
TRawStr::eNCompareStr(  const   tCIDLib::Tch* const pszStr1
                        , const tCIDLib::Tch* const pszStr2
                        , const tCIDLib::TCard4     c4Count)
{
    return __eXlatSortRes(_wcsncoll(pszStr1, pszStr2, c4Count));
}

tCIDLib::ESortComps
TRawStr::eNCompareStr(  const   tCIDLib::Tsch* const pszStr1
                        , const tCIDLib::Tsch* const pszStr2
                        , const tCIDLib::TCard4      c4Count)
{
    return __eXlatSortRes(_strncoll(pszStr1, pszStr2, c4Count));
}

tCIDLib::ESortComps
TRawStr::eICompareStr(  const  tCIDLib::Tch* const pszStr1
                        , const tCIDLib::Tch* const pszStr2)
{
    return __eXlatSortRes(_wcsicoll(pszStr1, pszStr2));

}

tCIDLib::ESortComps
TRawStr::eICompareStr(  const  tCIDLib::Tsch* const    pszStr1
                        , const tCIDLib::Tsch* const    pszStr2)
{
    return __eXlatSortRes(_stricoll(pszStr1, pszStr2));
}

tCIDLib::ESortComps
TRawStr::eNICompareStr( const   tCIDLib::Tch* const pszStr1
                        , const tCIDLib::Tch* const pszStr2
                        , const tCIDLib::TCard4     c4Count)
{
    return __eXlatSortRes(_wcsnicoll(pszStr1, pszStr2, c4Count));

}

tCIDLib::ESortComps
TRawStr::eNICompareStr( const   tCIDLib::Tsch* const pszStr1
                        , const tCIDLib::Tsch* const pszStr2
                        , const tCIDLib::TCard4     c4Count)
{
    return __eXlatSortRes(_strnicoll(pszStr1, pszStr2, c4Count));
}


tCIDLib::TVoid TRawStr::FillString(          tCIDLib::Tsch* const   pszBuf
                                    , const tCIDLib::Tsch           chFill
                                    , const tCIDLib::TCard4         c4Count)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        mov         edi, pszBuf
        mov         ecx, c4Count
        mov         al, chFill
        rep stosb

        xor         al, al
        mov         [edi], al
    }
    #else
    memset(pszBuf, chFill, c4Count);
    pszBuf[c4Count] = 0;
    #endif
}

tCIDLib::TVoid
TRawStr::FillString(        tCIDLib::Tch* const pszBuf
                    , const tCIDLib::Tch        chFill
                    , const tCIDLib::TCard4     c4Count)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        mov         edi, pszBuf
        mov         ecx, c4Count
        mov         ax, chFill
        rep stosw

        xor         ax, ax
        mov         [edi], ax
    }
    #else

    //
    //  Optimize if the high and low bytes are the same, since we can then
    //  just do a memset.
    //
    if (((chFill & 0xFF00) >> 8) == (chFill & 0x00FF))
    {
        memset(pszBuf, tCIDLib::TCard1(chFill & 0x00FF), c4Count * 2);
    }
     else
    {
        tCIDLib::Tch* pchBuf = pszBuf;
        for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
        {
            *pchBuf = chFill;
            pchBuf++;
        }
    }
    pszBuf[c4Count] = 0;
    #endif
}


tCIDLib::TVoid
TRawStr::FormatStr( const   tCIDLib::Tch* const pszVal
                    ,       tCIDLib::Tch* const pszBuf
                    , const tCIDLib::TCard4     c4FldWidth
                    , const tCIDLib::Tch        chFill
                    , const tCIDLib::EHJustify  eJustify)
{
    //
    //  If no field width is indicated, then just copy as much of the
    //  source string as possible.
    //
    if (c4FldWidth == 0)
    {
        CopyStr(pszBuf, pszVal, c4FldWidth);
        return;
    }

    // Remember the current length of the value string and the buffer
    tCIDLib::TCard4  c4ValLen = c4StrLen(pszVal);
    tCIDLib::TCard4  c4Tmp;

    //
    //  Regardless of what route we take below, we want to fill the
    //  destination buffer with the fill character and cap if off at
    //  the field width, so go ahead and do that.
    //
    FillString(pszBuf, chFill, c4FldWidth);
    pszBuf[c4FldWidth] = 0;

    //
    //  If there is no text in the source buffer, then we have nothing to
    //  do. Otherwise, we need to do the justification thing.
    //
    if (pszVal[0])
    {
        if (eJustify == tCIDLib::EHJustify_Left)
        {
            //
            //  Copy the value into the field, minus the null terminator.
            //  If the value string is longer than the field, then clip
            //  it.
            //
            c4Tmp = (c4ValLen > c4FldWidth) ? c4FldWidth : c4ValLen;
            CopyMemory(pszBuf, pszVal, c4Tmp * kCIDLib::c4CharBytes);
        }
         else if (eJustify == tCIDLib::EHJustify_Right)
        {
            //
            //  Copy the value into the field so that it ends at the end
            //  of the field. If the value string is longer than the
            //  field, then we have to start copying past the 0th char.
            //  Make sure that the terminating null on the tmp buffer
            //  falls onto the null of the result buffer.
            //
            if (c4ValLen > c4FldWidth)
                CopyStr(pszBuf, &pszVal[c4ValLen-c4FldWidth], c4FldWidth);
             else
                CopyStr(&pszBuf[c4FldWidth-c4ValLen], pszVal, c4ValLen);
        }
         else if (eJustify == tCIDLib::EHJustify_Center)
        {
            //
            //  Copy the value into the field so that it is centered.
            //  If an even number of chars are not available on both
            //  sides of the centered field, then it tends toward the
            //  left since the integer divide truncates and loses the
            //  partial remainder. If the value string is longer than
            //  the field width, then it is clipped on both ends so
            //  that the center portion of it gets into the result
            //  buffer.
            //
            if (c4ValLen > c4FldWidth)
            {
                CopyMemory
                (
                    pszBuf
                    , &pszVal[(c4ValLen-c4FldWidth) >> 1]
                    , ((c4ValLen-c4FldWidth) >> 1) * kCIDLib::c4CharBytes
                );
            }
             else
            {
                CopyMemory
                (
                    &pszBuf[(c4FldWidth >> 1) - (c4ValLen >> 1)]
                    , pszVal
                    , c4StrLen(pszVal) * kCIDLib::c4CharBytes
                );
            }
        }
         else
        {
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcBadJustification);
        }
    }
}


tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TCard1     c1Val
                    ,       tCIDLib::Tch* const pszBuf
                    , const tCIDLib::TCard4     c4MaxChars
                    , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::TZStr64 szTmp;
    _ultow(c1Val, szTmp, eRadix);
    CopyStr(pszBuf, szTmp, c4MaxChars);
    if (eRadix == tCIDLib::ERadix_Hex)
        _wcsupr(pszBuf);
}

tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TCard2     c2Val
                    ,       tCIDLib::Tch* const pszBuf
                    , const tCIDLib::TCard4     c4MaxChars
                    , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::TZStr64 szTmp;
    _ultow(c2Val, szTmp, eRadix);
    CopyStr(pszBuf, szTmp, c4MaxChars);
    if (eRadix == tCIDLib::ERadix_Hex)
        _wcsupr(pszBuf);
}

tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TCard4     c4Val
                    ,       tCIDLib::Tch* const pszBuf
                    , const tCIDLib::TCard4     c4MaxChars
                    , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::TZStr128 szTmp;
    _ultow(c4Val, szTmp, eRadix);
    CopyStr(pszBuf, szTmp, c4MaxChars);
    if (eRadix == tCIDLib::ERadix_Hex)
        _wcsupr(pszBuf);
}

tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TInt1      i1Val
                    ,       tCIDLib::Tch* const pszBuf
                    , const tCIDLib::TCard4     c4MaxChars
                    , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::TZStr128 szTmp;
    _ltow(i1Val, szTmp, eRadix);
    CopyStr(pszBuf, szTmp, c4MaxChars);
    if (eRadix == tCIDLib::ERadix_Hex)
        _wcsupr(pszBuf);
}

tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TInt2      i2Val
                    ,       tCIDLib::Tch* const pszBuf
                    , const tCIDLib::TCard4     c4MaxChars
                    , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::TZStr64 szTmp;
    _ltow(i2Val, szTmp, eRadix);
    CopyStr(pszBuf, szTmp, c4MaxChars);
    if (eRadix == tCIDLib::ERadix_Hex)
        _wcsupr(pszBuf);
}

tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TInt4      i4Val
                    ,       tCIDLib::Tch* const pszBuf
                    , const tCIDLib::TCard4     c4MaxChars
                    , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::TZStr128 szTmp;
    _ltow(i4Val, szTmp, eRadix);
    CopyStr(pszBuf, szTmp, c4MaxChars);
    if (eRadix == tCIDLib::ERadix_Hex)
        _wcsupr(pszBuf);
}

tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TUInt      uVal
                    ,       tCIDLib::Tch* const pszBuf
                    , const tCIDLib::TCard4     c4MaxChars
                    , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::TZStr64 szTmp;
    _ultow(uVal, szTmp, eRadix);
    CopyStr(pszBuf, szTmp, c4MaxChars);
    if (eRadix == tCIDLib::ERadix_Hex)
        _wcsupr(pszBuf);
}

tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TInt8&     i8Val
                    ,       tCIDLib::Tch* const pszBuf
                    , const tCIDLib::TCard4     c4MaxChars
                    , const tCIDLib::ERadices   eRadix)
{
    // <TBD> eRadix is not used yet
    eRadix;

    FormatVal
    (
        tCIDLib::TFloat8(i8Val)
        , pszBuf
        , 0
        , c4MaxChars
        , tCIDLib::ETrail_Ignore
    );
}

tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TFloat4&       f4Val
                    ,       tCIDLib::Tch* const     pszBuf
                    , const tCIDLib::TCard4         c4Precision
                    , const tCIDLib::TCard4         c4MaxChars
                    , const tCIDLib::ETrailFormats  eTrail)
{
    tCIDLib::TFloat8 f8Tmp = f4Val;
    FormatVal(f8Tmp, pszBuf, c4Precision, c4MaxChars, eTrail);
}

tCIDLib::TVoid
TRawStr::FormatVal( const   tCIDLib::TFloat8&       f8Val
                    ,       tCIDLib::Tch* const     pszBuf
                    , const tCIDLib::TCard4         c4Precision
                    , const tCIDLib::TCard4         c4MaxChars
                    , const tCIDLib::ETrailFormats  eTrail)
{
    // Init the string to empty just in case
    pszBuf[0] = 0;

    // Check the psycho scenario
    if (!c4MaxChars)
        return;

    // And create some locals for temp formatting
    tCIDLib::Tch    chFill;
    tCIDLib::TCard4 c4Len, c4Ind;
    tCIDLib::TSInt  iSign, iDec;
    tCIDLib::Tch*   pszCur;
    tCIDLib::Tch    szTmp[320];
    tCIDLib::Tch    szFmt[320];

    // Set up the trailing character
    if (eTrail == tCIDLib::ETrail_Spaces)
        chFill = kCIDLib::chSpace;
    else if (eTrail == tCIDLib::ETrail_Zeroes)
        chFill = kCIDLib::chDigit0;
    else if (eTrail == tCIDLib::ETrail_Ignore)
        chFill = kCIDLib::chQuestionMark;

    // Query the locale information
    TKrnlLocale::TLocaleInfo lociFmt;
    TKrnlLocale::QueryLocaleInfo(lociFmt);

    //
    //  Handle the special case of 0 up front since its easy. We just put
    //  in a 0, then enough fill chars according to the fill style.
    //
    if (!f8Val)
    {
        tCIDLib::TCard4 c4Ind = 0;

        pszBuf[c4Ind++] = kCIDLib::chDigit0;

        if (c4MaxChars == c4Ind)
        {
            pszBuf[c4Ind++] = 0;
            return;
        }

        if (c4Precision)
        {
            pszBuf[c4Ind++] = lociFmt.chDecimalSym;
            if (c4MaxChars == c4Ind)
            {
                pszBuf[c4Ind] = 0;
                return;
            }

            for (tCIDLib::TCard4 c4Ind2 = 0; c4Ind2 < c4Precision; c4Ind2++)
            {
                if (c4Ind == c4MaxChars)
                    break;
                pszBuf[c4Ind++] = kCIDLib::chDigit0;
            }
        }
        pszBuf[c4Ind] = 0;
        return;
    }

    //
    //  Convert the value into a raw string, using 1 more decimal of
    //  precision than the user wants so as to avoid rounding error. This
    //  string has no sign or decimal place and only contains significant
    //  trailing digits. We also then convert it to a UNICode string.
    //
    pszConvert(_fcvt(f8Val, c4Precision+1, &iDec, &iSign), szTmp, 320);

    // Get the length of the string
    c4Len = c4StrLen(szTmp);

    //
    //  0 or negative decimal places all mean the same thing, so normalize
    //  it back to 0 in order to make the logic below a lot simpler.
    //
    if (iDec < 0)
        iDec = 0;

    //
    //  Work backwards from the end and strip off any trailing zeros because
    //  that makes it easier below (though we may in fact put them right
    //  back if the trailing format requested is zeros.) Don't go past the
    //  decimal position though.
    //
    while (szTmp[c4Len-1] == kCIDLib::chDigit0)
    {
        if (c4Len == tCIDLib::TCard4(iDec))
            break;

        c4Len--;
        if (!c4Len)
        {
            pszBuf[0] = kCIDLib::chNull;
            return;
        }
    }
    szTmp[c4Len] = 0;

    if ((eTrail == tCIDLib::ETrail_Ignore) && c4Len)
    {
        //
        //  Loop down from the last char until we hit the decimal position
        //  or a signficant digit.
        //
        for (tCIDLib::TInt4 i4Ind = c4Len-1; i4Ind >= 0, i4Ind >= iDec; i4Ind--)
        {
            if (szTmp[i4Ind] != kCIDLib::chDigit0)
                break;
        }
        szTmp[i4Ind+1] = 0;
    }

    // Use a pointer to run up through the string
    pszCur = szFmt;

    // If signed, then add the sign first
    if (iSign)
    {
        *pszCur = lociFmt.chNegSign;
        pszCur++;
    }

    //
    //  Handle a special case here. If the string is empty, then there were
    //  no significant digits, so we can build the string ourself. Otherwise
    //  we do the normal thang.
    //
    tCIDLib::TCard4 c4DecCnt = 0;
    if (szTmp[0])
    {
        c4Ind = 0;
        if (!iDec)
        {
            *pszCur = kCIDLib::chDigit0;
            pszCur++;
        }
         else
        {
            for (; c4Ind < tCIDLib::TCard4(iDec); c4Ind++)
            {
                *pszCur = szTmp[c4Ind];
                pszCur++;
            }
        }

        //
        //  If there are any precision digits were requested, then we need
        //  to deal with them. We copy out as many as are available from
        //  the raw formatted string, discarding trailing zeros. Then
        //  we do any trailing fill that was requested.
        //
        if (c4Precision)
        {
            *pszCur = lociFmt.chDecimalSym;
            pszCur++;

            while (szTmp[c4Ind])
            {
                *pszCur = szTmp[c4Ind++];

                c4DecCnt++;
                pszCur++;

                if (c4DecCnt >= c4Precision)
                    break;
            }

            if ((c4DecCnt < c4Precision)
            &&  (eTrail != tCIDLib::ETrail_Ignore))
            {
                while (c4DecCnt < c4Precision)
                {
                    *pszCur = chFill;
                    pszCur++;
                    c4DecCnt++;
                }
            }
        }
    }
     else
    {
        *pszCur++ = kCIDLib::chDigit0;

        if (c4Precision && (eTrail != tCIDLib::ETrail_Ignore))
        {
            *pszCur++ = lociFmt.chDecimalSym;
            for (c4DecCnt = 0; c4DecCnt < c4Precision; c4DecCnt++)
                *pszCur++ = chFill;
        }
    }

    // If nothing happened, then we are done
    if (pszCur == szFmt)
        return;

    // If the last char is the decimal, then waste it
    if (*(pszCur-1) == lociFmt.chDecimalSym)
        pszCur--;

    // Cap it off
    *pszCur = 0;

    // Now copy back as much as we can to the caller's buffer
    TRawStr::CopyStr(pszBuf, szFmt, c4MaxChars);
}


#pragma warning(disable : 4035)
tCIDLib::THashVal
TRawStr::hshHashStr(const   tCIDLib::Tch* const pszStr
                    , const tCIDLib::TCard4     c4Modulus)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        sub     eax, eax            // Clear out return
        mov     esi, pszStr         // Load up the string address
        xor     edx, edx            // Use edx as accumulator, so zero

    LoopStart:                      // And enter the loop
        lodsw                       // Load up a UNICode char from str
        or      ax, ax              // Check for nul
        jz      HashDone            // And exit if so

        rol     edx, 1              // Rotate accum left 1 bit
        xor     dx, ax              // XOR w/ low word of accum
        jmp     LoopStart           // And do the next word

    HashDone:
        mov     eax, edx            // Get accum to eax
        sub     edx, edx            // Clear edx for divide
        div     c4Modulus           // Divide by modulus
        mov     eax, edx            // Keep remainder
    }
    #else

    tCIDLib::THashVal   hshRet = 0;
    const tCIDLib::Tch* pszBuf = pszStr;
    while (*pszBuf)
    {
        hshRet <<= 1;
        hshRet ^= tCIDLib::THashVal(*pszBuf);
        pszBuf++;
    }

    // Divide by the modulus for return
    return tCIDLib::THashVal(hshRet % c4Modulus);

    #endif
}
#pragma warning(default : 4035)


tCIDLib::TCard4
TRawStr::c4AsBinary(const   tCIDLib::Tch* const pszToConvert
                    , const tCIDLib::ERadices   eRadix
                    ,       tCIDLib::TBoolean&  bValid)
{
    tCIDLib::TCard4 c4Ret = wcstoul(pszToConvert, 0, eRadix);
    if (!c4Ret && (errno == ERANGE))
    {
        bValid = kCIDLib::False;
        return 0;
    }
    bValid = kCIDLib::True;
    return c4Ret;
}

tCIDLib::TFloat8
TRawStr::f8AsBinary(const   tCIDLib::Tch* const pszToConvert
                    ,       tCIDLib::TBoolean&  bValid)
{
    // Try to convert it
    tCIDLib::TFloat8 f8Ret = wcstod(pszToConvert, 0);

    if ((f8Ret == -HUGE_VAL) || (f8Ret == HUGE_VAL))
    {
        bValid = kCIDLib::False;
        return 0.0;
    }
    bValid = kCIDLib::True;
    return f8Ret;
}

tCIDLib::TInt4
TRawStr::i4AsBinary(const   tCIDLib::Tch* const pszToConvert
                    , const tCIDLib::ERadices   eRadix
                    ,       tCIDLib::TBoolean&  bValid)
{
    tCIDLib::TInt4 i4Ret = wcstol(pszToConvert, 0, eRadix);

    if (!i4Ret && (errno == ERANGE))
    {
        bValid = kCIDLib::False;
        return 0;
    }
    bValid = kCIDLib::True;
    return i4Ret;
}

tCIDLib::TInt8
TRawStr::i8AsBinary(const   tCIDLib::Tch* const pszToConvert
                    , const tCIDLib::ERadices   eRadix
                    ,       tCIDLib::TBoolean&  bValid)
{
    tCIDLib::TFloat8 f8Ret = wcstod(pszToConvert, 0);

    if (!f8Ret
    &&  (errno == ERANGE)
    ||  ((f8Ret < kCIDLib::i8MinInt) || (f8Ret > kCIDLib::i8MaxInt)))
    {
        bValid = kCIDLib::False;
        return 0;
    }
    bValid = kCIDLib::True;
    return tCIDLib::TInt8(f8Ret);
}


tCIDLib::Tch* TRawStr::pszReplicate(const tCIDLib::Tch* const pszToReplicate)
{
    tCIDLib::Tch* pszNew = new tCIDLib::Tch[c4StrLen(pszToReplicate)+1];
    wcscpy(pszNew, pszToReplicate);
    return pszNew;
}


tCIDLib::Tch*
TRawStr::pszConvert(const   tCIDLib::Tsch* const    pszToConvert
                    ,       tCIDLib::Tch* const     pszTarget
                    , const tCIDLib::TCard4         c4MaxChars)
{
    mbstowcs(pszTarget, pszToConvert, c4MaxChars);
    pszTarget[c4MaxChars] = 0;
    return pszTarget;
}

tCIDLib::Tch*
TRawStr::pszConvert(const tCIDLib::Tsch* const pszToConvert)
{
    tCIDLib::Tch* pszNew = 0;

    // Allocate a buffer big enoug and put a janitor on it just in case
    tCIDLib::TCard4 c4Len = c4StrLen(pszToConvert);
    pszNew = new tCIDLib::Tch[c4Len+1];
    THeapJanitor janNew(pszNew);

    // Convert into the new string and cap it off
    mbstowcs(pszNew, pszToConvert, c4Len);
    pszNew[c4Len] = 0;

    // Release the janitor and return the new buffer
    janNew.Orphan();
    return pszNew;
}

tCIDLib::Tsch*
TRawStr::pszConvert(const   tCIDLib::Tch* const     pszToConvert
                    ,       tCIDLib::Tsch* const    pszTarget
                    , const tCIDLib::TCard4         c4MaxChars)
{
    wcstombs(pszTarget, pszToConvert, c4MaxChars);
    pszTarget[c4MaxChars] = 0;
    return pszTarget;
}

tCIDLib::Tsch*
TRawStr::pszConvert(const tCIDLib::Tch* const pszToConvert)
{
    tCIDLib::Tsch* pszNew = 0;

    // Allocate a buffer big enoug and put a janitor on it just in case
    tCIDLib::TCard4 c4Len = c4StrLen(pszToConvert);
    pszNew = new tCIDLib::Tsch[c4Len+1];
    THeapJanitor janNew(pszNew);

    // Convert into the new string and cap it off
    wcstombs(pszNew, pszToConvert, c4Len);
    pszNew[c4Len] = 0;

    // Release the janitor and return the new buffer
    janNew.Orphan();
    return pszNew;
}


const tCIDLib::Tch*
TRawStr::pszFindChar(   const   tCIDLib::Tch* const pszToSearch
                        , const tCIDLib::Tch        chToFind)
{
    return wcschr(pszToSearch, chToFind);
}

tCIDLib::Tch* TRawStr::pszFindChar(         tCIDLib::Tch* const pszToSearch
                                    , const tCIDLib::Tch        chToFind)
{
    return wcschr(pszToSearch, chToFind);
}

const tCIDLib::Tch*
TRawStr::pszFindChars(  const   tCIDLib::Tch* const pszToSearch
                        , const tCIDLib::Tch* const pszChars)
{
    return wcspbrk(pszToSearch, pszChars);
}

tCIDLib::Tch*
TRawStr::pszFindChars(          tCIDLib::Tch* const pszToSearch
                        , const tCIDLib::Tch* const pszChars)
{
    return wcspbrk(pszToSearch, pszChars);
}

const tCIDLib::Tch*
TRawStr::pszFindLastChar(   const   tCIDLib::Tch* const pszToSearch
                            , const tCIDLib::Tch        chToFind)
{
    return wcsrchr(pszToSearch, chToFind);
}

tCIDLib::Tch*
TRawStr::pszFindLastChar(           tCIDLib::Tch* const pszToSearch
                            , const tCIDLib::Tch        chToFind)
{
    return wcsrchr(pszToSearch, chToFind);
}

const tCIDLib::Tch*
TRawStr::pszFindSubStr( const   tCIDLib::Tch* const pszToSearch
                        , const tCIDLib::Tch* const pszSubStr)
{
    return wcsstr(pszToSearch, pszSubStr);
}

tCIDLib::Tch*
TRawStr::pszLoadResString(  const   tCIDLib::TModHandle hmodSource
                            , const tCIDLib::TResId     ridToLoad)
{
    //
    //  Try to find the resource first. We convert the string id into a
    //  string table resource id.
    //
    tCIDLib::TRscHandle hRsc = FindResource
    (
        hmodSource
        , MAKEINTRESOURCE((ridToLoad / 16UL)+1)
        , MAKEINTRESOURCE(RT_STRING)
    );

    if (!hRsc)
        return 0;

    // Try to load that puppy
    tCIDLib::THandle hTmp = LoadResource(hmodSource, hRsc);
    if (!hTmp)
        return 0;

    //
    //  Now we need to parse out the correct string. Basically the format
    //  is a length value, followed by the text, then a length byte,
    //  followed by the text, etc.... Up to 16 times for a full string
    //  table.
    //
    tCIDLib::TCard2* pSize = (tCIDLib::TCard2*)hTmp;

    // Find out which of the strings in this table we want
    tCIDLib::TCard4 c4Tmp = ridToLoad;
    c4Tmp %= 16;

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4Tmp; c4Ind++)
    {
        tCIDLib::TCard2 c2Skip = *pSize;
        pSize += c2Skip + 1;
    }

    //
    //  Return the length. We left the pointer at the length word of the
    //  string of interest, so we just have to dereference it.
    //
    tCIDLib::TCard4 c4Size = (tCIDLib::TCard4)*pSize;

    //
    //  Allocate the buffer and load up the resource into it.
    //
    tCIDLib::Tch* pszRet = new tCIDLib::Tch[c4Size+1];

    if (!::LoadStringW(hmodSource, ridToLoad, pszRet, c4Size+1))
    {
        delete pszRet;
        return 0;
    }

    return pszRet;
}


tCIDLib::Tch* TRawStr::pszLowerCase(tCIDLib::Tch* pszToLower)
{
    _wcslwr(pszToLower);
    return pszToLower;
}

tCIDLib::Tsch* TRawStr::pszLowerCase(tCIDLib::Tsch* pszToLower)
{
    strlwr(pszToLower);
    return pszToLower;
}

tCIDLib::Tch* TRawStr::pszUpperCase(tCIDLib::Tch* pszToUpper)
{
    _wcsupr(pszToUpper);
    return pszToUpper;
}

tCIDLib::Tsch* TRawStr::pszUpperCase(tCIDLib::Tsch* pszToUpper)
{
    strupr(pszToUpper);
    return pszToUpper;
}


tCIDLib::Tch*
TRawStr::pszStrTokenize(        tCIDLib::Tch* const pszSource
                        , const tCIDLib::Tch* const pszWhitespace)
{
    return wcstok(pszSource, pszWhitespace);
}


tCIDLib::TVoid
TRawStr::StripStr(          tCIDLib::Tch* const     pszStripBuf
                    , const tCIDLib::Tch* const     pszStripChars
                    , const tCIDLib::EStripModes    eMode)
{
    // If the string is already empty, then do nothing
    if (!(*pszStripBuf) || !(*pszStripChars))
        return;

    // If no stripping options set if flags then do nothing
    if (eMode == 0)
        return;

    //
    //  Handle special case of single char. We either kill the char and return
    //  an empty string or we return with no action.
    //
    if (!(*(pszStripBuf+1)))
    {
        if (pszFindChar(pszStripChars, *pszStripBuf))
            *pszStripBuf = 0;

        return;
    }

    // Get the current size, which we use as the max in string ops
    tCIDLib::TCard4 c4MaxChars = TRawStr::c4StrLen(pszStripBuf);

    // Start off the start pointer at the base of the passed buffer
    tCIDLib::Tch* pszStart = pszStripBuf;

    // Start the end pointer at the nul at the end
    tCIDLib::Tch* pszEnd = pszStripBuf + c4StrLen(pszStripBuf);

    if (eMode & tCIDLib::EStripMode_Leading)
    {
        //
        //  If leading character stripping is on, then handle that first.
        //  We will just run a pointer up until we hit the first non-strip
        //  char.
        //
        while (*pszStart)
        {
            // If the character is not in the strip list, then break
            if (!pszFindChar(pszStripChars, *pszStart))
                break;

            // Move up a character
            pszStart++;
        }

        // We stripped it away, so zero the passed string and return
        if (!(*pszStart))
        {
            *pszStripBuf = 0;
            return;
        }
    }

    //
    //  Ok, the pszStart pointer points to the first non-strip char. So now
    //  lets do the trailing stripping if needed. We know that there is
    //  text there, at least 1 char, between the start and end.
    //
    if (eMode & tCIDLib::EStripMode_Trailing)
    {
        while (pszEnd > pszStart)
        {
            // If previous char is not in the strip char list, then done
            if (!pszFindChar(pszStripChars, *(pszEnd-1)))
                break;
            pszEnd--;
        }

        // If we hit the leading edge, then we are done
        if (pszEnd == pszStart)
        {
            pszStripBuf[0] = 0;
            return;
        }

        // Otherwise, put a nul in the end
        *pszEnd = 0;
    }

    //
    //  Ok we now have the starting and ending pointers, which we know are
    //  not the same so something is in there. If middle stripping is on,
    //  the we have to handle it. If the string that is left at this point
    //  is 1 or 2 chars, then nothing to do because there cannot be any
    //  middle.
    //
    //  !!!!! We initially reference pszEnd, but once we start stripping
    //  !!!!! in the middle, it is no longer valid and is not used anymore
    //
    if ((eMode & tCIDLib::EStripMode_Middle) && (c4StrLen(pszStart) > 2))
    {
        tCIDLib::TBoolean bTotComp = kCIDLib::False;

        // Test for total compaction flag. If on, then set the bTotComp flag.
        if (eMode & tCIDLib::EStripMode_Total)
            bTotComp = kCIDLib::True;

        //
        //  We have to start at the first non-strip char. If leading stripping
        //  was done, then the loop will fall out immediately. If we go all
        //  the way to the end, then strip the whole string clean and return.
        //
        while (pszStart < pszEnd)
        {
            if (!pszFindChar(pszStripChars, *pszStart))
                break;

            pszStart++;
        }

        // If we stripped the whole thing, then we are done
        if (pszStart == pszEnd)
        {
            *pszStripBuf = 0;
            return;
        }

        //
        //  If there is only 1 char left, we are done. We just have to copy
        //  the 1 char down and nul terminate it.
        //
        if (!(*(pszStart+1)))
        {
            if (pszStart != pszStripBuf)
            {
                *pszStripBuf = *pszStart;
                *(pszStripBuf+1) = 0;
            }
            return;
        }

        // !!! pszEnd is no longer valid from here on potentially

        //
        //  We use two pointers here, one of which is anchor that is dropped
        //  when we find a strip char, and the other run up to the next
        //  non-strip char. Then we can just copy the string downward to get
        //  rid of the chars in between.
        //
        //  Start running the search up at the next char after the start
        //  char which cannot be changed by middle stripping if leading
        //  stripping was done.
        //
        tCIDLib::Tch*   pszAnchor = pszStart+1;
        tCIDLib::Tch*   pszSearch = pszAnchor;

        while (*pszSearch)
        {
            // Look for a strip char via the search pointer
            while (*pszSearch)
            {
                if (pszFindChar(pszStripChars, *pszSearch))
                    break;
                pszSearch++;
            }

            //
            //  If we hit the end then done, because they are trailing,
            //  not middle, strip chars.
            //
            if (!(*pszSearch))
                break;

            //
            //  Drop the anchor pointer here and start looking for a non
            //  strip char at the next location.
            //
            pszAnchor = pszSearch;
            pszSearch++;
            while (*pszSearch)
            {
                if (!pszFindChar(pszStripChars, *pszSearch))
                    break;

                pszSearch++;
            }

            //
            //  If we hit the end then done, because they are trailing,
            //  not middle, strip chars.
            //
            if (!(*pszSearch))
                break;

            //
            //  Ok, we can copy down from the char where the search hit a
            //  non-strip, to the anchor pointer. We have to deal with the
            //  special case of total compaction vs. non-total. If not
            //  total, then put a space in at the anchor and bump it up
            //  before doing the copy.
            //
            if (!bTotComp)
            {
                *pszAnchor = kCIDLib::chSpace;
                pszAnchor++;
            }

            //
            //  If we did not bump the anchor back up to the search, then copy
            //  the contents back down.
            //
            if (pszAnchor != pszSearch)
                CopyStr(pszAnchor, pszSearch, c4MaxChars);

            // Start the search at the anchor point
            pszSearch = pszAnchor;
        }
    }

    //
    //  Ok, we need to copy from pszStart to the start of the buffer. These
    //  ranges overlap but pszStart is guaranteed to be greater than the
    //  buffer start (if they are not equal.) If they are equal, then we are
    //  done.
    //
    if (pszStart == pszStripBuf)
        return;

    CopyStr(pszStripBuf, pszStart, c4MaxChars);
}
