//
//  FILE NAME: CIDKernel_RawMemory.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/13/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_RawMemory.Cpp module. This
//  module provides APIs for manipulating raw memory buffers.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"
#include    <memory.h>


// ----------------------------------------------------------------------------
//  Global functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TRawMem::CommitPages(           tCIDLib::TVoid*         pPageAdr
                        , const tCIDLib::TCard4         c4PageCount
                        , const tCIDLib::EMemAccFlags   eAccess)
{
    if (!VirtualAlloc
    (
        pPageAdr
        , c4PageCount * kCIDLib::c4MemPageSize
        , MEM_COMMIT
        , eAccess))
    {
        TKrnlError::ThrowKrnlError();
    }
}


tCIDLib::TVoid TRawMem::FreeSysMem(const tCIDLib::TVoid* pToFree)
{
    if (!VirtualFree((tCIDLib::TVoid*)pToFree, 0, MEM_RELEASE))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid
TRawMem::CopyMemBuf(        tCIDLib::TVoid* const   pDest
                    , const tCIDLib::TVoid* const   pSrc
                    , const tCIDLib::TCard4         c4Count)
{
    memcpy(pDest, pSrc, c4Count);
}


tCIDLib::ESortComps
TRawMem::eCompareMemBuf(const   tCIDLib::TVoid* const   p1
                        , const tCIDLib::TVoid* const   p2
                        , const tCIDLib::TCard4         c4Size)
{
     tCIDLib::TInt4 i4Res = memcmp(p1, p2, c4Size);
    if (i4Res < 0)
        return tCIDLib::ESort_FirstLess;
    else if (i4Res > 0)
        return tCIDLib::ESort_FirstGreater;
    return tCIDLib::ESort_Equal;

}

#pragma warning(disable : 4035)
tCIDLib::THashVal
TRawMem::hshHashBuffer( const   tCIDLib::TVoid* const   pBuf
                        , const tCIDLib::TCard4         c4Modulus
                        , const tCIDLib::TCard4         c4BufSz)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        sub     eax, eax            // Clear out return
        mov     ecx, c4BufSz        // Load up max size
        jecxz   Exit                // Exit w/ 0 if no size
        mov     esi, pBuf           // Load up the buf address
        mov     edx, 0              // Use edx as accumulator

    LoopStart:                      // And enter the loop
        lodsb                       // Load up a byte from str
        rol     edx, 1              // Rotate accum left 1 bit
        xor     dl, al              // XOR w/ low byte of accum
        Loop    LoopStart           // And do the next byte

        mov     eax, edx            // Get accum to eax
        sub     edx, edx            // Clear edx for divide
        div     c4Modulus           // Divide by modulus
        mov     eax, edx            // Keep remainder
    Exit:
    }
    #else
    tCIDLib::THashVal       hshRet = 0;
    const tCIDLib::TCard1*  pc1Buf = (const tCIDLib::TCard1*)pBuf;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4BufSz; c4Index++)
    {
        hshRet <<= 1;
        hshRet ^= tCIDLib::THashVal(*pc1Buf);
        pc1Buf++;
    }

    // Divide by the modulus for return
    return tCIDLib::THashVal(hshRet % c4Modulus);
    #endif
}
#pragma warning(default : 4035)

tCIDLib::TVoid
TRawMem::MoveMemBuf(        tCIDLib::TVoid* const   pDest
                    ,       tCIDLib::TVoid* const   pSrc
                    , const tCIDLib::TCard4         c4Count)
{
    memmove(pDest, pSrc, c4Count);
}


tCIDLib::TVoid*
TRawMem::pAllocSysMem(  const   tCIDLib::TCard4         c4Size
                        , const tCIDLib::EMemAccFlags   eAccess
                        , const TRawMem::EMemAllFlags   eAlloc)
{
    // Try to allocate the buffer
    tCIDLib::TVoid* pBuf = VirtualAlloc(0, c4Size, eAlloc, eAccess);

    if (!pBuf)
        TKrnlError::ThrowKrnlError();

    return pBuf;
}


tCIDLib::TVoid* TRawMem::pPageBaseAdr(const tCIDLib::TVoid* pBufContained)
{
    MEMORY_BASIC_INFORMATION RawInfo;
    VirtualQuery(pBufContained, &RawInfo, sizeof(MEMORY_BASIC_INFORMATION));
    return RawInfo.BaseAddress;
}


tCIDLib::TVoid*
TRawMem::pCompareAndExchange(           tCIDLib::TVoid*&        pToFill
                                , const tCIDLib::TVoid* const   pNew
                                , const tCIDLib::TVoid* const   pCompare)
{
    // <TBD> Get rid of casts when InterlockedCompareExchange prototype is fixed
    return InterlockedCompareExchange
    (
        &pToFill
        , (tCIDLib::TVoid*)pNew
        , (tCIDLib::TVoid*)pCompare
    );
}

tCIDLib::TVoid*
TRawMem::pExchange(         tCIDLib::TVoid*&        pToFill
                    , const tCIDLib::TVoid* const   pNew)
{
    return (tCIDLib::TVoid*)InterlockedExchange
    (
        (tCIDLib::TInt4*)&pToFill
        , (tCIDLib::TInt4)pNew
    );
}


tCIDLib::TVoid*
TRawMem::pQueryMemFlags(const   tCIDLib::TVoid*         pBufToQuery
                        ,       TRawMem::TSysMemInfo&   MemInfo)
{
    MEMORY_BASIC_INFORMATION RawInfo;
    VirtualQuery(pBufToQuery, &RawInfo, sizeof(MEMORY_BASIC_INFORMATION));

    MemInfo.c4RegionSize  = RawInfo.RegionSize;
    MemInfo.eAllocAccess  = tCIDLib::EMemAccFlags(RawInfo.AllocationProtect);
    MemInfo.eRegionAccess = tCIDLib::EMemAccFlags(RawInfo.Protect);
    MemInfo.eRegionState  = tCIDLib::EMemStates(RawInfo.State);
    MemInfo.eRegionType   = tCIDLib::EMemTypes(RawInfo.Type);
    MemInfo.pAllocBase    = RawInfo.AllocationBase;

    return RawInfo.BaseAddress;
}


tCIDLib::TVoid
TRawMem::SetMemBuf(         tCIDLib::TVoid* const   pMem
                    , const tCIDLib::TCard1         c1Fill
                    , const tCIDLib::TCard4         c4Count)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        mov         edi, pMem
        mov         al, c1Fill
        mov         ecx, c4Count
        rep stosb
    }
    #else
    memset(pMem, c1Fill, c4Count);
    #endif
}

tCIDLib::TVoid
TRawMem::SetMemBuf(         tCIDLib::TVoid* const   pMem
                    , const tCIDLib::TInt1          i1Fill
                    , const tCIDLib::TCard4         c4Count)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        mov         edi, pMem
        mov         al, i1Fill
        mov         ecx, c4Count
        rep stosb
    }
    #else
    memset(pMem, i1Fill, c4Count);
    #endif
}

tCIDLib::TVoid
TRawMem::SetMemBuf(         tCIDLib::TVoid* const   pMem
                    , const tCIDLib::TCard2         c2Fill
                    , const tCIDLib::TCard4         c4Count)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        mov         edi, pMem
        mov         ax, c2Fill
        mov         ecx, c4Count
        rep stosw
    }
    #else

    //
    //  Optimize if the high and low bytes are the same, since we can just
    //  do a memset. Otherwise, have to do it manually.
    //
    if (((c2Fill & 0xFF00) >> 8) == (c2Fill & 0x00FF))
    {
        memset(pMem, tCIDLib::TCard1(c2Fill & 0x00FF), c4Count * 2);
        return;
    }

    tCIDLib::TCard2* pc2Buf = (tCIDLib::TCard2*)pMem;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        *pc2Buf = c2Fill;
        pc2Buf++;
    }
    #endif
}

tCIDLib::TVoid
TRawMem::SetMemBuf(         tCIDLib::TVoid* const   pMem
                    , const tCIDLib::TInt2          i2Fill
                    , const tCIDLib::TCard4         c4Count)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        mov         edi, pMem
        mov         ax, i2Fill
        mov         ecx, c4Count
        rep stosw
    }
    #else

    //
    //  Optimize if the high and low bytes are the same, since we can just
    //  do a memset. Otherwise, have to do it manually.
    //
    if (((i2Fill & 0xFF00) >> 8) == (i2Fill & 0x00FF))
    {
        memset(pMem, tCIDLib::TCard1(i2Fill & 0x00FF), c4Count * 2);
        return;
    }

    tCIDLib::TInt2* pi2Buf = (tCIDLib::TInt2*)pMem;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        *pi2Buf = i2Fill;
        pi2Buf++;
    }
    #endif
}

tCIDLib::TVoid
TRawMem::SetMemBuf(         tCIDLib::TVoid* const   pMem
                    , const tCIDLib::TCard4         c4Fill
                    , const tCIDLib::TCard4         c4Count)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        mov         eax, c4Fill
        mov         ecx, c4Count
        mov         edi, pMem
        rep stosd
    }
    #else

    // Optimize if the fill is 0. Otherwise, have to do it manually.
    if (!c4Fill)
    {
        memset(pMem, 0, c4Count * 4);
        return;
    }

    tCIDLib::TCard4* pc4Buf = (tCIDLib::TCard4*)pMem;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        *pc4Buf = c4Fill;
        pc4Buf++;
    }
    #endif
}

tCIDLib::TVoid
TRawMem::SetMemBuf(         tCIDLib::TVoid* const   pMem
                    , const tCIDLib::TInt4          i4Fill
                    , const tCIDLib::TCard4         c4Count)
{
    #if defined(CIDLIB_CPU_X86)
    _asm
    {
        mov         eax, i4Fill
        mov         ecx, c4Count
        mov         edi, pMem
        rep stosd
    }
    #else

    // Optimize if the fill is 0. Otherwise, have to do it manually.
    if (!i4Fill)
    {
        memset(pMem, 0, c4Count * 4);
        return;
    }

    tCIDLib::TInt4* pi4Buf = (tCIDLib::TInt4*)pMem;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        *pi4Buf = i4Fill;
        pi4Buf++;
    }
    #endif
}

tCIDLib::TVoid
TRawMem::SetMemBuf(         tCIDLib::TVoid* const   pMem
                    , const tCIDLib::TFloat4        f4Fill
                    , const tCIDLib::TCard4         c4Count)
{
    // Optimize if the fill is 0. Otherwise, have to do it manually.
    if (!f4Fill)
    {
        memset(pMem, 0, c4Count * sizeof(f4Fill));
        return;
    }

    tCIDLib::TFloat4* pf4Buf = (tCIDLib::TFloat4*)pMem;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        *pf4Buf = f4Fill;
        pf4Buf++;
    }
}

tCIDLib::TVoid
TRawMem::SetMemBuf(         tCIDLib::TVoid* const   pMem
                    , const tCIDLib::TFloat8        f8Fill
                    , const tCIDLib::TCard4         c4Count)
{
    // Optimize if the fill is 0. Otherwise, have to do it manually.
    if (!f8Fill)
    {
        memset(pMem, 0, c4Count * sizeof(f8Fill));
        return;
    }

    tCIDLib::TFloat8* pf8Buf = (tCIDLib::TFloat8*)pMem;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        *pf8Buf = f8Fill;
        pf8Buf++;
    }
}
