//
//  FILE NAME: CIDKernel_RawBits.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 09/22/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_RawBits.Cpp module. This module
//  implements a number of methods for manipulating bit fields, which are
//  placed within a namespace called TRawBits.
//
//  CAVEATS/GOTCHAS:
//

namespace TRawBits
{
    // -----------------------------------------------------------------------
    //  Bit rotations for various integral types
    // -----------------------------------------------------------------------
    KRNLEXTERN KRNLEXPORT tCIDLib::TCard4 c4RotateLeft
    (
        const   tCIDLib::TCard4     c4Value
        , const tCIDLib::TCard4     c4RotateCount
    );

    KRNLEXTERN KRNLEXPORT tCIDLib::TCard4 c4RotateRight
    (
        const   tCIDLib::TCard4     c4Value
        , const tCIDLib::TCard4     c4RotateCount
    );


    // -----------------------------------------------------------------------
    //  Extracts various fields from a 32 or 16 bit value
    // -----------------------------------------------------------------------
    inline tCIDLib::TCard1 c1Low8From16(const tCIDLib::TCard2 c2Val)
    {
        return tCIDLib::TCard1(c2Val & 0xFFUL);
    }

    inline tCIDLib::TCard1 c1High8From16(const tCIDLib::TCard2 c2Val)
    {
        return tCIDLib::TCard1((c2Val & 0xFF00UL) >> 8);
    }

    inline tCIDLib::TCard2 c2Low16From32(const tCIDLib::TCard4 c4Val)
    {
        return tCIDLib::TCard2(c4Val & 0xFFFFUL);
    }

    inline tCIDLib::TCard2 c2High16From32(const tCIDLib::TCard4 c4Val)
    {
        return tCIDLib::TCard2((c4Val & 0xFFFF0000UL) >> 16);
    }

    inline tCIDLib::TCard1 c1Low8From32(const tCIDLib::TCard4 c4Val)
    {
        return tCIDLib::TCard1(c4Val & 0xFFUL);
    }

    inline tCIDLib::TCard1 c1Sec8From32(const tCIDLib::TCard4 c4Val)
    {
        return tCIDLib::TCard1((c4Val & 0xFF00UL) >> 8);
    }

    inline tCIDLib::TCard1 c1Third8From32(const tCIDLib::TCard4 c4Val)
    {
        return tCIDLib::TCard1((c4Val & 0xFF0000UL) >> 16);
    }

    inline tCIDLib::TCard1 c1High8From32(const tCIDLib::TCard4 c4Val)
    {
        return tCIDLib::TCard1((c4Val & 0xFF000000UL) >> 24);
    }


    // -----------------------------------------------------------------------
    //  Creates a 32 bit value from two 16 bit values or a 16 bit value from
    //  two 8 bit values or a 32 bit value from 4 8 bit values.
    // -----------------------------------------------------------------------
    inline tCIDLib::TCard4 c4From16(const   tCIDLib::TCard2 c2Low
                                    , const tCIDLib::TCard2 c2High)
    {
        return ((tCIDLib::TCard4(c2High) << 16) | c2Low);
    }

    inline tCIDLib::TCard2 c2From8( const   tCIDLib::TCard1 c1Low
                                    , const tCIDLib::TCard1 c1High)
    {
        return tCIDLib::TCard2
        (
            (tCIDLib::TCard2(c1High) << 8) | tCIDLib::TCard2(c1Low)
        );
    }


    // -----------------------------------------------------------------------
    //  Creates a 64 bit integer from 2 32 bit values, or extracts the 32 bit
    //  fields from a 64 bit value.
    // -----------------------------------------------------------------------
    inline tCIDLib::TInt8 i8From32( const   tCIDLib::TCard4 c4Low
                                    , const tCIDLib::TCard4 c4High)
    {
        tCIDLib::TInt8 i8Ret = c4High;
        i8Ret <<= 32;
        i8Ret |= c4Low;
        return i8Ret;
    }

    inline tCIDLib::TCard4 c4Low32From64(const tCIDLib::TInt8& i8Val)
    {
        return tCIDLib::TCard4(i8Val & kCIDLib::c4MaxCard);
    }

    inline tCIDLib::TCard4 c4High32From64(const tCIDLib::TInt8& i8Val)
    {
        return tCIDLib::TCard4((i8Val & 0xFFFFFFFF00000000) >> 32);
    }


    // -----------------------------------------------------------------------
    //  Swap byte order
    // -----------------------------------------------------------------------
    inline tCIDLib::TCard2 c2SwapBytes(const tCIDLib::TCard2 c2ToSwap)
    {
        tCIDLib::TCard2 c2Ret = (c2ToSwap >> 8);
        c2Ret |= c2ToSwap & 0xFF;
        return c2Ret;
    }

    inline tCIDLib::TCard4 c4SwapBytes(const tCIDLib::TCard4 c4ToSwap)
    {
        tCIDLib::TCard4 c4Ret = c4ToSwap >> 24;
        c4Ret |= ((c4ToSwap >> 16) & 0xFF) << 8;
        c4Ret |= ((c4ToSwap >> 8) & 0xFF) << 16;
        c4Ret |= (c4ToSwap << 24);
        return c4Ret;
    }

    inline tCIDLib::TInt2 i2SwapBytes(const tCIDLib::TInt2 i2ToSwap)
    {
        tCIDLib::TInt2 i2Ret = (i2ToSwap >> 8);
        i2Ret |= i2ToSwap & 0xFF;
        return i2Ret;
    }

    inline tCIDLib::TInt4 i4SwapBytes(const tCIDLib::TInt4 i4ToSwap)
    {
        tCIDLib::TInt4 i4Ret = i4ToSwap >> 24;
        i4Ret |= ((i4ToSwap >> 16) & 0xFF) << 8;
        i4Ret |= ((i4ToSwap >> 8) & 0xFF) << 16;
        i4Ret |= (i4ToSwap << 24);
        return i4Ret;
    }
};
