//
// NAME: CIDKernel_ProcessRegistry.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDKernel_ProcessRegistry.Cpp module. This
//  module provides the core support for the process registry mechanism
//  of the CIDLib system. The process registry is basically an area of
//  shared memory that contains information about every CIDLib process that
//  is currently running on the workstation.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 03/24/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//

#pragma pack(push, CIDLIBPACK)

namespace tCIDLib
{
    // ------------------------------------------------------------------------
    //  This is the format of a single entry in the process registry. An
    //  array of c4MaxRegProc of these is contained in the main process
    //  registry data area.
    //
    //  bUsed
    //      This indicates whether this entry is used or not, since
    //      processes can come and go.
    //
    //  pidThis
    //      This is the process id of this process. This can be used to
    //      create the names of many CIDLib process specific named
    //      resources. And its used to create handles when required.
    //
    //  szProcName
    //      This is extracted from the .Exe module, so it is the process
    //      name given in the .Def file for the .Exe facility.
    //
    //  szInfoLine
    //      The optional info line that client process can set.
    //
    //  c4InfoCard
    //      The optional info cardinal value that the client process can
    //      set.
    // ------------------------------------------------------------------------
    struct  TRawProcessEntry
    {
        tCIDLib::TBoolean       bUsed;
        tCIDLib::TProcessId     pidThis;
        tCIDLib::TZStr32        szProcName;

        tCIDLib::TZStr64        szInfoLine;
        tCIDLib::TCard4         c4InfoCard;
    };
};

namespace kCIDLib
{
    // ------------------------------------------------------------------------
    //  Constant data
    // ------------------------------------------------------------------------
    const tCIDLib::TCard4   c4RegMemPages   = 2;
    const tCIDLib::TCard4   c4MaxRegProc    = (((c4MemPageSize*c4RegMemPages)-64)
                                             / sizeof(tCIDLib::TRawProcessEntry));
}

namespace tCIDLib
{
    // ------------------------------------------------------------------------
    //  This is the format of the main process registry memory area.
    //
    //  szMagicVal
    //      This is a magic value (a string of "Dean Roddey") that is
    //      embedded here as a sanity check for external programs, to let
    //      them make sure that they are looking at the right memory in
    //      the right format.
    //
    //  c4ProcCount
    //      This is the number of entries in the aProcs[] array. These
    //      entries are not always compacted, because processes can come
    //      and go, so check the bUsed member of the entry to see if
    //      valid data is in the entry.
    //
    //  apreList
    //      The list of process entry structures in the registry.
    // ------------------------------------------------------------------------
    struct  TRawProcRegistry
    {
        tCIDLib::Tch        szMagicValue[12];
        tCIDLib::TCard4     c4ProcCount;
        TRawProcessEntry    apreList[kCIDLib::c4MaxRegProc];
    };
};


#if !defined(CIDKRNL_NOCLASSES)

// ----------------------------------------------------------------------------
//  Create a typed shared buffer class for the raw process registry
// ----------------------------------------------------------------------------
typedef TKrnlTypedSharedBuf<tCIDLib::TRawProcRegistry>  TProcRegistryBuf;


// ----------------------------------------------------------------------------
//   CLASS: TKrnlProcRegistry
//  PREFIX: kpreg
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlProcRegistry
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKrnlProcRegistry();

        ~TKrnlProcRegistry();


        // --------------------------------------------------------------------
        //  Public, static methods
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean bIsRunning
        (
            const   tCIDLib::TCard4         c4Index
        );

        static tCIDLib::TCard4 c4InfoCard();

        static tCIDLib::TCard4 c4ProcessCount();

        static tCIDLib::TVoid CloseHandle
        (
            const   tCIDLib::TProcessHandle hprocToClose
        );

        static tCIDLib::EExitCodes eWaitForDeath
        (
            const   tCIDLib::TProcessHandle hprocToWaitFor
            , const tCIDLib::TCard4         c4MilliSecs
        );

        static tCIDLib::TProcessHandle hprocFromId
        (
            const   tCIDLib::TProcessId     pidFrom
        );

        static const tCIDLib::TRawProcessEntry& preEntryAt
        (
            const   tCIDLib::TCard4         c4Index
        );

        static const tCIDLib::TRawProcessEntry& preFindProcess
        (
            const   tCIDLib::TProcessId&    pidToFind
        );

        static const tCIDLib::TRawProcessEntry& preThisProcess();

        static const tCIDLib::Tch* pszInfoLine();

        static tCIDLib::TVoid SetInfoCard
        (
            const   tCIDLib::TCard4         c4New
        );

        static tCIDLib::TVoid SetInfoLine
        (
            const   tCIDLib::TZStr64        pszNew
        );

        static tCIDLib::TVoid ValidateRegistry();

    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TKrnlProcRegistry(const TKrnlProcRegistry&);

        tCIDLib::TVoid operator=(const TKrnlProcRegistry&);
};

#endif

#pragma pack(pop)
