//
// NAME: CIDKernel_ProcessRegistry.Cpp
//
// DESCRIPTION:
//
//  This module implements the process registry's core functionality. It
//  basically creates and maintains the shared memory area that makes up
//  the process registry. The registry is initialized during DLL init.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 03/24/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"



// ----------------------------------------------------------------------------
//  Local data
//
//  __c4OurIndex
//      This is the index that our process got put into the registry. This
//      makes it easier for us to get to it without a search.
//
//  __pkmtxAccess
//      This is a pointer to the named mutex that is used to control access
//      to the registry.
//
//  __pktsbRegistry
//      This is the typed shared buffer to the registry header. It is pointed
//      at the registry shared memory area. We have to use a pointer here
//      because of bootstrapping issues.
// ----------------------------------------------------------------------------
static tCIDLib::TCard4              __c4OurIndex = kCIDLib::c4MaxCard;
static TKrnlMutex*                  __pkmtxAccess = 0;
static TProcRegistryBuf*            __pktsbRegistry;


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __bCheckIsRunning
//
// DESCRIPTION:
//
//  This function will create a handle for the passed id and test it for
//  having been signaled, then close it.
// ---------------------------------------
//   INPUT: pidToCheck is the process id to check
//
//  OUTPUT: None
//
//  RETURN: True if running, else False.
//
static tCIDLib::TBoolean
__bCheckIsRunning(const tCIDLib::TProcessId pidToCheck)
{
#if 0
    //
    //  We have to duplicate the handle in order to get a valid handle for
    //  use within our process. If this fails, then the handle is obviously
    //  invalid and we can return false right away.
    //
    if (!DuplicateHandle
    (
        hprocToCheck
        , hprocToCheck
        , TKrnlSysInfo::hprocThis()
        , &hprocTmp
        , SYNCHRONIZE
        , FALSE
        , 0))
    {
        return kCIDLib::False;
    }
#endif

    tCIDLib::TProcessHandle hprocTmp = OpenProcess
    (
        SYNCHRONIZE
        , FALSE
        , pidToCheck
    );

    // Try to wait on the handle
    tCIDLib::TCard4 c4Result = WaitForSingleObject(hprocTmp, 0);

    // Close the temporary handle before we do anything else
    ::CloseHandle(hprocTmp);

    if (c4Result == WAIT_FAILED)
        return kCIDLib::False;

    if (c4Result == WAIT_OBJECT_0)
        return kCIDLib::False;

    return kCIDLib::True;
}


// ----------------------------------------------------------------------------
//  Intra-facility functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid
_InitTermProcessRegistry(const  tCIDLib::EInitTerm      eInitTerm
                        , const tCIDLib::EGlobalStates  eGlobals
                        , const tCIDLib::TModHandle     hmodThis
                        , const tCIDLib::TCard4         c4MaxChars
                        ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszReason = kMessages::pszGen_Unknown;
    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //
            //  Create the process registry resource name. Its used for the
            //  memory and mutex.
            //
            TKrnlRscName krsnProcReg
            (
                L"CIDCorp"
                , L"ProcRegistry"
                , L"MainRegistry"
            );

            //
            //  Construct the mutex, giving it the correct name. Then try to create
            //  or open it.
            //
            try
            {
                tCIDLib::Tch szTmpName[1024];

                // Build the full name to the mutex
                krsnProcReg.BuildFullName
                (
                    szTmpName
                    , c4MaxBufChars(szTmpName)
                    , tCIDLib::ENamedRsc_Mutex
                );

                __pkmtxAccess = new TKrnlMutex(szTmpName);
                __pkmtxAccess->CreateOrOpen(tCIDLib::ELockState_Unlocked);

                //
                //  Lock the semaphore now. We will wait for up to 5 seconds worst
                //  case to get it locked. In normal circumstances it should never
                //  be locked more than a very small time.
                //
                pszReason = kMessages::pszPReg_Locking;
                TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);

                // Build the full name to the memory
                krsnProcReg.BuildFullName
                (
                    szTmpName
                    , c4MaxBufChars(szTmpName)
                    , tCIDLib::ENamedRsc_Memory
                );

                 //
                //  Ok, now we need to open or create the shared memory area of
                //  the registry. Since we now own the access list mutex, we can
                //  first check for its existence, then open it or create it as
                //  needed. If we have to create it, then its our responsibility
                //  to init the data.
                //
                pszReason = kMessages::pszPReg_OpenMem;
                tCIDLib::TBoolean bCreated;
                __pktsbRegistry = new TProcRegistryBuf
                (
                    szTmpName
                    , tCIDLib::EMemAcc_ReadWrite
                    , tCIDLib::ECreateAct_OpenOrCreate
                    , bCreated
                );

                if (bCreated)
                {
                    //
                    //  Initialize it. We zero the whole thing out then put the
                    //  magic value in, which is my name.
                    //
                    pszReason = kMessages::pszPReg_InitMem;
                    __pktsbRegistry->Zero();
                    TRawStr::CopyStr
                    (
                        (*__pktsbRegistry)->szMagicValue
                        , L"Dean Roddey"
                        , c4MaxBufChars((*__pktsbRegistry)->szMagicValue)
                    );
                    (*__pktsbRegistry)->c4ProcCount = 0;
                }

                //
                //  Find an unused entry in the registry for us to use. Save this
                //  as our index in a local static.
                //
                pszReason = kMessages::pszPReg_EntryInit;
                tCIDLib::TCard4 c4TmpIndex;
                for (c4TmpIndex = 0; c4TmpIndex < kCIDLib::c4MaxRegProc; c4TmpIndex++)
                {
                    if (!(*__pktsbRegistry)->apreList[c4TmpIndex].bUsed)
                        break;
                }

                if (c4TmpIndex == kCIDLib::c4MaxRegProc)
                    TKrnlError::ThrowKrnlError(kKrnlErrors::errcProcRegFull);

                // Store out index
                __c4OurIndex = c4TmpIndex;

                //
                //  Ok, either way now the process registry is created and
                //  initalized, so let's add ourself to the registry.
                //
                tCIDLib::TRawProcessEntry* pEntry =
                                    &(*__pktsbRegistry)->apreList[__c4OurIndex];

                // Zero our entry
                TRawMem::SetMemBuf
                (
                    pEntry
                    , tCIDLib::TCard1(0)
                    , sizeof(tCIDLib::TRawProcessEntry)
                );

                pEntry->pidThis = TKrnlSysInfo::pidThis();
                TRawStr::CopyStr
                (
                    pEntry->szProcName
                    , TKrnlSysInfo::pszProcessName()
                    , c4MaxBufChars(pEntry->szProcName)
                );

                // Make this entry used and bump up the count of entries
                pEntry->bUsed = kCIDLib::True;
                (*__pktsbRegistry)->c4ProcCount++;
            }

            catch(...)
            {
                // Free the access mutex if we got it allocated
                delete __pkmtxAccess;
                __pkmtxAccess = 0;

                // Free the process registry memory if we got it allocated
                if (__pktsbRegistry)
                {
                    // Don't allow an exception here to skip out
                    try
                    {
                        delete __pktsbRegistry;
                    }

                    catch(const TKrnlError& errToCatch)
                    {
                        _ERRPOPUP_(kMessages::pszPReg_FreeMem, errToCatch.errcId())
                    }
                    __pktsbRegistry = 0;
                }
                throw;
            }
        }
         else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
              &&  (eGlobals == tCIDLib::EGlobalState_After))
        {
            //
            //  We need to remove ourself from the process registry. We have to
            //  be very careful here because we may be going down due to some
            //  very primal error and cannot count on anything.
            //
            //  First check to see if the access mutex ever got created. If not,
            //  then just give up now.
            //
            if (!__pkmtxAccess)
                return;

            // Lock the mutex so that we can remove ourselves from registry
            pszReason = kMessages::pszPReg_Locking;
            TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 15 Seconds);

            //
            //  See if we ever got our entry set. We can know by checking
            //  our locally stored process registry index.
            //
            if (__c4OurIndex == kCIDLib::c4MaxCard)
                return;

            if ((*__pktsbRegistry).pData())
            {
                pszReason = kMessages::pszPReg_EntryClean;
                tCIDLib::TRawProcessEntry* pEntry
                                = &(*__pktsbRegistry)->apreList[__c4OurIndex];

                pEntry->bUsed = kCIDLib::False;
                pEntry->pidThis = kCIDLib::pidInvalid;
                pEntry->szProcName[0] = 0;

                // And bump down the count of processes
                (*__pktsbRegistry)->c4ProcCount--;
            }
        }
    }

    catch(...)
    {
        TRawStr::CopyStr
        (
            pszFailReason
            , pszReason
            , c4MaxChars
        );
        throw;
    }
}




// ----------------------------------------------------------------------------
//   CLASS: TKrnlProcRegistry
//  PREFIX: kpreg
// ----------------------------------------------------------------------------

TKrnlProcRegistry::TKrnlProcRegistry()
{
}

TKrnlProcRegistry::~TKrnlProcRegistry()
{
}

// ----------------------------------------------------------------------------
//  TKrnlProcRegistry: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TKrnlProcRegistry::bIsRunning(const tCIDLib::TCard4 c4Index)
{
    if (c4Index >= kCIDLib::c4MaxRegProc)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcProcRegIndex);

    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);
    return __bCheckIsRunning((*__pktsbRegistry)->apreList[c4Index].pidThis);
}


tCIDLib::TCard4 TKrnlProcRegistry::c4InfoCard()
{
    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);
    return (*__pktsbRegistry)->apreList[__c4OurIndex].c4InfoCard;
}


tCIDLib::TCard4 TKrnlProcRegistry::c4ProcessCount()
{
    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);
    return (*__pktsbRegistry)->c4ProcCount;
}

tCIDLib::TVoid
TKrnlProcRegistry::CloseHandle(const tCIDLib::TProcessHandle hprocToClose)
{
    if (!::CloseHandle(hprocToClose))
        TKrnlError::ThrowKrnlError();
}

tCIDLib::EExitCodes
TKrnlProcRegistry::eWaitForDeath(const   tCIDLib::TProcessHandle hprocToWaitFor
                                 , const tCIDLib::TCard4         c4MilliSecs)
{
    tCIDLib::TCard4 c4Res = WaitForSingleObject(hprocToWaitFor, c4MilliSecs);

    if ((c4Res == WAIT_FAILED) || (c4Res == WAIT_TIMEOUT))
        TKrnlError::ThrowKrnlError();

    // Get the exit code and return it
    tCIDLib::TCard4 c4Exit;
    if (!GetExitCodeProcess(hprocToWaitFor, &c4Exit))
    {
//        if (c4Exit == STILL_ACTIVE)
//            TKrnlError::ThrowKrnlError(kKrnlErrors::errcStillRunning);
        TKrnlError::ThrowKrnlError();
    }
    return tCIDLib::EExitCodes(c4Exit);
}

tCIDLib::TProcessHandle
TKrnlProcRegistry::hprocFromId(const tCIDLib::TProcessId pidFrom)
{
    tCIDLib::TProcessHandle hprocTmp = OpenProcess
    (
        SYNCHRONIZE
        | PROCESS_QUERY_INFORMATION
        | PROCESS_SET_INFORMATION
        | PROCESS_TERMINATE
        , FALSE
        , pidFrom
    );

    if (hprocTmp == kCIDLib::hprocInvalid)
        TKrnlError::ThrowKrnlError();
    return hprocTmp;
}

const tCIDLib::TRawProcessEntry&
TKrnlProcRegistry::preEntryAt(const tCIDLib::TCard4 c4Index)
{
    if (c4Index >= kCIDLib::c4MaxRegProc)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcProcRegIndex);

    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);

    // Return a pointer to the indicated index
    return (*__pktsbRegistry)->apreList[c4Index];
}


const tCIDLib::TRawProcessEntry&
TKrnlProcRegistry::preFindProcess(const tCIDLib::TProcessId& pidToFind)
{
    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);

    //
    //  Look for the entry with this process id and get a pointer to it
    //  for return. Default it to 0 in case we dont find it.
    //
    const tCIDLib::TRawProcessEntry* pEntry = 0;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4MaxRegProc; __c4OurIndex++)
    {
        if ((*__pktsbRegistry)->apreList[c4Index].pidThis == pidToFind)
        {
            pEntry = &(*__pktsbRegistry)->apreList[c4Index];
            break;
        }
    }
    return *pEntry;
}


const tCIDLib::Tch* TKrnlProcRegistry::pszInfoLine()
{
    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);
    return (*__pktsbRegistry)->apreList[__c4OurIndex].szInfoLine;
}


tCIDLib::TVoid TKrnlProcRegistry::SetInfoCard(const tCIDLib::TCard4 c4New)
{
    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);
    (*__pktsbRegistry)->apreList[__c4OurIndex].c4InfoCard = c4New;
}


tCIDLib::TVoid TKrnlProcRegistry::SetInfoLine(const tCIDLib::TZStr64 pszNew)
{
    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);

    TRawStr::CopyStr
    (
        (*__pktsbRegistry)->apreList[__c4OurIndex].szInfoLine
        , pszNew
        , sizeof((*__pktsbRegistry)->apreList[__c4OurIndex].szInfoLine)
    );
}


const tCIDLib::TRawProcessEntry& TKrnlProcRegistry::preThisProcess()
{
    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);
    return (*__pktsbRegistry)->apreList[__c4OurIndex];
}


tCIDLib::TVoid TKrnlProcRegistry::ValidateRegistry()
{
    TKrnlMutexLocker kmtxlReg(__pkmtxAccess, 5 Seconds);

    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4MaxRegProc; c4Index++)
    {
        tCIDLib::TRawProcessEntry* ppreEntry
                                    = &(*__pktsbRegistry)->apreList[c4Index];
        if (!ppreEntry->bUsed)
            continue;

        if (!__bCheckIsRunning(ppreEntry->pidThis))
        {
            ppreEntry->bUsed = kCIDLib::False;
            ppreEntry->pidThis = kCIDLib::pidInvalid;

            if (!(*__pktsbRegistry)->c4ProcCount)
                TKrnlError::ThrowKrnlError(kKrnlErrors::errcProcRegUnderflow);
            (*__pktsbRegistry)->c4ProcCount--;
        }
    }
}
