//
//  FILE NAME: CIDKernel_MutexSemaphore.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/10/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_Semaphore.Cpp module. This module
//  implements the TKrnlMutex class, which is the platform wrapper class
//  for mutex sempahores.
//
//  CAVEATS/GOTCHAS:
//
//  1)  bDuplicate() will duplicate a mutex. This is the only way to
//      open multiple copies of an unnamed mutex. For named mutexes, you
//      can either create another mutex with the same name, or use the
//      duplicate method.
//
//  2)  bOpen() and bCreateOrOpen() are only valid for named mutexes.
//      Unnamed ones can only be created.
//
//  3)  The lock state is a mutable attribute (its not a member anyway) so
//      the lock/unlock methods are const. This is a requirement of thestandard
//      MLockable mixin interface used at higher levels also. On another
//      platform, where the lock state might actually be a member, it would
//      have to be mutable.
//



#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TKrnlMutex
//  PREFIX: kmtx
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlMutex
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKrnlMutex();

        TKrnlMutex
        (
            const   tCIDLib::EShareStates   eShared
        );

        TKrnlMutex
        (
            const   tCIDLib::Tch* const     pszName
        );

        ~TKrnlMutex();


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bNamed() const;

        tCIDLib::TBoolean bValid() const;

        tCIDLib::TVoid Close();

        tCIDLib::TVoid Create
        (
            const   tCIDLib::ELockStates    eInitState
        );

        tCIDLib::TVoid CreateOrOpen
        (
            const   tCIDLib::ELockStates    eInitState
        );

        tCIDLib::EShareStates eShareState() const;

        tCIDLib::TVoid Lock
        (
            const   tCIDLib::TCard4         c4MilliSecs = kCIDLib::c4MaxWait
        )   const;

        tCIDLib::TVoid Duplicate
        (
            const   TKrnlMutex&             kmtxToDup
        );

        tCIDLib::TVoid Open();

        const tCIDLib::Tch* pszName() const;

        const tCIDLib::TVoid SetName
        (
            const   tCIDLib::Tch* const     pszNewName
        );

        tCIDLib::TVoid Unlock() const;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors or operators
        // --------------------------------------------------------------------
        TKrnlMutex(const TKrnlMutex&);

        tCIDLib::TVoid operator=(const TKrnlMutex&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CreateNamed
        (
            const   tCIDLib::ELockStates    eInitState
            , const tCIDLib::TBoolean       bFailIfExists
        );


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __eShareState
        //      Indicates whether this semaphore is shareable. If it has
        //      a name, this will be forced to shareable.
        //
        //  __hmtxThis
        //      This is the handle to the mutex sem. It is 0 when not
        //      valid.
        //
        //  __pszName
        //      The name of the semaphore.
        // --------------------------------------------------------------------
        tCIDLib::EShareStates   __eShareState;
        tCIDLib::TMtxHandle     __hmtxThis;
        tCIDLib::Tch*           __pszName;
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TKrnlMutex::bNamed() const
{
    return (__pszName != 0);
}

inline tCIDLib::TBoolean TKrnlMutex::bValid() const
{
    return (__hmtxThis != kCIDLib::hmtxInvalid);
}

inline tCIDLib::EShareStates TKrnlMutex::eShareState() const
{
    return __eShareState;
}

inline const tCIDLib::Tch* TKrnlMutex::pszName() const
{
    return __pszName;
}
