//
//  FILE NAME: CIDKernel_MutexSemaphore.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/11/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the the mutex wrapper class.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"



// ----------------------------------------------------------------------------
//   CLASS: TKrnlMutex
//  PREFIX: kmtx
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlMutex: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlMutex::TKrnlMutex() :

    __eShareState(tCIDLib::EShareState_Unshared)
    , __hmtxThis(kCIDLib::hmtxInvalid)
    , __pszName(0)
{
}

TKrnlMutex::TKrnlMutex(const tCIDLib::EShareStates eShared) :

    __eShareState(eShared)
    , __hmtxThis(kCIDLib::hmtxInvalid)
    , __pszName(0)
{
}

TKrnlMutex::TKrnlMutex(const tCIDLib::Tch* const pszName) :

    __eShareState(tCIDLib::EShareState_Shared)
    , __hmtxThis(kCIDLib::hmtxInvalid)
    , __pszName(0)
{
    if (pszName)
        __pszName = TRawStr::pszReplicate(pszName);
}

TKrnlMutex::~TKrnlMutex()
{
    if (__pszName)
    {
        delete __pszName;
        __pszName = 0;
    }

    if (__hmtxThis != kCIDLib::hmtxInvalid)
        Close();
}


// ----------------------------------------------------------------------------
//  TKrnlMutex: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlMutex::Close()
{
    if (__hmtxThis != kCIDLib::hmtxInvalid)
    {
        tCIDLib::TMtxHandle hTmp = __hmtxThis;
        __hmtxThis = kCIDLib::hmtxInvalid;

        if (!CloseHandle(hTmp))
            TKrnlError::ThrowKrnlError();
    }
}


tCIDLib::TVoid
TKrnlMutex::Create(const tCIDLib::ELockStates eInitState)
{
    if (__hmtxThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (__pszName)
    {
        __CreateNamed(eInitState, kCIDLib::True);
        return;
    }

    tCIDLib::TMtxHandle hTmp = CreateMutex(0, eInitState, 0);
    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    // We survived, so store the handle
    __hmtxThis = hTmp;
}

tCIDLib::TVoid
TKrnlMutex::CreateOrOpen(const tCIDLib::ELockStates eInitState)
{
    if (__hmtxThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (!__pszName)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNullName);

    __CreateNamed(eInitState, kCIDLib::False);
}


tCIDLib::TVoid TKrnlMutex::Lock(const tCIDLib::TCard4 c4MilliSecs) const
{
    if (__hmtxThis == kCIDLib::hmtxInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    tCIDLib::TCard4 c4Reason = WaitForSingleObject(__hmtxThis, c4MilliSecs);

    if (c4Reason == WAIT_FAILED)
        TKrnlError::ThrowKrnlError();

    if (c4Reason == WAIT_TIMEOUT)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcTimeout);
}


tCIDLib::TVoid TKrnlMutex::Duplicate(const TKrnlMutex& kmtxToDup)
{
    // Clean up the name
    delete __pszName;
    __pszName = 0;

    // Close it if open
    if (__hmtxThis != kCIDLib::hmtxInvalid)
        Close();

    // Copy over the info
    __eShareState = kmtxToDup.__eShareState;

    // Recreate the name
    if (kmtxToDup.__pszName)
        __pszName = TRawStr::pszReplicate(kmtxToDup.__pszName);

    // Duplicate the handle
    tCIDLib::TMtxHandle hTmp;
    if (!DuplicateHandle
    (
        kmtxToDup.__hmtxThis
        , TKrnlSysInfo::hprocThis()
        , TKrnlSysInfo::hprocThis()
        , &hTmp
        , 0
        , 0
        , DUPLICATE_SAME_ACCESS))
    {
        TKrnlError::ThrowKrnlError();
    }

    __hmtxThis = hTmp;
}


tCIDLib::TVoid TKrnlMutex::Open()
{
    if (__hmtxThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (!__pszName)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNullName);

    tCIDLib::TMtxHandle hTmp = OpenMutex
    (
        MUTEX_ALL_ACCESS | SYNCHRONIZE
        , 0
        , __pszName
    );

    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    // We survived, so store the handle
    __hmtxThis = hTmp;
}


const tCIDLib::TVoid
TKrnlMutex::SetName(const tCIDLib::Tch* const pszNewName)
{
    if (__hmtxThis != kCIDLib::hmtxInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    // Delete the old name and replicate the new one
    delete __pszName;
    __pszName = 0;
    __pszName = TRawStr::pszReplicate(pszNewName);
}


tCIDLib::TVoid TKrnlMutex::Unlock() const
{
    if (__hmtxThis == kCIDLib::hmtxInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    if (!ReleaseMutex(__hmtxThis))
        TKrnlError::ThrowKrnlError();
}


// ----------------------------------------------------------------------------
//  TKrnlMutex: Private, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlMutex::__CreateNamed(  const   tCIDLib::ELockStates    eState
                            , const tCIDLib::TBoolean       bFailIfExists)
{
    tCIDLib::TMtxHandle hmtxTmp;

    //
    //  First try to open it. If that succeeds, then see if we need to
    //  fail if it exists. If so, then close it and return an error.
    //
    hmtxTmp = OpenMutex(MUTEX_ALL_ACCESS, FALSE, __pszName);

    if (hmtxTmp)
    {
        if (bFailIfExists)
        {
            CloseHandle(hmtxTmp);
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyExists);
        }
        __hmtxThis = hmtxTmp;
        return;
    }

    // The open did not work, so try to create it
    hmtxTmp = CreateMutex(0, eState, __pszName);

    if (!hmtxTmp)
        TKrnlError::ThrowKrnlError();

    __hmtxThis = hmtxTmp;
}
