//
//  FILE NAME: CIDKernel_Module.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/22/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the the TKrnlModule class, the core wrapper
//  class for module handles.
//
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"
#include    "CIDKernel_MsgFileSupport.Hpp"



// ----------------------------------------------------------------------------
//   CLASS: TKrnlModule
//  PREFIX: kmod
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlModule: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlModule::TKrnlModule() :

    __bViaLoad(kCIDLib::False)
    , __eAdopted(tCIDLib::EAdoptOpt_NoAdopt)
    , __hmodThis(0)
    , __pMsgData(0)
{
}

TKrnlModule::TKrnlModule(   const   tCIDLib::Tch* const pszModName
                            , const tCIDLib::EModTypes  eModType
                            , const tCIDLib::TBoolean   bLoad) :
    __bViaLoad(bLoad)
    , __eAdopted(tCIDLib::EAdoptOpt_Adopt)
    , __hmodThis(0)
    , __pMsgData(0)
{
    // Either load or query according to the flag
    if (bLoad)
        LoadFromName(pszModName, eModType);
    else
        QueryFromName(pszModName, eModType);
}

TKrnlModule::TKrnlModule(   const   tCIDLib::TModHandle hmodToUse
                            , const tCIDLib::EAdoptOpts eAdopt
                            , const tCIDLib::TBoolean   bViaLoad) :
    __bViaLoad(bViaLoad)
    , __eAdopted(eAdopt)
    , __hmodThis(hmodToUse)
    , __pMsgData(0)
{
    __LoadMessages();
}

TKrnlModule::~TKrnlModule()
{
    __ClearHandle();
}


// ----------------------------------------------------------------------------
//  TKrnlModule: Public, static methods
// ----------------------------------------------------------------------------

const tCIDLib::Tch*
TKrnlModule::pszRawLoadCIDMsg( const   tCIDLib::TModHandle hmodSource
                               , const tCIDLib::TMsgId     midToLoad)
{
    //
    //  Lets try to load up our message buffer resource. We just store it
    //  as a big binary blob under a single resource id.
    //
    tCIDLib::TRscHandle hRsc = ::FindResourceExW
    (
        hmodSource
        , MAKEINTRESOURCE(kMsgFileIds::c4ResType)
        , MAKEINTRESOURCE(kMsgFileIds::c4ResId)
        , MAKELANGID(kMsgFileIds::c4Language, kMsgFileIds::c4SubLanguage)
    );

    // There is no message resource, so just return
    if (!hRsc)
        return kMessages::pszMod_NoMsgs;

    // Try to load that puppy
    tCIDLib::THandle hTmp = ::LoadResource(hmodSource, hRsc);
    if (!hTmp)
        return kMessages::pszMod_NoMsgs;

    // And the handle is now our pointer
    const CIDMsgFile::TMsgIndex* pmiModule = (const CIDMsgFile::TMsgIndex*)hTmp;
    try
    {
        // Check specifically for no messages
        if (!pmiModule->c4MsgCount)
            return kMessages::pszMod_NoMsgs;

        for (tCIDLib::TCard4 c4Index = 0; c4Index < pmiModule->c4MsgCount; c4Index++)
        {
            //
            //  See if its id matches our target id. If so, then this is
            //  our guy.
            //
            if (pmiModule->amieList[c4Index].midThis == midToLoad)
            {
                // Get a pointer to the base resource data
                const tCIDLib::Tch* pszRet = (const tCIDLib::Tch*)pmiModule;

                // And bump it up to the offset of the text for this entry
                pszRet = pOffsetPtr<tCIDLib::Tch>
                (
                    pszRet
                    , pmiModule->amieList[c4Index].c4TextOffset
                );

                // Return a pointer to this entry
                return pszRet;
            }
        }
    }

    catch(...)
    {
    }
    return kMessages::pszMod_MsgNotFound;
}

tCIDLib::TVoid
TKrnlModule::RawQueryModName(   const   tCIDLib::TModHandle hmodToQuery
                                ,       tCIDLib::Tch* const pszToFill
                                , const tCIDLib::TCard4     c4MaxChars)
{
    // Init the buffer to empty
    pszToFill[0] = kCIDLib::chNull;

    tCIDLib::Tch szTmpBuf[kCIDLib::c4MaxPathLen+1];

    if (!GetModuleFileName(hmodToQuery, szTmpBuf, kCIDLib::c4MaxPathLen))
        TKrnlError::ThrowKrnlError();

    //
    //  Now we need to whittle it down to just the base name of the file
    //  itself, throwing away the path and the extension.
    //
    tCIDLib::Tch* pszTmp;

    // Find the last period and put a nul there
    pszTmp = TRawStr::pszFindLastChar(szTmpBuf, kCIDLib::chPeriod);
    if (pszTmp)
        *pszTmp = 0;

    // Now find the last separator
    pszTmp = TRawStr::pszFindChar(szTmpBuf, kCIDLib::chPathSeparator);
    if (!pszTmp)
        pszTmp = szTmpBuf;
    TRawStr::CopyStr(pszToFill, pszTmp, c4MaxChars);
}


// ----------------------------------------------------------------------------
//  TKrnlModule: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlModule::AdoptHandle(   const   tCIDLib::TModHandle hModule
                            , const tCIDLib::TBoolean   bViaLoad)
{
    // Clean up current handle if any
    __ClearHandle();

    // And adopt the new handle
    __bViaLoad = bViaLoad;
    __eAdopted = tCIDLib::EAdoptOpt_Adopt;
    __hmodThis = hModule;

    // See if we have a message resource and load it up if so
    __LoadMessages();
}


tCIDLib::TBoolean
TKrnlModule::bLoadCIDMsg(   const   tCIDLib::TMsgId     midToLoad
                            ,       tCIDLib::Tch* const pszTarget
                            , const tCIDLib::TCard4     c4MaxChars) const
{
    tCIDLib::TBoolean bLoaded;
    const tCIDLib::Tch* pszTmp = pszLoadCIDMsg(midToLoad, bLoaded);
    TRawStr::CopyStr(pszTarget, pszTmp, c4MaxChars);
    return bLoaded;
}


tCIDLib::TBoolean
TKrnlModule::bLoadResString(const   tCIDLib::TResId     ridString
                            ,       tCIDLib::Tch* const pszTarget
                            , const tCIDLib::TCard4     c4MaxChars) const
{
    if (!::LoadStringW(__hmodThis, ridString, pszTarget, c4MaxChars+1))
    {
        tCIDLib::TZStr32 szId;
        TRawStr::FormatVal(ridString, szId, c4MaxBufChars(szId));

        // Load up a default string
        TRawStr::CopyCatStr
        (
            pszTarget
            , c4MaxChars
            , kMessages::pszMod_ResNotFound
            , szId
            , L"."
        );
        return kCIDLib::False;
    }
    return kCIDLib::True;
}


tCIDLib::TCard4
TKrnlModule::c4QueryResourceSize(const   tCIDLib::ERscTypes  eRscType
                                 , const tCIDLib::TResId     ridToQuery) const
{
    tCIDLib::TCard4 c4Size;
    if (eRscType == tCIDLib::ERscType_String)
    {
        c4Size = __c4QueryStringResourceSize(ridToQuery);
    }
     else
    {
        // Try to find the resource first
        tCIDLib::TRscHandle hRsc = FindResourceW
        (
            __hmodThis
            , MAKEINTRESOURCE(ridToQuery)
            , MAKEINTRESOURCE(eRscType)
        );

        if (!hRsc)
            TKrnlError::ThrowKrnlError();

        c4Size = SizeofResource(__hmodThis, hRsc);
        if (!c4Size)
            TKrnlError::ThrowKrnlError();
    }
    return c4Size;
}


tCIDLib::TVoid
TKrnlModule::DeleteBitmap(const tCIDLib::TBmpHandle hToDelete) const
{
    if (!::DeleteObject(hToDelete))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid TKrnlModule::FreeResource(tCIDLib::TVoid*) const
{
    // 
    //  This is a no-op it seems on NT because resources are just memory
    //  mapped into the program and were not 'allocated' to begin with.
    //
}


tCIDLib::TBmpHandle
TKrnlModule::LoadBitmap(const tCIDLib::TResId ridBitmap) const
{
    tCIDLib::TBmpHandle hTmp = ::LoadBitmapW
    (
        __hmodThis
        , MAKEINTRESOURCE(ridBitmap));

    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    return hTmp;
}

tCIDLib::TBmpHandle
TKrnlModule::LoadSysBitmap(const tCIDLib::ESysBitmaps eBitmap) const
{
    tCIDLib::TBmpHandle hTmp = ::LoadBitmapW(0, MAKEINTRESOURCE(eBitmap));

    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    return hTmp;
}


tCIDLib::TVoid
TKrnlModule::LoadFromName(  const   tCIDLib::Tch* const pszModName
                            , const tCIDLib::EModTypes  eModType)
{
    // Clear the current handle in the appropriate way
    __ClearHandle();

    //
    //  If the passed type is an Exe facility, then we need to add the Exe
    //  extension under the NT platform or it won't work. The facility names
    //  names that we get here never have extensions.
    //
    //  So we alloc a new buffer big enough for the extension, put a janitor
    //  on it, and copy over the passed name.
    //
    const tCIDLib::TCard4 c4NameChars = TRawStr::c4StrLen(pszModName)+8;
    tCIDLib::Tch* pszName = new tCIDLib::Tch[c4NameChars+1];
    THeapJanitor  janName(pszName);
    TRawStr::CopyStr(pszName, pszModName, c4NameChars);

    // If its an Exe, then tack on the needed extension
    if (eModType == tCIDLib::EModType_Exe)
        TRawStr::CatStr(pszName, kCIDLib::pszExeExtension, c4NameChars);

    // And now try to load the other one
    tCIDLib::TModHandle hTmp = ::LoadLibraryW(pszName);

    // If it failed, return the error
    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    //
    //  It worked, so store the handle. Set the flag that indicates this
    //  handle came from a load operation.
    //
    __bViaLoad = kCIDLib::True;
    __eAdopted = tCIDLib::EAdoptOpt_Adopt;
    __hmodThis = hTmp;

    // See if we have a message resource blob and load it up if so
    __LoadMessages();
}


tCIDLib::TVoid*
TKrnlModule::pLoadResource( const   tCIDLib::ERscTypes  eRscType
                            , const tCIDLib::TResId     ridToLoad) const
{
    // Try to find the resource first
    tCIDLib::TRscHandle hRsc = ::FindResourceW
    (
        __hmodThis
        , MAKEINTRESOURCE(ridToLoad)
        , MAKEINTRESOURCE(eRscType)
    );

    if (!hRsc)
        TKrnlError::ThrowKrnlError();

    // Try to load that puppy
    tCIDLib::THandle hTmp = ::LoadResource(__hmodThis, hRsc);
    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    return (tCIDLib::TVoid*)hTmp;
}


const tCIDLib::Tch*
TKrnlModule::pszLoadCIDMsg( const   tCIDLib::TMsgId     midToLoad
                            ,       tCIDLib::TBoolean&  bLoaded) const
{
    // Assume a failure, since we are pessimists
    if (&bLoaded)
        bLoaded = kCIDLib::False;

    //
    //  If the message data pointer is 0, then there is no messages for this
    //  module so we return a default message. We cannot throw exceptions
    //  here because that would almost certainly kick off a circle jerk.
    //
    if (!__pMsgData)
        return kMessages::pszMod_NoMsgs;

    try
    {
        // Get a pointer to the buffer as a TMsgIndex
        const CIDMsgFile::TMsgIndex* pmiModule =
                                    (const CIDMsgFile::TMsgIndex*)__pMsgData;

        // Check specifically for no messages
        if (!pmiModule->c4MsgCount)
            return kMessages::pszMod_NoMsgs;

        for (tCIDLib::TCard4 c4Index = 0; c4Index < pmiModule->c4MsgCount; c4Index++)
        {
            //
            //  See if its id matches our target id. If so, then this is
            //  our guy.
            //
            if (pmiModule->amieList[c4Index].midThis == midToLoad)
            {
                // Get a pointer to the base resource data
                const tCIDLib::Tch* pszRet = (const tCIDLib::Tch*)__pMsgData;

                // And bump it up to the offset of the text for this entry
                pszRet = pOffsetPtr<tCIDLib::Tch>
                (
                    pszRet
                    , pmiModule->amieList[c4Index].c4TextOffset
                );

                if (&bLoaded)
                    bLoaded = kCIDLib::True;

                // Return a pointer to this entry
                return pszRet;
            }
        }
    }

    catch(...)
    {
    }
    return kMessages::pszMod_MsgNotFound;
}


const tCIDLib::Tch*
TKrnlModule::pszLoadResString(  const tCIDLib::TResId       ridToLoad
                                ,       tCIDLib::TBoolean&  bLoaded) const
{
    //
    //  Try to find the resource first. We convert the string id into a
    //  string table resource id.
    //
    tCIDLib::TRscHandle hRsc = ::FindResourceW
    (
        __hmodThis
        , MAKEINTRESOURCE((ridToLoad / 16UL)+1)
        , MAKEINTRESOURCE(RT_STRING)
    );

    if (!hRsc)
        return kMessages::pszMod_MsgNotFound;

    // Try to load that puppy
    tCIDLib::THandle hTmp = ::LoadResource(__hmodThis, hRsc);
    if (!hTmp)
        return kMessages::pszMod_MsgNotFound;

    //
    //  Now we need to parse out the correct string. Basically the format
    //  is a length value, followed by the text, then a length word,
    //  followed by the text, etc.... Up to 16 times for a full string
    //  table.
    //
    tCIDLib::TCard2* pSize = (tCIDLib::TCard2*)hTmp;

    // Find out which of the strings in this table we want
    tCIDLib::TCard4 c4Tmp = ridToLoad;
    c4Tmp %= 16;

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4Tmp; c4Ind++)
    {
        tCIDLib::TCard2 c2Skip = *pSize;
        pSize += c2Skip + 1;
    }

    // Return the text pointer
    return (const tCIDLib::Tch*)pOffsetPtr(pSize, sizeof(tCIDLib::TCard2));
}


tCIDLib::TVoid
TKrnlModule::QueryFromName( const   tCIDLib::Tch* const pszModName
                            , const tCIDLib::EModTypes  eModType)
{
    // Clear the handle in the appropriate way
    __ClearHandle();

    //
    //  If the passed type is an Exe facility, then we need to add the Exe
    //  extension under the NT platform or it won't work. The facility names
    //  names that we get here never have extensions.
    //
    //  So we alloc a new buffer big enough for the extension, put a janitor
    //  on it, and copy over the passed name.
    //
    const tCIDLib::TCard4 c4NameChars = TRawStr::c4StrLen(pszModName)+8;
    tCIDLib::Tch* pszName = new tCIDLib::Tch[c4NameChars+1];
    THeapJanitor  janName(pszName);
    TRawStr::CopyStr(pszName, pszModName, c4NameChars);

    // If its an Exe, then tack on the needed extension
    if (eModType == tCIDLib::EModType_Exe)
        TRawStr::CatStr(pszName, kCIDLib::pszExeExtension, c4NameChars);

    // And now try to look up the other one
    tCIDLib::TModHandle hTmp = ::GetModuleHandleW(pszName);

    // If it failed, return the error
    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    //
    //  It worked, so store the handle. And set the field that indicates
    //  that this was not a load, just a query.
    //
    __bViaLoad = kCIDLib::False;
    __eAdopted = tCIDLib::EAdoptOpt_Adopt;
    __hmodThis = hTmp;

    // See if we have a message resource blob and load it up if so
    __LoadMessages();
}


// ----------------------------------------------------------------------------
//  TKrnlModule: Private, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlModule::__ClearHandle()
{
    tCIDLib::TBoolean bRet = kCIDLib::True;

    //
    //  If there is a module handle, it was via a load, and we have adopted it
    //  then we need to free it.
    //
    if (__bViaLoad && __hmodThis && __eAdopted)
    {
        if (!::FreeLibrary(__hmodThis))
            TKrnlError::ThrowKrnlError();
    }

    __hmodThis = 0;
    __eAdopted = tCIDLib::EAdoptOpt_NoAdopt;
    __bViaLoad = kCIDLib::False;
}


tCIDLib::TVoid TKrnlModule::__LoadMessages()
{
    //
    //  Lets try to load up our message buffer resource. We just store it
    //  as a big binary blob under a single resource id.
    //
    tCIDLib::TRscHandle hRsc = ::FindResourceExW
    (
        __hmodThis
        , MAKEINTRESOURCE(kMsgFileIds::c4ResType)
        , MAKEINTRESOURCE(kMsgFileIds::c4ResId)
        , MAKELANGID(kMsgFileIds::c4Language, kMsgFileIds::c4SubLanguage)
    );

    // There is no message resource, so just return
    if (!hRsc)
        return;

    // Try to load that puppy
    tCIDLib::THandle hTmp = ::LoadResource(__hmodThis, hRsc);
    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    // And the handle is now our pointer
    __pMsgData = hTmp;
}


tCIDLib::TCard4
TKrnlModule::__c4QueryStringResourceSize(const tCIDLib::TResId ridToQuery) const
{
    //
    //  Try to find the resource first. We convert the string id into a
    //  string table resource id.
    //
    tCIDLib::TRscHandle hRsc = ::FindResourceW
    (
        __hmodThis
        , MAKEINTRESOURCE((ridToQuery / 16UL)+1)
        , MAKEINTRESOURCE(RT_STRING)
    );

    if (!hRsc)
        TKrnlError::ThrowKrnlError();

    // Try to load that puppy
    tCIDLib::THandle hTmp = ::LoadResource(__hmodThis, hRsc);
    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    //
    //  Now we need to parse out the correct string. Basically the format
    //  is a length value, followed by the text, then a length byte,
    //  followed by the text, etc.... Up to 16 times for a full string
    //  table.
    //
    tCIDLib::TCard2* pSize = (tCIDLib::TCard2*)hTmp;

    // Find out which of the strings in this table we want
    tCIDLib::TCard4 c4Tmp = ridToQuery;
    c4Tmp %= 16;

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4Tmp; c4Ind++)
    {
        tCIDLib::TCard2 c2Skip = *pSize;
        pSize += c2Skip + 1;
    }

    //
    //  Return the length. We left the pointer at the length word of the
    //  string of interest, so we just have to dereference it.
    //
    return (tCIDLib::TCard4)*pSize;
}
