//
// NAME: CIDKernel_Metrics.Hpp
// 
// DESCRIPTION: 
// 
//  This is the header for the CIDKernel_Metrics.Cpp module. This module
//  implements the host specific aspects of the metrics system, hiding its
//  actual details from the higher level facilities.
//
//  TKrnlMetricSystem is the main access to the metrics system for a process.
//  This allows the process to register metrics and set its process state.
//  This is only for a process to access its own metrics though.
//
//  To access another process' metrics, the TKrnlMetricDirectory class
//  provides access to another process' metrics information. It is given a
//  process id and it manages the access.
//
// 
// AUTHOR: Dean Roddey
// 
// CREATE DATE: 01/11/94
// 
// COPYRIGHT: 1992..1997, 'CIDCorp
// 

#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  Metrics oriented constant values
// -----------------------------------------------------------------------------
namespace kCIDLib
{
    const tCIDLib::TCard4   c4MaxMetricsPerGrp      = 16;
};


namespace tCIDLib
{
    // -------------------------------------------------------------------------
    //  This structure defines a single user metric.
    //
    //  If any pointers are added to this structure, then the copying of
    //  incoming metrics during group registration will have to be modified!
    //
    //  szMetricName
    //      This is the name of the metric. It is used for display to the user
    //      in the metrics viewer and in reports.
    //
    //  szMetricHelp
    //      This is a help string about what the metric is measuring. It is
    //      displayed to the user in the viewer.
    //
    //  eType
    //      This is the type of this metric. It determines what APIs can be
    //      used to access it and which of the union members is used for
    //      storage.
    //
    //  bToggle
    //  c4CValue
    //  i4IValue
    //  f4FValue
    //      These are the storage for the metric value. They are in a union
    //      so that they all overlap. The eType member indicates which one
    //      to use.
    // -------------------------------------------------------------------------
    struct TRawMetric
    {
        tCIDLib::TZStr32        szMetricName;
        tCIDLib::TZStr64        szMetricHelp;
        tCIDLib::EMetricTypes   eType;

        union
        {
            tCIDLib::TBoolean   bToggle;
            tCIDLib::TCard4     c4CValue;
            tCIDLib::TInt4      i4IValue;
            tCIDLib::TFloat4    f4FValue;
        };
    };


    // ------------------------------------------------------------------------
    //  This a metrics group. There can be up to c4MaxMetricsPerGrp Metrics
    //  in a single group.
    //
    //  szGroupName
    //      This is the name of the group. It is used to identify the group
    //      in many places, such as metrics displays and reports.
    //
    //  szGroupHelp
    //      This is a help string that is displayed by the viewer when the
    //      user wants help about what the metric is measuring.
    //
    //  c4MetricCount
    //      This is the count of TMetric structures used in the aMetrics
    //      array.
    //
    //  amtrMetrics
    //      This is an array of TRawMetric structures. c4MetricCount
    //      indicates how many are in use. They are always compacted because
    //      they can never be removed while the program runs.
    // ------------------------------------------------------------------------
    struct TRawMetricGroup
    {
        tCIDLib::TZStr32        szGroupName;
        tCIDLib::TZStr64        szGroupHelp;

        tCIDLib::TCard4         c4MetricCount;
        TRawMetric              amtrMetrics[kCIDLib::c4MaxMetricsPerGrp];
    };
};


namespace kCIDLib
{
    // ------------------------------------------------------------------------
    //  Define the maximum number of metric groups to the amount we can fit
    //  into the defined number of pages. We have to allocate in 4K pages,
    //  so might as well use it all! Subtract a little for the fields before
    //  the array of groups before we divide. 
    // ------------------------------------------------------------------------
    const tCIDLib::TCard4   c4MetricMemoryPages = 4;
    const tCIDLib::TCard4   c4MaxMetricGroups 
                    = (((kCIDLib::c4MemPageSize*kCIDLib::c4MetricMemoryPages)-64)
                                         / sizeof(tCIDLib::TRawMetricGroup));
};


namespace tCIDLib
{
    // ------------------------------------------------------------------------
    //  This is the format of the main process shared memory area. This is
    //  where the viewer looks to find out what groups are registered by the
    //  process. A list of these is in the metric process directory, defined
    //  below.
    //
    //	szMagicVal
    //		This is a magic value placed into the memory to allow the metrics
    //		viewer to be sure it is looking at initialized, valid data. It is
    //		a string "Dean Roddey" that should be random enough.
    //
    //  eState
    //      This is the state of the process which the process can set to
    //      indicate its basic state.
    //
    //  c4GroupCount
    //      This is the number of groups actually in use in agrpList. Groups
    //      cannot be deregistered, so the list always stays compacted.
    //
    //  akmtrgList
    //      An array of metrics groups. c4GroupCount indicates how many
    //      entries are in use.
    // ------------------------------------------------------------------------
    struct TRawMetricDirectory
    {
        tCIDLib::Tch            szMagicVal[12];
        tCIDLib::EProcStates    eState;

        tCIDLib::TCard4         c4GroupCount;
        TRawMetricGroup         amtrgList[kCIDLib::c4MaxMetricGroups];
    };
};


#if !defined(CIDKRNL_NOCLASSES)

// ----------------------------------------------------------------------------
//  TMetricsSharedBuf
//      This is a typedef of a typed shared buffer, for the metrics directory
//      type.
// ----------------------------------------------------------------------------
typedef TKrnlTypedSharedBuf<tCIDLib::TRawMetricDirectory>   TMetricsSharedBuf;


// ----------------------------------------------------------------------------
//   CLASS: TKrnlMetricSystem
//  PREFIX: kmsys
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlMetricSystem
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKrnlMetricSystem();

        ~TKrnlMetricSystem();


        // --------------------------------------------------------------------
        //  Public, static methods
        // --------------------------------------------------------------------
        static tCIDLib::TRawMetricGroup* pmtrgRegisterGroup
        (
            const   tCIDLib::Tch* const     pszGroupName
            , const tCIDLib::Tch* const     pszGroupHelp
            , const tCIDLib::TRawMetric*    amtrList
            , const tCIDLib::TCard4         c4Count
        );

        static tCIDLib::TVoid SetProcessState
        (
            const   tCIDLib::EProcStates    eNewState
        );


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TKrnlMetricSystem(const TKrnlMetricSystem&);

        tCIDLib::TVoid operator=(const TKrnlMetricSystem&);
};


// ----------------------------------------------------------------------------
//   CLASS: TKrnlMetricDirectory
//  PREFIX: kmtr
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlMetricDirectory
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKrnlMetricDirectory
        (
            const   tCIDLib::TProcessId     pidToMonitor
        );

        ~TKrnlMetricDirectory();

        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid CopyDirectory
        (
                    tCIDLib::TRawMetricDirectory& mtrToFill
        )   const;

        tCIDLib::TVoid CopyGroup
        (
                    tCIDLib::TRawMetricGroup& mtrgToFill
            , const tCIDLib::TCard4         c4Index
        )   const;

        tCIDLib::TVoid CopyMetric
        (
                    tCIDLib::TRawMetric&    mtrToFill
            , const tCIDLib::TCard4         c4GroupIndex
            , const tCIDLib::TCard4         c4MetricIndex
        )   const;

        tCIDLib::EProcStates eState() const;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TKrnlMetricDirectory(const TKrnlMetricDirectory&);

        tCIDLib::TVoid operator=(const TKrnlMetricDirectory&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pidOurProc
        //      This is the process id of our process. It is used to find
        //      the name of the metrics directory and its mutex.
        //
        //  __pkmtxDirectory
        //      This is our opened copy of the metrics directory mutex. It
        //      is used to synchronize access to the directory.
        //
        //  __pktsbThis
        //      This is our pointer to the metrics directory for the process
        //      we are monitoring.
        // --------------------------------------------------------------------
        TKrnlMutex*             __pkmtxDirectory;
        tCIDLib::TProcessId     __pidOurProc;
        TMetricsSharedBuf*      __pktsbThis;
};

#endif

#pragma pack(pop)
