//
// NAME: CIDKernel_Metrics.Cpp
// 
// DESCRIPTION: 
// 
//  This module contains the metrics support for low level metrics viewer
//  programs and for the CIDLib facility, which provides the standard client
//  access to the metrics, via wrapper classes that use this classes.
//
// 
// AUTHOR: Dean Roddey
// 
// CREATE DATE: 01/11/94
// 
// COPYRIGHT: 1992..1997, 'CIDCorp
// 
// CAVEATS/GOTCHAS: 
// 


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//  Local data
//
//  __pktsbDirectory
//      This is the shared buffer to the process metrics directory. It is
//      implemented via the typed shared buffer template class. We have
//      to use a pointer here, because of order of init issues.
//
//  __pkmtxMetrics
//      This is the named semaphore that protects the metrics memory
//      area for this process and any viewer app that is accessing the
//      metrics.
// ----------------------------------------------------------------------------
static TMetricsSharedBuf*   __pktsbDirectory;
static TKrnlMutex*          __pkmtxMetrics;


// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid _InitTermMetrics(const   tCIDLib::EInitTerm      eInitTerm
                                , const tCIDLib::EGlobalStates  eGlobals
                                , const tCIDLib::TModHandle     hmodThis
                                , const tCIDLib::TCard4         c4MaxChars
                                ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszReason = kMessages::pszGen_Unknown;

    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            try
            {
                //
                //  Build up the name of the shared mutex and memory, which is
                //  unique per-process. So we pass our process id to the
                //  resource name object.
                //
                pszReason = kMessages::pszMtr_CreateResName;
                TKrnlRscName krsnMetricsRsc
                (
                    L"CIDCorp"
                    , L"Metrics"
                    , L"Directory"
                    , TKrnlSysInfo::pidThis()
                );

                //
                //  Construct the named semaphore. Then create it, or open it
                //  if it already exists. Create it locked
                //
                tCIDLib::Tch szTmpName[1024];
                krsnMetricsRsc.BuildFullName
                (
                    szTmpName
                    , c4MaxBufChars(szTmpName)
                    , tCIDLib::ENamedRsc_Mutex
                );
                pszReason = kMessages::pszMtr_CreateMtx;
                __pkmtxMetrics = new TKrnlMutex(szTmpName);
                __pkmtxMetrics->CreateOrOpen(tCIDLib::ELockState_Unlocked);

                //
                //  Now lock the mutex while we see if we need to be the one
                //  that creates and sets up the main metrics directory memory.
                //
                pszReason = kMessages::pszMtr_Locking;
                TKrnlMutexLocker kmtxlLock(__pkmtxMetrics);

                tCIDLib::TBoolean bCreated;
                pszReason = kMessages::pszMtr_OpenMem;

                // Construct the memory name
                krsnMetricsRsc.BuildFullName
                (
                    szTmpName
                    , c4MaxBufChars(szTmpName)
                    , tCIDLib::ENamedRsc_Memory
                );

                // Create the shared memory buffer
                __pktsbDirectory = new TMetricsSharedBuf
                (
                    szTmpName
                    , tCIDLib::EMemAcc_ReadWrite
                    , tCIDLib::ECreateAct_OpenOrCreate
                    , bCreated
                );

                // If we created it, then we have to set it up
                if (bCreated)
                {
                    __pktsbDirectory->Zero();

                    TRawStr::CopyStr
                    (
                        (*__pktsbDirectory)->szMagicVal
                        , L"Dean Roddey"
                        , c4MaxBufChars((*__pktsbDirectory)->szMagicVal)
                    );
                    (*__pktsbDirectory)->eState = tCIDLib::EProcState_Initializing;
                }
            }

            catch(...)
            {
                // If we got the memory open, then close it
                if (__pktsbDirectory)
                {
                    //
                    //  Don't let any exception propogate from this cleanup,
                    //  since that would override the original one.
                    //
                    try
                    {
                        delete __pktsbDirectory;
                    }

                    catch(const TKrnlError& errToCatch)
                    {
                        _ERRPOPUP_(kMessages::pszMtr_FreeMem, errToCatch.errcId());
                    }
                    __pktsbDirectory = 0;
                };
                throw;
            }
        }
         else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
              &&  (eGlobals == tCIDLib::EGlobalState_After))
        {
            try
            {
                //
                //  Update the process state in the metrics. It might seem
                //  wierd to do this when we are about to delete it. But a
                //  viewer object might have it open, so our delete of it does
                //  not necessarily make it go away from the system.
                //
                if (__pkmtxMetrics && __pktsbDirectory)
                {
                    TKrnlMutexLocker kmtxlLock(__pkmtxMetrics);
                    (*__pktsbDirectory)->eState = tCIDLib::EProcState_Dead;
                }

                // If we created the mutex, then free it
                pszReason = kMessages::pszMtr_DeleteMtx;
                if (__pkmtxMetrics)
                    delete __pkmtxMetrics;

                // If we allocated the shared memory, then free it
                pszReason = kMessages::pszMtr_DelteMem;
                if (__pktsbDirectory)
                {
                    delete __pktsbDirectory;
                    __pktsbDirectory = 0;
                }
            }

            catch(...)
            {
                // Make sure it gets zeroed out
                __pktsbDirectory = 0;
                throw;
            }
        }
    }

    catch(...)
    {
        TRawStr::CopyStr
        (
            pszFailReason
            , pszReason
            , c4MaxChars
        );
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TKrnlMetricSystem
//  PREFIX: kmsys
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlMetricSystem: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlMetricSystem::TKrnlMetricSystem()
{
}

TKrnlMetricSystem::~TKrnlMetricSystem()
{
}


// ----------------------------------------------------------------------------
//  TKrnlMetricSystem: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TRawMetricGroup* TKrnlMetricSystem::pmtrgRegisterGroup
            (
                const   tCIDLib::Tch* const     pszGroupName
                , const tCIDLib::Tch* const     pszGroupHelp
                , const tCIDLib::TRawMetric*    amtrList
                , const tCIDLib::TCard4         c4Count)
{
    // Sanity check the count
    if (!c4Count || (c4Count > kCIDLib::c4MaxMetricsPerGrp))
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcBadMetricCount);

    TKrnlMutexLocker kmtxlLock(__pkmtxMetrics);

    // See if we have room for another metric group in this process.
    if ((*__pktsbDirectory)->c4GroupCount >= kCIDLib::c4MaxMetricGroups)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcMetricDirFull);

    // Ok, we have room so get our index and bump up the count
    const tCIDLib::TCard4 c4OurIndex = (*__pktsbDirectory)->c4GroupCount++;

    //
    //  Get a pointer to our entry for convenience and fill it in with
    //  the passed info.
    //
    tCIDLib::TRawMetricGroup* pOurGroup =
                                &(*__pktsbDirectory)->amtrgList[c4OurIndex];

    // Copy over the group name, clipping it if necessary
    TRawStr::CopyStr
    (
        pOurGroup->szGroupName
        , pszGroupName
        , c4MaxBufChars(pOurGroup->szGroupName)
    );

    TRawStr::CopyStr
    (
        pOurGroup->szGroupHelp
        , pszGroupHelp
        , c4MaxBufChars(pOurGroup->szGroupHelp)
    );

    //
    //  Now copy over the metric count and the metrics themselves. There
    //  are no pointers in the structures so we can just copy them
    //  as is.
    //
    pOurGroup->c4MetricCount = c4Count;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        pOurGroup->amtrMetrics[c4Index] = amtrList[c4Index];
    }

    return pOurGroup;
}


tCIDLib::TVoid
TKrnlMetricSystem::SetProcessState(const tCIDLib::EProcStates eNewState)
{
    // Lock the access mutex update the field and unlock again
    TKrnlMutexLocker kmtxlLock(__pkmtxMetrics);
    (*__pktsbDirectory)->eState = eNewState;
}




// ----------------------------------------------------------------------------
//   CLASS: TKrnlMetricDirectory
//  PREFIX: kmsys
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlMetricDirectory: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlMetricDirectory::TKrnlMetricDirectory(const tCIDLib::TProcessId pidToMonitor) :

    __pkmtxDirectory(0)
    , __pidOurProc(pidToMonitor)
    , __pktsbThis(0)
{
    //
    //  Build up the name of the shared mutex and memory, which is unique
    //  per-process. So we pass our monitor process id to the resource name
    //  object.
    //
    TKrnlRscName krsnMetricsRsc
    (
        L"CIDCorp"
        , L"Metrics"
        , L"Directory"
        , pidToMonitor
    );

    try
    {
        tCIDLib::Tch szTmpName[1024];

        // Build the full name to the mutex
        krsnMetricsRsc.BuildFullName
        (
            szTmpName
            , c4MaxBufChars(szTmpName)
            , tCIDLib::ENamedRsc_Mutex
        );

        // Create the mutex object and try to open it
        __pkmtxDirectory = new TKrnlMutex(szTmpName);
        __pkmtxDirectory->Open();

        // Build the full name to the memory
        krsnMetricsRsc.BuildFullName
        (
            szTmpName
            , c4MaxBufChars(szTmpName)
            , tCIDLib::ENamedRsc_Memory
        );

        //
        //  Try to now open the shared memory for this process' metrics
        //  directory.
        //
        __pktsbThis = new TMetricsSharedBuf
        (
            szTmpName
            , tCIDLib::EMemAcc_ReadOnly
            , tCIDLib::ECreateAct_OpenIfExists
        );
    }

    catch(...)
    {
        if (__pkmtxDirectory)
        {
            delete __pkmtxDirectory;
            __pkmtxDirectory = 0;
        }

        if (__pktsbThis)
        {
            delete __pktsbThis;
            __pktsbThis = 0;
        }
        throw;
    }
}

TKrnlMetricDirectory::~TKrnlMetricDirectory()
{
    try
    {
        // Undo any stuff that we go ahold of
        if (__pkmtxDirectory)
        {
            delete __pkmtxDirectory;
            __pkmtxDirectory = 0;
        }

        if (__pktsbThis)
        {
            delete __pktsbThis;
            __pktsbThis = 0;
        }
    }

    // Don't allow propogation out of the destructor
    catch(...)
    {
        #if CID_DEBUG_ON
        _MSGPOPUP_(kMessages::pszMtr_DtorError)
        #endif
    }
}


// ----------------------------------------------------------------------------
//  TKrnlMetricDirectory: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlMetricDirectory::CopyDirectory(tCIDLib::TRawMetricDirectory& mtrToFill) const
{
    // Make sure we opened the directory ok
    if (!__pktsbThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    // Lock the metrics directory mutex
    TKrnlMutexLocker kmtxlDir(__pkmtxDirectory);

    // Copy out the whole directory
    mtrToFill = (*__pktsbThis)();
}

tCIDLib::TVoid
TKrnlMetricDirectory::CopyGroup(        tCIDLib::TRawMetricGroup&   mtrgToFill
                                , const tCIDLib::TCard4             c4GroupIndex) const
{
    // Make sure we opened the directory ok
    if (!__pktsbThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    // Lock the metrics directory mutex
    TKrnlMutexLocker kmtxlDir(__pkmtxDirectory);

    // Check the index against the directory's group count
    if (c4GroupIndex >= (*__pktsbThis)->c4GroupCount)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidParameter);

    // Copy out the group
    mtrgToFill = (*__pktsbThis)->amtrgList[c4GroupIndex];
}

tCIDLib::TVoid
TKrnlMetricDirectory::CopyMetric(       tCIDLib::TRawMetric& mtrToFill
                                , const tCIDLib::TCard4      c4GroupIndex
                                , const tCIDLib::TCard4      c4MetricIndex) const
{
    // Make sure we opened the directory ok
    if (!__pktsbThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    // Lock the metrics directory mutex
    TKrnlMutexLocker kmtxlDir(__pkmtxDirectory);

    // Check the index against the directory's group count
    if (c4GroupIndex >= (*__pktsbThis)->c4GroupCount)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidParameter);

    // Check the metric index against the group's metric count
    if (c4MetricIndex >= (*__pktsbThis)->amtrgList[c4GroupIndex].c4MetricCount)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidParameter);

    // Copy out the metric
    mtrToFill = (*__pktsbThis)->amtrgList[c4GroupIndex].amtrMetrics[c4MetricIndex];
}

tCIDLib::EProcStates TKrnlMetricDirectory::eState() const
{
    // Lock the metrics directory mutex
    TKrnlMutexLocker kmtxlDir(__pkmtxDirectory);
    return (*__pktsbThis)->eState;
}
