//
//  FILE NAME: CIDKernel_Locale.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/06/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements TKrnlLocale namespace's functions.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"
#include    <locale.h>
#include    <limits.h>



// ----------------------------------------------------------------------------
//  Local, static data
//
//  __lociCurrent
//      The information on the currently set locale info. Its updated when
//      the SetLocale function is called.
// ----------------------------------------------------------------------------
static TKrnlLocale::TLocaleInfo __lociCurrent;



// ----------------------------------------------------------------------------
//  Intra-facility functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid
_InitTermLocale(const   tCIDLib::EInitTerm      eInitTerm
                , const tCIDLib::EGlobalStates  eGlobals
                , const tCIDLib::TModHandle     hmodThis
                , const tCIDLib::TCard4         c4MaxChars
                ,       tCIDLib::Tch* const     pszFailReason)
{
    // We only have pre-constructor init
    const tCIDLib::Tch* pszPhase = kMessages::pszGen_Unknown;

    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //
            //  Check the environment and see if the user has told us a locale.
            //  If not, then use the default of US English. If so, then use it.
            //
            pszPhase = kMessages::pszLoc_FindDefLoc;
            tCIDLib::Tch szBuffer[512];
            if (!TKrnlEnvironment::bFind
            (
                L"CIDLOCALE"
                , szBuffer
                , c4MaxBufChars(szBuffer)))
            {
                TRawStr::CopyStr
                (
                    szBuffer
                    , kMessages::pszLoc_DefLocale
                    , c4MaxBufChars(szBuffer)
                );
            }

            // Set the locale
            TKrnlLocale::SetLocale(szBuffer);
        }
    }

    catch(...)
    {
        TRawStr::CopyStr
        (
            pszFailReason
            , pszPhase
            , c4MaxChars
        );
        throw;
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TKrnlLocale
//  PREFIX: kloc
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlLocale: Public methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlLocale::QueryLocaleInfo(TLocaleInfo& lociToFill)
{
    // Copy over our local cached locale info
    lociToFill = __lociCurrent;
}


tCIDLib::TVoid TKrnlLocale::SetLocale(const tCIDLib::Tch* const pszLocaleName)
{
    // Set the passed locale info
    _wsetlocale(LC_ALL, pszLocaleName);

    //
    //  Get a temp pointer to the local information and update our cached
    //  copy of the information.
    //
    struct lconv* pLocInfo = localeconv();

    // ------------------------------------------
    //  Do non-monetary values
    // ------------------------------------------
    __lociCurrent.c4GroupSize = tCIDLib::TCard4(*pLocInfo->grouping);

    __lociCurrent.chThousandsSep = TRawStr::chConvert(*pLocInfo->thousands_sep);
    __lociCurrent.chPosSign = TRawStr::chConvert(*pLocInfo->positive_sign);
    __lociCurrent.chNegSign = TRawStr::chConvert(*pLocInfo->negative_sign);
    __lociCurrent.chDecimalSym = TRawStr::chConvert(*pLocInfo->decimal_point);

    // ------------------------------------------
    // And now do monetary values
    // ------------------------------------------
    __lociCurrent.bNegSpacedSym = (pLocInfo->n_sep_by_space == 1);
    __lociCurrent.bNegSymPrecedes = (pLocInfo->n_cs_precedes == 1);
    __lociCurrent.bPosSpacedSym = (pLocInfo->p_sep_by_space == 1);
    __lociCurrent.bPosSymPrecedes = (pLocInfo->p_cs_precedes == 1);
    __lociCurrent.c4MonetaryDecDigits = tCIDLib::TCard4(pLocInfo->frac_digits);
    __lociCurrent.c4MonetaryGrpSize = tCIDLib::TCard4(*pLocInfo->mon_grouping);

    if (*pLocInfo->mon_decimal_point)
        __lociCurrent.chMonetaryDecimalSym = TRawStr::chConvert(*pLocInfo->mon_decimal_point);
    else
        __lociCurrent.chMonetaryDecimalSym = kCIDLib::chNull;

    if (*pLocInfo->mon_thousands_sep)
        __lociCurrent.chMonetaryGrpSep = TRawStr::chConvert(*pLocInfo->mon_thousands_sep);
    else
        __lociCurrent.chMonetaryGrpSep = kCIDLib::chNull;

    if (*pLocInfo->currency_symbol)
        __lociCurrent.chCurrencySymbol = TRawStr::chConvert(*pLocInfo->currency_symbol);
    else
        __lociCurrent.chCurrencySymbol = kCIDLib::chNull;

    // These are cast to an enum, so they have to line up correctly!
    __lociCurrent.eNegSignPosition = tCIDLib::ESignPositions(pLocInfo->n_sign_posn);
    __lociCurrent.ePosSignPosition = tCIDLib::ESignPositions(pLocInfo->p_sign_posn);
}
