//
// NAME: CIDKernel_HashMap.Cpp
//
// DESCRIPTION:
//
//	This module implements the TKrnlHashMap class, which is a very simple
//  hashed collection for some use internally within the kernel, or by
//  programs which are written to the kernel layer.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/13/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//   CLASS: TKrnlHashMap
//  PREFIX: khshm
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlHashMap: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlHashMap::TKrnlHashMap(const tCIDLib::TCard4 c4Modulus) :

    __apnodeTable(0)
    , __c4CurList(0)
    , __c4ElemCount(0)
    , __c4Modulus(c4Modulus)
    , __pnodeCur(0)
{
    // Allocate the node list
    __apnodeTable = new TNode*[c4Modulus];

    // And zero them out
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Modulus; c4Index++)
        __apnodeTable[c4Index] = 0;
}

TKrnlHashMap::~TKrnlHashMap()
{
    // Flush the list
    Flush();

    // And delete the table itself
    delete [] __apnodeTable;
}


// ----------------------------------------------------------------------------
//  TKrnlHashMap: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlHashMap::Add(  const   tCIDLib::Tch* const pszKey
                    , const tCIDLib::Tch* const pszValue)
{
    // See if its already in the list
    tCIDLib::THashVal   hshKey;
    TNode* pnodeNew = __pnodeFind(pszKey, hshKey);
    if (pnodeNew)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyExists);

    // Create a new node for this guy
    pnodeNew = new TNode;

    // Replicate the passed strings into our node
    pnodeNew->pszKey = TRawStr::pszReplicate(pszKey);
    pnodeNew->pszValue = TRawStr::pszReplicate(pszValue);

    //
    //  Put it into the list. We put it at the head of the list it
    //  hashes to, since this is fast.
    //
    pnodeNew->pnodeNext = __apnodeTable[hshKey];
    __apnodeTable[hshKey] = pnodeNew;

    // Bump the element count
    __c4ElemCount++;
}


tCIDLib::TBoolean TKrnlHashMap::bNext()
{
    if (!__pnodeCur)
        return kCIDLib::False;

    //
    //  See if there is another element in the current list. If so, just
    //  move up to the next node.
    //
    __pnodeCur = __pnodeCur->pnodeNext;
    if (__pnodeCur)
        return kCIDLib::True;

    //
    //  Look for another non-empty list after the current one. if we cannot
    //  find one, then give up.
    //
    __c4CurList++;
    for (; __c4CurList < __c4Modulus; __c4CurList++)
    {
        // If nothing in this list, move on
        if (__apnodeTable[__c4CurList])
            break;
    }

    if (__c4CurList == __c4Modulus)
        return kCIDLib::False;

    // We found one, so point at its first element
    __pnodeCur = __apnodeTable[__c4CurList];

    return kCIDLib::True;
}


tCIDLib::TBoolean TKrnlHashMap::bResetIter()
{
    //
    //  Search for the first non-empty list slot. If there is not one,
    //  just return kCIDLib::False.
    //
    __pnodeCur = 0;
    __c4CurList = 0;
    for (; __c4CurList < __c4Modulus; __c4CurList++)
    {
        // If nothing in this list, move on
        if (__apnodeTable[__c4CurList])
            break;
    }

    if (__c4CurList == __c4Modulus)
        return kCIDLib::False;

    // We found one, so point current node at first element in that list
    __pnodeCur = __apnodeTable[__c4CurList];

    return kCIDLib::True;
}


tCIDLib::TVoid TKrnlHashMap::Flush()
{
    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4Modulus; c4Index++)
    {
        // If nothing in this list, move on
        if (!__apnodeTable[c4Index])
            continue;

        TNode* pnodeCur = __apnodeTable[c4Index];
        while (pnodeCur)
        {
            TNode* pnodeNext = pnodeCur->pnodeNext;

            // Delete the strings and then the node itself
            delete pnodeCur->pszKey;
            delete pnodeCur->pszValue;
            delete pnodeCur;

            pnodeCur = pnodeNext;
        }

        // Zero out this list header
        __apnodeTable[c4Index] = 0;
    }

    // Zero out the element count
    __c4ElemCount = 0;
}


const tCIDLib::Tch* TKrnlHashMap::pszCurKey() const
{
    // Make sure that the iterator is valid
    if (!__pnodeCur)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidIterator);

    return __pnodeCur->pszKey;
}


const tCIDLib::Tch* TKrnlHashMap::pszCurValue() const
{
    // Make sure that the iterator is valid
    if (!__pnodeCur)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidIterator);

    return __pnodeCur->pszValue;
}


const tCIDLib::Tch*
TKrnlHashMap::pszValueForKey(const tCIDLib::Tch* const pszKey)
{
    // See if the node is here
    tCIDLib::THashVal hshKey;
    TNode* pnodeFind = __pnodeFind(pszKey, hshKey);

    if (!pnodeFind)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotFound);

    return pnodeFind->pszValue;    
}


tCIDLib::TVoid TKrnlHashMap::RemoveKey(const tCIDLib::Tch* const pszKey)
{
    // Hash the passed key
    tCIDLib::THashVal hshKey = TRawStr::hshHashStr(pszKey, __c4Modulus);

    //
    //  We need to go through the list that this key is in and find this
    //  node and its previous node.
    //
    TNode* pnodePrev = 0;
    TNode* pnodeFind = __apnodeTable[hshKey];

    while (pnodeFind)
    {
        // If this key matches, then break out
        if (!TRawStr::eCompareStr(pnodeFind->pszKey, pszKey))
            break;

        // Save the current as previous and move up
        pnodePrev = pnodeFind;
        pnodeFind = pnodeFind->pnodeNext;
    }

    if (!pnodeFind)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotFound);

    // If we are deleting the iterator node, move it up
    if (__pnodeCur == pnodeFind)
        bNext();

    //
    //  If the previous node is 0, then this is the head node, else we have
    //  to point the previous node out the found node's next node.
    //
    if (!pnodePrev)
        __apnodeTable[hshKey] = pnodeFind->pnodeNext;
    else
        pnodePrev->pnodeNext = pnodeFind->pnodeNext;

    // Now clean up the found node
    delete pnodeFind->pszKey;
    delete pnodeFind->pszValue;
    delete pnodeFind;

    // Bump down the element count, checking for underflow
    if (!__c4ElemCount)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcElemCountUnderflow);

    __c4ElemCount--;
}


tCIDLib::TVoid
TKrnlHashMap::UpdateKey(const   tCIDLib::Tch* const pszKey
                        , const tCIDLib::Tch* const pszNewValue)
{
    // See if the node is here
    tCIDLib::THashVal hshKey;
    TNode* pnodeFind = __pnodeFind(pszKey, hshKey);

    if (!pnodeFind)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotFound);
 
    // Delete the current value field
    delete pnodeFind->pszValue;
    pnodeFind->pszValue = TRawStr::pszReplicate(pszNewValue);
}



// ----------------------------------------------------------------------------
//  TKrnlHashMap: Private, non-virtual methods
// ----------------------------------------------------------------------------

TKrnlHashMap::TNode* 
TKrnlHashMap::__pnodeFind(  const  tCIDLib::Tch* const pszKeyToFind
                            ,       tCIDLib::THashVal&  hshElem) const
{
    // Get the hash of the element
    hshElem = TRawStr::hshHashStr(pszKeyToFind, __c4Modulus);

    // If this list is empty, then obviously not here
    if (!__apnodeTable[hshElem])
        return 0;

    // The list is not empty so check the list
    TNode* pnodeCur = __apnodeTable[hshElem];
    while (pnodeCur)
    {
        if (!TRawStr::eCompareStr(pnodeCur->pszKey, pszKeyToFind))
            return pnodeCur;
        pnodeCur = pnodeCur->pnodeNext;
    }
    return 0;
}
