//
//  FILE NAME: CIDKernel_FileSystem.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/19/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_FileSystem.Cpp module. This module
//  implements the TKrnlFileSys class, which provides access to file system
//  services, i.e. the stuff not directly related to file content access
//  such as directory searching, file deleting, setting of file attributes,
//  etc...
//
//  Also implemented here is the TKrnlDirSearch class, which is used to
//  abstract the mechanisms needed to maintain context across a directory
//  search.
//
//
//  CAVEATS/GOTCHAS:
//
//  1)  bFindFirst() and bFindNext() return a boolean to indicate whether
//      a match was found. The only reason False is returned is if the file
//      is not found. Any other error is thrown as an exception.
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TKrnlFileSys
//  PREFIX: kfsys
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlFileSys
{
    public  :
        // --------------------------------------------------------------------
        //  The structure returned by errcFindFirst() and errcFindNext().
        // --------------------------------------------------------------------
        struct TRawFileFind
        {
            tCIDLib::TInt8          i8CreationTime;
            tCIDLib::TInt8          i8LastAccessTime;
            tCIDLib::TInt8          i8LastWriteTime;
            tCIDLib::TFilePos       fposFileSize;
            tCIDLib::EFileAttrs     eAttrs;
            tCIDLib::Tch            szName[kCIDLib::c4MaxPathLen+1];
        };

        struct TRawVolumeInfo
        {
            tCIDLib::TCard4         c4BytesPerSector;
            tCIDLib::TCard4         c4MaxPathCompLen;
            tCIDLib::TCard4         c4SectorsPerUnit;
            tCIDLib::TCard4         c4TotalUnits;
            tCIDLib::TCard4         c4UnitsAvailable;
            tCIDLib::TCard4         c4VolumeSerialNum;
            tCIDLib::EFileSysFlags  eFlags;
            tCIDLib::TZStr64        szFileSysType;
            tCIDLib::TZStr128       szVolumeLabel;
        };


        // -------------------------------------------------------------------
        //  Constructors and destructors
        // -------------------------------------------------------------------
        TKrnlFileSys();

        ~TKrnlFileSys();


        // --------------------------------------------------------------------
        //  Public, static methods
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean bExists
        (
            const   tCIDLib::Tch* const     pszToFind
            , const tCIDLib::EFileAttrs     eCanBe
            , const tCIDLib::EFileAttrs     eMustBe = tCIDLib::EFileAttr_None
            ,       tCIDLib::TBoolean&      bIsUnique = NUL_TBoolean
        );

        static tCIDLib::TBoolean bIsSpecialDir
        (
            const   tCIDLib::Tch* const     pszDirToCheck
        );

        static tCIDLib::TVoid CopyFile
        (
            const   tCIDLib::Tch* const     pszSourceFile
            , const tCIDLib::Tch* const     pszTargetFile
            , const tCIDLib::TBoolean       bFailIfExists = kCIDLib::False
        );

        static tCIDLib::TVoid CreateTmpFileName
        (
                    tCIDLib::Tch* const     pszToFillIn
            , const tCIDLib::TCard4         c4BufChars
        );

        static tCIDLib::TVoid DeleteFile
        (
            const   tCIDLib::Tch* const     pszToDelete
        );

        static tCIDLib::EDriveTypes eDriveType
        (
            const   tCIDLib::Tch* const     pszVolumePath
        );

        static tCIDLib::TVoid FindInPath
        (
            const   tCIDLib::Tch* const     pszPath
            , const tCIDLib::Tch* const     pszSpec
            ,       tCIDLib::Tch* const     pszMatch
            , const tCIDLib::TCard4         c4BufChars
        );

        static tCIDLib::TVoid MakeDirectory
        (
            const   tCIDLib::Tch* const     pszToCreate
            , const tCIDLib::Tch* const     pszTemplate = 0
        );

        static tCIDLib::Tch* pszQueryVolumeList();

        static tCIDLib::TVoid QueryCurrentDir
        (
                    tCIDLib::Tch* const     pszToFillIn
            , const tCIDLib::TCard4         c4BufChars
        );

        static tCIDLib::TVoid QueryFullPath
        (
            const   tCIDLib::Tch* const     pszPartialPath
            ,       tCIDLib::Tch* const     pszToFillIn
            , const tCIDLib::TCard4         c4BufChars
        );

        static tCIDLib::TVoid QueryTmpPath
        (
                    tCIDLib::Tch* const     pszToFillIn
            , const tCIDLib::TCard4         c4BufChars
        );

        static tCIDLib::TVoid QueryVolumeInfo
        (
            const   tCIDLib::Tch* const     pszVolumePath
            ,       TRawVolumeInfo&         VolInfo
        );

        static tCIDLib::TVoid RemoveDirectory
        (
            const   tCIDLib::Tch* const     pszToDelete
        );

        static tCIDLib::TVoid RemovePath
        (
            const   tCIDLib::Tch* const     pszStartDir
        );

        static tCIDLib::TVoid Rename
        (
            const   tCIDLib::Tch* const     pszOldName
            , const tCIDLib::Tch* const     pszNewName
        );


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TKrnlFileSys(const TKrnlFileSys&);

        tCIDLib::TVoid operator=(const TKrnlFileSys&);
};



// ----------------------------------------------------------------------------
//   CLASS: TKrnlDirSearch
//  PREFIX: kds
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlDirSearch
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKrnlDirSearch();

        ~TKrnlDirSearch();


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bFindFirst
        (
            const   tCIDLib::Tch* const         pszToSearch
            , const tCIDLib::EFileAttrs         eCanBe
            , const tCIDLib::EFileAttrs         eMustBe
            ,       TKrnlFileSys::TRawFileFind& fndbToFill
        );

        tCIDLib::TBoolean bFindNext
        (
                    TKrnlFileSys::TRawFileFind& fndbToFill
        );

        tCIDLib::TBoolean bSearchIsOpen() const;

        tCIDLib::TVoid Close();

        tCIDLib::EFileAttrs eCanBe() const;

        tCIDLib::EFileAttrs eMustBe() const;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TKrnlDirSearch(const TKrnlDirSearch&);

        tCIDLib::TVoid operator=(const TKrnlDirSearch&);


        // --------------------------------------------------------------------
        //  Private data members
        // --------------------------------------------------------------------
        tCIDLib::TDirHandle     __hdirSearch;
        tCIDLib::EFileAttrs     __eCanBe;
        tCIDLib::EFileAttrs     __eMustBe;
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  TKrnlDirSearch: Public, non-virtual methods
// ----------------------------------------------------------------------------

inline tCIDLib::TBoolean TKrnlDirSearch::bSearchIsOpen() const
{
    if (__hdirSearch)
        return kCIDLib::True;
    return kCIDLib::False;
}

inline tCIDLib::EFileAttrs TKrnlDirSearch::eCanBe() const
{
    return __eCanBe;
}

inline tCIDLib::EFileAttrs TKrnlDirSearch::eMustBe() const
{
    return __eMustBe;
}
