//
//  FILE NAME: CIDKernel_FileSystem.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/19/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the the TKrnlFileSys class, the core wrapper
//  class for file system APIs. It also implements the small TKrnlFileSysInfo
//  class which is used to return file system info on a volume.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//  Local static data
// ----------------------------------------------------------------------------
static const tCIDLib::TCard4 __c4IgnoreAttrs =  FILE_ATTRIBUTE_ARCHIVE
                                                | FILE_ATTRIBUTE_NORMAL;


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __TreeDelete
//
// DESCRIPTION:
//
//  This is a recursive function to delete an entire directory tree. Its
//  a depth first function so it goes into each subdirectory as it finds
//  them, deletes any files in the starting directory, then removes the
//  starting directory and returns.
// ---------------------------------------
//   INPUT: pszStartDir is the starting directory for this level of
//              recursion. It must not have the trailing separator!
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TreeDelete(const tCIDLib::Tch* const pszStartDir)
{
    //
    //  Note that we allocate both the file find data and the search
    //  spec string in order to minimize stack usage in case of heavily
    //  nested paths.
    //
    WIN32_FIND_DATA*    pFileData = 0;
    tCIDLib::TDirHandle hTmp = 0;
    tCIDLib::TErrCode   errcRet = 0;
    tCIDLib::Tch*       pszSubSearch = 0;
    try
    {   
        //
        //  Create a string that consists of the passed path and with an
        //  appended extension to find everything in this directory.
        //
        tCIDLib::TCard4 c4Max = TRawStr::c4StrLen(pszStartDir);
                                + TRawStr::c4StrLen(kCIDLib::pszAllFilesSpec)
                                + 1;
        pszSubSearch = new tCIDLib::Tch[c4Max+1];

        tCIDLib::Tch szTmp[2];
        szTmp[0] = kCIDLib::chPathSeparator;
        szTmp[1] = kCIDLib::chNull;
        TRawStr::CopyCatStr
        (
            pszSubSearch
            , c4Max
            , pszStartDir
            , szTmp
            , kCIDLib::pszAllFilesSpec
        );

        //
        //  Allocate the buffer so we don't eat up stack space on a 
        //  large disk with deep directory structure. We have a catch
        //  block in place.
        //
        pFileData = new WIN32_FIND_DATA;
        hTmp = FindFirstFile(pszSubSearch, pFileData);
        if (hTmp == INVALID_HANDLE_VALUE)
            TKrnlError::ThrowKrnlError();

        while (1)
        {
            //
            //  If its a directory, lets recurse, otherwise, we need to 
            //  set its attributes to normal (if its not already) and then
            //  delete it.
            //
            if (pFileData->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
            {
                // Recurse on this path
                __TreeDelete(pFileData->cFileName);
            }
             else
            {
                tCIDLib::TCard4 c4Override =FILE_ATTRIBUTE_HIDDEN
                                            | FILE_ATTRIBUTE_READONLY
                                            | FILE_ATTRIBUTE_SYSTEM;

                //
                //  If any of the attributes are on that we have to override
                //  in order to delete the file, then lets ste it back to
                //  normal.
                //
                if (pFileData->dwFileAttributes & c4Override)
                {
                    if (!SetFileAttributes
                    (
                        pFileData->cFileName
                        , FILE_ATTRIBUTE_NORMAL))
                    {
                        errcRet = GetLastError();
                        throw errcRet;
                    }
                }

                // And now try to delete the file
                if (!DeleteFileW(pFileData->cFileName))
                    TKrnlError::ThrowKrnlError();
            }

            // Now find the next file/directory
            if (!FindNextFile(hTmp, pFileData))
            {
                //
                //  If the reason is no more files, that's cool and its
                //  not an error, i.e. we just return. Otherwise its an
                //  error and we throw it to unwind.
                //
                TKrnlError kerrToThrow;
                if (kerrToThrow.errcId() == kKrnlErrors::errcNoMoreFiles)
                    break;
                throw kerrToThrow;
            }
        }

        //
        //  Ok, we've cleaned out everything under the starting directory,
        //  so lets remove it now.
        //
        if (!RemoveDirectoryW(pszStartDir))
            TKrnlError::ThrowKrnlError();
    }

    catch(...)
    {
        delete pFileData;
        delete pszSubSearch;
        if (hTmp)
        {
            if (!FindClose(hTmp))
                TKrnlError::ThrowKrnlError();
        }
        throw;
    }

    // Clean up the buffers we allocated
    delete pFileData;
    delete pszSubSearch;

    if (hTmp)
    {
        if (!FindClose(hTmp))
            TKrnlError::ThrowKrnlError();
    }
}


//
// FUNCTION/METHOD NAME: __ConvertFindBuf
//
// DESCRIPTION:
//
//  Converts a system file information buffer to our buffer type.
// ---------------------------------------
//   INPUT: FileData is the system find buffer to convert.
//
//  OUTPUT: fndbToFill is the CIDLib find buffer to fill in
//
//  RETURN: None
//
static tCIDLib::TVoid
__ConvertFindBuf(   const   WIN32_FIND_DATA&            FileData
                    ,       TKrnlFileSys::TRawFileFind& fndbToFill)
{
    //
    //  Copy over all of the time members. We store these as the UCT
    //  value in a TInt8 value.
    //
    FILETIME LocalTime;
    if (!FileTimeToLocalFileTime(&FileData.ftCreationTime, &LocalTime))
        TKrnlError::ThrowKrnlError();
    fndbToFill.i8CreationTime = TRawBits::i8From32
    (
        LocalTime.dwLowDateTime, LocalTime.dwHighDateTime
    );

    if (!FileTimeToLocalFileTime(&FileData.ftLastAccessTime, &LocalTime))
        TKrnlError::ThrowKrnlError();
    fndbToFill.i8LastAccessTime = TRawBits::i8From32
    (
        LocalTime.dwLowDateTime, LocalTime.dwHighDateTime
    );

    if (!FileTimeToLocalFileTime(&FileData.ftLastWriteTime, &LocalTime))
        TKrnlError::ThrowKrnlError();
    fndbToFill.i8LastWriteTime = TRawBits::i8From32
    (
        LocalTime.dwLowDateTime, LocalTime.dwHighDateTime
    );

    fndbToFill.fposFileSize = TRawBits::i8From32
    (
        FileData.nFileSizeLow, FileData.nFileSizeHigh
    );

    // Convert the file attributes over
    fndbToFill.eAttrs = tCIDLib::EFileAttrs(FileData.dwFileAttributes);

    // And copy the file name
    TRawStr::CopyStr
    (
        fndbToFill.szName
        , FileData.cFileName
        , c4MaxBufChars(fndbToFill.szName)
    );
}




// -----------------------------------------------------------------------------
//   CLASS: TKrnlDirSearch
//  PREFIX: kds
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TKrnlDirSearch: Constructors and Destructors
// -----------------------------------------------------------------------------

TKrnlDirSearch::TKrnlDirSearch() :

    __hdirSearch(kCIDLib::hdirInvalid)
    , __eCanBe(tCIDLib::EFileAttr_None)
    , __eMustBe(tCIDLib::EFileAttr_None)
{
}

TKrnlDirSearch::~TKrnlDirSearch()
{
    Close();
}


// -----------------------------------------------------------------------------
//  TKrnlDirSearch: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean
TKrnlDirSearch::bFindFirst( const   tCIDLib::Tch* const         pszToSearch
                            , const tCIDLib::EFileAttrs         eCanBe
                            , const tCIDLib::EFileAttrs         eMustBe
                            ,       TKrnlFileSys::TRawFileFind& fndbToFill)
{
    // If there is an existing search, then close it
    if (__hdirSearch)
        Close();

    WIN32_FIND_DATA FileData;
    tCIDLib::TDirHandle hTmp = FindFirstFile(pszToSearch, &FileData);

    if (hTmp == INVALID_HANDLE_VALUE)
    {
        TKrnlError kerrToThrow;
        if (kerrToThrow.errcId() == kKrnlErrors::errcFileNotFound)
            return kCIDLib::False;

        throw kerrToThrow;
    }

    // It will be easier to deal with the attributes here as non-enums
    tCIDLib::TCard4 c4CanBe = eCanBe;
    tCIDLib::TCard4 c4MustBe = eMustBe;

    //
    //  Now we continue to check until we find one that meets the attribute
    //  criteria provided.
    //
    tCIDLib::TBoolean bFound = kCIDLib::False;
    while (!bFound)
    {
        //
        //  See if there are any attributes on in the found file that are
        //  not in the attribute mask. It can have fewer, but not more.
        //  However, we ignore the archive and normal bit. So if we AND the
        //  file's attributes with the not of the desired ones, if anything is
        //  still on, then it had more than was asked for.
        //
        if (!(FileData.dwFileAttributes & ~(c4CanBe | __c4IgnoreAttrs)))
        {
            //                                    
            //  There was at least some match, so make sure that the must
            //  flags are ok.
            //
            if ((FileData.dwFileAttributes & c4MustBe) == c4MustBe)
                bFound = kCIDLib::True;
        }

        if (!bFound)
        {
            if (!FindNextFile(hTmp, &FileData))
            {
                TKrnlError kerrToThrow;
                if (kerrToThrow.errcId() == kKrnlErrors::errcNoMoreFiles)
                    break;
                throw kerrToThrow;
            }
        }
    }

    if (bFound)
    {
        //
        //  We found at least one, so convert the find buffer to our
        //  format, into the caller's buffer.
        //
        __ConvertFindBuf(FileData, fndbToFill);

        // Store the temp handle and save the attributes
        __hdirSearch = hTmp;
        __eCanBe = eCanBe;
        __eMustBe = eMustBe;
    }
     else
    {
        // We never found one, so close the handle
        FindClose(hTmp);
    }
    return bFound;
}


tCIDLib::TBoolean
TKrnlDirSearch::bFindNext(TKrnlFileSys::TRawFileFind& fndbToFill)
{
    if (!__hdirSearch)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    // It will be easier to deal with the attributes here as non-enums
    tCIDLib::TCard4 c4CanBe = __eCanBe;
    tCIDLib::TCard4 c4MustBe = __eMustBe;

    //
    //  Now we continue to check until we find one that meets the attribute
    //  criteria provided.
    //
    tCIDLib::TBoolean   bFound = kCIDLib::False;
    WIN32_FIND_DATA     FileData;
    while (!bFound)
    {
        if (!FindNextFile(__hdirSearch, &FileData))
        {
            TKrnlError kerrToThrow;
            if (kerrToThrow.errcId() != kKrnlErrors::errcNoMoreFiles)
                TKrnlError::ThrowKrnlError();
            break;
        }

        //
        //  See if there are any attributes on in the found file that are
        //  not in the attribute mask. It can have fewer, but not more.
        //  However, we ignore the archive bit. So if we AND the file's
        //  attributes with the not of the desired ones, if anything is
        //  still on, then it had more than was asked for.
        //
        if (!(FileData.dwFileAttributes & ~(c4CanBe | __c4IgnoreAttrs)))
        {
            //                                    
            //  There was at least some match, so make sure that the must
            //  flags are ok.
            //
            if ((FileData.dwFileAttributes & c4MustBe) == c4MustBe)
                bFound = kCIDLib::True;
        }
    }

    // Convert the system structure to our structure, if we found one.
    if (bFound)
        __ConvertFindBuf(FileData, fndbToFill);

    return bFound;
}


tCIDLib::TVoid TKrnlDirSearch::Close()
{
    if (__hdirSearch != kCIDLib::hdirInvalid)
    {
        tCIDLib::TDirHandle hTmp = __hdirSearch;
        __hdirSearch = kCIDLib::hdirInvalid;

        if (!FindClose(hTmp))
            TKrnlError::ThrowKrnlError();
    }
}





// -----------------------------------------------------------------------------
//   CLASS: TKrnlFileSys
//  PREFIX: kfsys
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TKrnlFileSys: Constructors and Destructors
// -----------------------------------------------------------------------------

TKrnlFileSys::TKrnlFileSys()
{
}

TKrnlFileSys::~TKrnlFileSys()
{
}


// -----------------------------------------------------------------------------
//  TKrnlFileSys: Public, static methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean
TKrnlFileSys::bExists(  const   tCIDLib::Tch* const pszToFind
                        , const tCIDLib::EFileAttrs eCanBe
                        , const tCIDLib::EFileAttrs eMustBe
                        ,       tCIDLib::TBoolean&  bIsUnique)
{
    // Init the unique flag
    if (&bIsUnique)
        bIsUnique = kCIDLib::True;

    TKrnlFileSys::TRawFileFind  fndbToFill;
    TKrnlDirSearch              kdsSearch;

    if (!kdsSearch.bFindFirst(pszToFind, eCanBe, eMustBe, fndbToFill))
        return kCIDLib::False;

    // If they care about uniqueness, then seach for the next
    if (&bIsUnique)
    {
        if (kdsSearch.bFindNext(fndbToFill))
            bIsUnique = kCIDLib::False;
    }
    return kCIDLib::True;
}

tCIDLib::TBoolean
TKrnlFileSys::bIsSpecialDir(const tCIDLib::Tch* const pszDirToCheck)
{
    if (!TRawStr::eCompareStr(pszDirToCheck, L".")
    ||  !TRawStr::eCompareStr(pszDirToCheck, L".."))
    {
        return kCIDLib::True;
    }
    return kCIDLib::False;
}

tCIDLib::TVoid
TKrnlFileSys::CopyFile( const   tCIDLib::Tch* const pszSourceFile
                        , const tCIDLib::Tch* const pszTargetFile
                        , const tCIDLib::TBoolean   bFailIfExists)
{
    int                 iCancel = 0;
    tCIDLib::TCard4     c4Flags = 0;

    if (bFailIfExists)
        c4Flags = COPY_FILE_FAIL_IF_EXISTS;

    if (!CopyFileEx(pszSourceFile, pszTargetFile, 0, 0, &iCancel, c4Flags))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid
TKrnlFileSys::CreateTmpFileName(        tCIDLib::Tch* const pszToFillIn
                                , const tCIDLib::TCard4     c4BufChars)
{
    tCIDLib::Tch pszTmp[kCIDLib::c4MaxPathLen];
    if (!GetTempFileName(L".", L"Tmp", 0, pszTmp))
        TKrnlError::ThrowKrnlError();

    if (TRawStr::c4StrLen(pszTmp) > c4BufChars)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInsufficientBuffer);

    TRawStr::CopyStr(pszToFillIn, pszTmp, c4BufChars);
}


tCIDLib::TVoid
TKrnlFileSys::DeleteFile(const tCIDLib::Tch* const pszToDelete)
{
    if (!::DeleteFileW(pszToDelete))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::EDriveTypes
TKrnlFileSys::eDriveType(const tCIDLib::Tch* const pszVolumePath)
{
    // Init the return
    tCIDLib::EDriveTypes eType = tCIDLib::EDriveType_Unknown;

    // Loop up the drive type of the volume
    tCIDLib::TCard4 c4Type = GetDriveType(pszVolumePath);

    if (c4Type == DRIVE_UNKNOWN)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcPathNotFound);

    if (c4Type == DRIVE_REMOVABLE)
        eType = tCIDLib::EDriveType_Removable;
     else if (c4Type == DRIVE_FIXED)
        eType = tCIDLib::EDriveType_Fixed;
     else if (c4Type == DRIVE_REMOTE)
        eType = tCIDLib::EDriveType_Remote;
     else if (c4Type == DRIVE_CDROM)
        eType = tCIDLib::EDriveType_CD;
     else if (c4Type == DRIVE_RAMDISK)
        eType = tCIDLib::EDriveType_RAMDisk;

    return eType;
}


tCIDLib::TVoid
TKrnlFileSys::FindInPath(   const   tCIDLib::Tch* const pszPath
                            , const tCIDLib::Tch* const pszSpec
                            ,       tCIDLib::Tch* const pszMatch
                            , const tCIDLib::TCard4     c4BufChars)
{
    tCIDLib::Tch* pszDummy;
    if (!::SearchPath(pszPath, pszSpec, 0, c4BufChars, pszMatch, &pszDummy))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::Tch* TKrnlFileSys::pszQueryVolumeList()
{
    //
    //  Find out the list of available volumes. First find out how big the
    //  string must be, then allocated it and call again.
    //
    tCIDLib::TCard4 c4Len = GetLogicalDriveStrings(0, 0);
    if (!c4Len)
        TKrnlError::ThrowKrnlError();

    // Allocate the string that we need now
    tCIDLib::Tch* pszRet = new tCIDLib::Tch[c4Len+2];

    // Now get the information for return
    if (!GetLogicalDriveStrings(c4Len+1, pszRet))
    {
        delete pszRet;
        TKrnlError::ThrowKrnlError();
    }
    return pszRet;
}


tCIDLib::TVoid
TKrnlFileSys::QueryCurrentDir(  tCIDLib::Tch* const     pszToFillIn
                                , const tCIDLib::TCard4 c4BufChars)
{
    if (!GetCurrentDirectory(c4BufChars, pszToFillIn))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid
TKrnlFileSys::QueryFullPath(const   tCIDLib::Tch* const pszPartialPath
                            ,       tCIDLib::Tch* const pszToFillIn
                            , const tCIDLib::TCard4     c4BufChars)
{
    tCIDLib::Tch*   pszDummy;
    tCIDLib::TCard4 c4Len = GetFullPathName
    (
        pszPartialPath
        , c4BufChars
        , pszToFillIn
        , &pszDummy
    );

    if (!c4Len)
        TKrnlError::ThrowKrnlError();

    if (c4Len > c4BufChars)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInsufficientBuffer);
}


tCIDLib::TVoid
TKrnlFileSys::QueryTmpPath(         tCIDLib::Tch* const pszToFillIn
                            , const tCIDLib::TCard4     c4BufChars)
{
    tCIDLib::TCard4 c4Len = GetTempPath(c4BufChars, pszToFillIn);
    if (!c4Len)
        TKrnlError::ThrowKrnlError();

    if (c4Len > c4BufChars)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInsufficientBuffer);
}


tCIDLib::TVoid TKrnlFileSys::QueryVolumeInfo
(
    const   tCIDLib::Tch* const             pszVolumePath
    ,       TKrnlFileSys::TRawVolumeInfo&   VolInfo)
{
    // Query the volume info for this volume
    if (!GetVolumeInformation
    (
        pszVolumePath
        , VolInfo.szVolumeLabel
        , c4MaxBufChars(VolInfo.szVolumeLabel)
        , &VolInfo.c4VolumeSerialNum
        , &VolInfo.c4MaxPathCompLen
        , (tCIDLib::TCard4*)&VolInfo.eFlags
        , VolInfo.szFileSysType
        , c4MaxBufChars(VolInfo.szFileSysType)))
    {
        TKrnlError::ThrowKrnlError();
    }

    // Now query the volume space info
    if (!GetDiskFreeSpace
    (
        pszVolumePath
        , &VolInfo.c4SectorsPerUnit
        , &VolInfo.c4BytesPerSector
        , &VolInfo.c4UnitsAvailable
        , &VolInfo.c4TotalUnits))
    {
        TKrnlError::ThrowKrnlError();
    }
}


tCIDLib::TVoid
TKrnlFileSys::MakeDirectory(const   tCIDLib::Tch* const pszToCreate
                            , const tCIDLib::Tch* const pszTemplate)
{
    if (pszTemplate)
    {
        if (!::CreateDirectoryEx(pszToCreate, pszTemplate, 0))
            TKrnlError::ThrowKrnlError();
    }
     else
    {
        if (!::CreateDirectory(pszToCreate, 0))
            TKrnlError::ThrowKrnlError();
    }
}


tCIDLib::TVoid
TKrnlFileSys::RemoveDirectory(const tCIDLib::Tch* const pszToDelete)
{
    if (!::RemoveDirectoryW(pszToDelete))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid
TKrnlFileSys::RemovePath(const tCIDLib::Tch* const pszStartDir)
{
    __TreeDelete(pszStartDir);
}


tCIDLib::TVoid
TKrnlFileSys::Rename(   const   tCIDLib::Tch* const pszOldName
                        , const tCIDLib::Tch* const pszNewName)
{
    if (!MoveFile(pszOldName, pszNewName))
        TKrnlError::ThrowKrnlError();
}
