//
//  FILE NAME: CIDKernel_File.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/10/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the the TKrnlFile class, the core wrapper
//  class for file handles.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//  Local static functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __ConvertByHandleBuf
//
// DESCRIPTION:
//
//  Converts a 'info by handle' buffer from the system format to our
//  standard file info format.
// ---------------------------------------
//   INPUT: FileData is the system find buffer to convert.
//
//  OUTPUT: fndbToFill is the CIDLib find buffer to fill in
//
//  RETURN: None
//
static tCIDLib::TVoid
__ConvertByHandleBuf(   const   BY_HANDLE_FILE_INFORMATION& FileData
                        ,       TKrnlFileSys::TRawFileFind& fndbToFill)
{
    //
    //  Copy over all of the time members. We store these as the UCT
    //  value in a TInt8 value.
    //
    fndbToFill.i8CreationTime = TRawBits::i8From32
    (
        FileData.ftCreationTime.dwLowDateTime
        , FileData.ftCreationTime.dwHighDateTime
    );

    fndbToFill.i8LastAccessTime = TRawBits::i8From32
    (
        FileData.ftLastAccessTime.dwLowDateTime
        , FileData.ftLastAccessTime.dwHighDateTime
    );

    fndbToFill.i8LastWriteTime = TRawBits::i8From32
    (
        FileData.ftLastWriteTime.dwLowDateTime
        , FileData.ftLastWriteTime.dwHighDateTime
    );

    fndbToFill.fposFileSize = TRawBits::i8From32
    (
        FileData.nFileSizeLow, FileData.nFileSizeHigh
    );

    // Convert the file attributes over
    fndbToFill.eAttrs = tCIDLib::EFileAttrs(FileData.dwFileAttributes);
}




// ----------------------------------------------------------------------------
//   CLASS: TKrnlFile
//  PREFIX: kfl
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlFile: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlFile::TKrnlFile() :

    __hflThis(0)
    , __pszName(0)
{
}

TKrnlFile::TKrnlFile(const tCIDLib::Tch* const pszName) :

    __hflThis(0)
    , __pszName(0)
{
    if (pszName)
        __pszName = TRawStr::pszReplicate(pszName);
}

TKrnlFile::~TKrnlFile()
{
    // Delete the name and empty it out now
    delete __pszName;
    __pszName = 0;

    Close();
}


// ----------------------------------------------------------------------------
//  TKrnlFile: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlFile::Close()
{
    if (__hflThis)
    {
        tCIDLib::TFileHandle hTmp = __hflThis;
        __hflThis = 0;
        if (!CloseHandle(hTmp))
            TKrnlError::ThrowKrnlError();
    }
}


tCIDLib::TFilePos TKrnlFile::fposCurSize() const
{
    tCIDLib::TInt8 i8Size = 0;

    // Get a pointer to high part of the return
    tCIDLib::TCard4* pc4High = (tCIDLib::TCard4*)&i8Size;
    pc4High++;

    tCIDLib::TCard4 c4Ret = GetFileSize(__hflThis, pc4High);

    if (c4Ret == kCIDLib::c4MaxCard)
    {
        // See if it was an error or a size larger than 32 bits
        if (TKrnlThread::errcGetLast())
            TKrnlError::ThrowKrnlError();
    }

    // Fill in the low word of the caller's buffer
    i8Size |= c4Ret;
    return i8Size;
}


tCIDLib::TFilePos TKrnlFile::fposFilePtr() const
{
    // Get the 0 offset into a temp
    tCIDLib::TInt8  i8New   = 0;

    // Get pointers to the low and high parts of it
    tCIDLib::TCard4*    pc4Low  = (tCIDLib::TCard4*)&i8New;
    tCIDLib::TInt4*     pi4High = (tCIDLib::TInt4*)&i8New;
    pi4High++;

    // Do a seek relative to the current position
    tCIDLib::TCard4 c4Ret = SetFilePointer
    (
        __hflThis
        , *pc4Low
        , pi4High
        , tCIDLib::ESeekFrom_Current
    );

    // See if it there was an error or just a position over 32 bits
    if (c4Ret == kCIDLib::c4MaxCard)
    {
        if (TKrnlThread::errcGetLast())
            TKrnlError::ThrowKrnlError();
    }

    // Fill in the caller's buffer
    i8New |= c4Ret;
    return i8New;
}


tCIDLib::TVoid
TKrnlFile::Open(const   tCIDLib::EAccessModes   eAccess
                , const tCIDLib::ECreateActions eAction
                , const tCIDLib::EFileAttrs     eAttrs
                , const tCIDLib::EFileFlags     eFlags)
{
    // Sanity check, just in case we are already open
    if (__hflThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (!__pszName)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNullName);

    //
    //  If the action indicates a fail if exists, then see if the file
    //  already exists and return an error if so.
    //
    if (eAction == tCIDLib::ECreateAct_FailIfExists)
    {
        if (TKrnlFileSys::bExists(__pszName, tCIDLib::EFileAttr_All))
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyExists);
    }

    //
    //  Map our access flags to the system specific flags. The others
    //  are already mapped correctly.
    //
    tCIDLib::TCard4 c4Access = 0;
    if (eAccess & tCIDLib::EAccess_Read)
        c4Access |= GENERIC_READ;

    if (eAccess & tCIDLib::EAccess_Write)
        c4Access |= GENERIC_WRITE;

    tCIDLib::TCard4 c4Share = FILE_SHARE_READ | FILE_SHARE_WRITE;
    if (eAccess & tCIDLib::EAccess_DenyRead)
        c4Share &= ~FILE_SHARE_READ;

    if (eAccess & tCIDLib::EAccess_DenyWrite)
        c4Share &= ~FILE_SHARE_WRITE;

    // Ok, lets try to do this thing.
    tCIDLib::TFileHandle hTmp = CreateFile
    (
        __pszName
        , c4Access
        , c4Share
        , 0
        , eAction
        , eAttrs | eFlags
        , 0
    );

    if (hTmp == INVALID_HANDLE_VALUE)
        TKrnlError::ThrowKrnlError();

    // We survived, so store the handle
    __hflThis = hTmp;
}


tCIDLib::TVoid TKrnlFile::Open(const tCIDLib::EStdFiles eStdFile)
{
    // Sanity check, just in case we are already open
    if (__hflThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if ((eStdFile == EStdFile_RedirIn)
         ||  (eStdFile == tCIDLib::EStdFile_RedirOut)
         ||  (eStdFile == tCIDLib::EStdFile_RedirError))
    {
        if (eStdFile == tCIDLib::EStdFile_RedirIn)
        {
            __pszName = TRawStr::pszReplicate(L"STDIN");
            __hflThis = GetStdHandle(STD_INPUT_HANDLE);
        }
        else if (eStdFile == tCIDLib::EStdFile_RedirOut)
        {
            __pszName = TRawStr::pszReplicate(L"STDOUT");
            __hflThis = GetStdHandle(STD_OUTPUT_HANDLE);
        }
        else if (eStdFile == tCIDLib::EStdFile_RedirError)
        {
            __pszName = TRawStr::pszReplicate(L"STDERR");
            __hflThis = GetStdHandle(STD_ERROR_HANDLE);
        }
    }
     else if ((eStdFile == tCIDLib::EStdFile_StdOut)
          ||  (eStdFile == tCIDLib::EStdFile_StdIn))
    {
        tCIDLib::EAccessModes   eAccess;

        if (eStdFile == tCIDLib::EStdFile_StdOut)
        {
            __pszName = TRawStr::pszReplicate(L"CONOUT$");
            eAccess = tCIDLib::EAccess_Write;
        }
         else
        {
            __pszName = TRawStr::pszReplicate(L"CONIN$");
            eAccess = tCIDLib::EAccess_Read;
        }

        Open
        (
            eAccess
            , tCIDLib::ECreateAct_OpenIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );
    }
}


tCIDLib::TVoid
TKrnlFile::QueryFileInfo(TKrnlFileSys::TRawFileFind& fndbToFill)
{
    BY_HANDLE_FILE_INFORMATION  FileData;
    if (!GetFileInformationByHandle(__hflThis, &FileData))
        TKrnlError::ThrowKrnlError();

    // Convert the system structure to our structure
    __ConvertByHandleBuf(FileData, fndbToFill);
}


tCIDLib::TVoid
TKrnlFile::ReadBuffer(          tCIDLib::TVoid* const   pBuffer
                        , const tCIDLib::TCard4         c4ToRead
                        ,       tCIDLib::TCard4&        c4Actual)
{
    // Default the actual parm if provided
    if (&c4Actual)
        c4Actual = 0;

    tCIDLib::TCard4     c4ActualActual;
    if (!ReadFile(__hflThis, pBuffer, c4ToRead, &c4ActualActual, 0))
    {
        if (&c4Actual)
            c4Actual = c4ActualActual;
        TKrnlError::ThrowKrnlError();
    }

    if (&c4Actual)
        c4Actual = c4ActualActual;
}


tCIDLib::TVoid
TKrnlFile::Seek(const   tCIDLib::TFilePos&  fposOffset
                , const tCIDLib::ESeekFrom  eSeek
                ,       tCIDLib::TFilePos&  fposActual)
{
    // Get the offset into a temp
    tCIDLib::TInt8  i8New   = fposOffset;

    // Get pointers to the low and high parts of it
    tCIDLib::TCard4*    pc4Low  = (tCIDLib::TCard4*)&i8New;
    tCIDLib::TInt4*     pi4High = (tCIDLib::TInt4*)&i8New;
    pi4High++;

    tCIDLib::TCard4 c4Ret = SetFilePointer
    (
        __hflThis
        , *pc4Low
        , pi4High
        , eSeek
    );

    if (c4Ret == kCIDLib::c4MaxCard)
    {
        if (TKrnlThread::errcGetLast())
            TKrnlError::ThrowKrnlError();
    }

    // Fill in the actual if we need to
    if (&fposActual)
    {
        fposActual = i8New & 0xFFFFFFFF00000000;
        fposActual |= c4Ret;
    }
}


tCIDLib::TVoid
TKrnlFile::SetFileTimes(const   TKrnlTimeStamp& ktmsCreate
                        , const TKrnlTimeStamp& ktmsLastAccess
                        , const TKrnlTimeStamp& ktmsLastWrite)
{
    // Assume any can be 
    FILETIME*   pCreate = 0;
    FILETIME*   pLastAccess = 0;
    FILETIME*   pLastWrite = 0;

    if (&ktmsCreate)
    {
        pCreate = new FILETIME;
        pCreate->dwLowDateTime = TRawBits::c4Low32From64(ktmsCreate.i8Time());
        pCreate->dwHighDateTime = TRawBits::c4High32From64(ktmsCreate.i8Time());
    }

    if (&ktmsLastAccess)
    {
        pLastAccess = new FILETIME;
        pLastAccess->dwLowDateTime = TRawBits::c4Low32From64(ktmsLastAccess.i8Time());
        pLastAccess->dwHighDateTime = TRawBits::c4High32From64(ktmsLastAccess.i8Time());
    }

    if (&ktmsLastWrite)
    {
        pLastWrite = new FILETIME;
        pLastWrite->dwLowDateTime = TRawBits::c4Low32From64(ktmsLastWrite.i8Time());
        pLastWrite->dwHighDateTime = TRawBits::c4High32From64(ktmsLastWrite.i8Time());
    }

    // And now do the system call
    if (!SetFileTime(__hflThis, pCreate, pLastAccess, pLastWrite))
		TKrnlError::ThrowKrnlError();

    // Clean them all up that got allocated
    delete pCreate;
    delete pLastAccess;
    delete pLastWrite;
}

tCIDLib::TVoid
TKrnlFile::SetFileTimes(const   tCIDLib::TInt8& i8Create
                        , const tCIDLib::TInt8& i8LastAccess
                        , const tCIDLib::TInt8& i8LastWrite)
{
    SetFileTimes
    (
        TKrnlTimeStamp(i8Create)
        , TKrnlTimeStamp(i8LastAccess)
        , TKrnlTimeStamp(i8LastWrite)
    );
}


tCIDLib::TVoid
TKrnlFile::SetName(const tCIDLib::Tch* const pszNewName)
{
    if (__hflThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    // Delete the current name and replicate the new one
    delete __pszName;
    __pszName = 0;
    __pszName = TRawStr::pszReplicate(pszNewName);
}


tCIDLib::TVoid TKrnlFile::TruncateAt(const tCIDLib::TInt8& i8Position)
{
    tCIDLib::TInt8 i8Actual;

    Seek(i8Position, tCIDLib::ESeekFrom_Start, i8Actual);
    if (!SetEndOfFile(__hflThis))
        TKrnlError::ThrowKrnlError();
}

tCIDLib::TVoid TKrnlFile::TruncateAt()
{
    if (!SetEndOfFile(__hflThis))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid
TKrnlFile::WriteBuffer( const   tCIDLib::TVoid* const   pBuffer
                        , const tCIDLib::TCard4         c4ToWrite
                        ,       tCIDLib::TCard4&        c4Actual)
{
    // Default the actual parm if provided
    if (&c4Actual)
        c4Actual = 0;

    tCIDLib::TCard4     c4ActualActual;
    if (!WriteFile(__hflThis, pBuffer, c4ToWrite, &c4ActualActual, 0))
    {
        if (&c4Actual)
            c4Actual = c4ActualActual;
        TKrnlError::ThrowKrnlError();
    }

    if (&c4Actual)
        c4Actual = c4ActualActual;
}
