//
// NAME: CIDKernel_ExternalProcess.Cpp
//
// DESCRIPTION:
//
//	This module implements the TKrnlExtProcess class.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/13/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"



// ----------------------------------------------------------------------------
//   CLASS: TKrnlExtProcess
//  PREFIX: kextp
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlExtProcess: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlExtProcess::TKrnlExtProcess() :

    __hprocThis(kCIDLib::hprocInvalid)
    , __pidThis(kCIDLib::pidInvalid)
{
}

TKrnlExtProcess::~TKrnlExtProcess()
{
    // Close the process handle if open
    if (__hprocThis != kCIDLib::hprocInvalid)
    {
        ::CloseHandle(__hprocThis);
        __hprocThis = kCIDLib::hprocInvalid;
    }
    __pidThis = kCIDLib::pidInvalid;
}


// ----------------------------------------------------------------------------
//  TKrnlExtProcess: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TKrnlExtProcess::bIsRunning() const
{
    if (__hprocThis == kCIDLib::hprocInvalid)
        return kCIDLib::False;

    tCIDLib::TCard4 c4Res = WaitForSingleObject(__hprocThis, 0);

    if (c4Res == WAIT_FAILED)
        TKrnlError::ThrowKrnlError();

    if (c4Res == WAIT_OBJECT_0)
        return kCIDLib::False;

    return kCIDLib::True;
}

tCIDLib::EPrioClasses TKrnlExtProcess::ePriorityClass() const
{
    if (!bIsRunning())
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotRunning);

    tCIDLib::TCard4 c4Prio = GetPriorityClass(__hprocThis);

    if (c4Prio == REALTIME_PRIORITY_CLASS)
        return tCIDLib::EPrioClass_RealTime;
    else if (c4Prio == HIGH_PRIORITY_CLASS)
        return tCIDLib::EPrioClass_High;
    else if (c4Prio == NORMAL_PRIORITY_CLASS)
        return tCIDLib::EPrioClass_Normal;
    else if (c4Prio == IDLE_PRIORITY_CLASS)
        return tCIDLib::EPrioClass_IdleTime;
    else
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcUnknownPrio);

    // Make the compiler happy
    return tCIDLib::EPrioClass_Normal;
}

tCIDLib::EPrioClasses
TKrnlExtProcess::ePriorityClass(const tCIDLib::EPrioClasses eNewClass)
{
    if (!bIsRunning())
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotRunning);

    tCIDLib::TCard4 c4NewPrio;

    if (eNewClass == tCIDLib::EPrioClass_IdleTime)
        c4NewPrio = IDLE_PRIORITY_CLASS;
    else if (eNewClass == tCIDLib::EPrioClass_Normal)
        c4NewPrio = NORMAL_PRIORITY_CLASS;
    else if (eNewClass == tCIDLib::EPrioClass_High)
        c4NewPrio = HIGH_PRIORITY_CLASS;
    else if (eNewClass == tCIDLib::EPrioClass_RealTime)
        c4NewPrio = REALTIME_PRIORITY_CLASS;
    else
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcUnknownPrio);

    return eNewClass;
}

tCIDLib::EExitCodes
TKrnlExtProcess::eWaitForDeath(const tCIDLib::TCard4 c4MilliSecs) const
{
    if (__hprocThis == kCIDLib::hprocInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNeverStarted);

    tCIDLib::TCard4 c4Res = WaitForSingleObject(__hprocThis, c4MilliSecs);

    if (c4Res == WAIT_FAILED)
        TKrnlError::ThrowKrnlError();

    if (c4Res == WAIT_TIMEOUT)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcTimeout);

    // Query tbe result code
    tCIDLib::TCard4 c4Code;
    if (!GetExitCodeProcess(__hprocThis, &c4Code))
        TKrnlError::ThrowKrnlError();
       
    return tCIDLib::EExitCodes(c4Code);
}

tCIDLib::TVoid TKrnlExtProcess::Kill()
{
    if (!bIsRunning())
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotRunning);

    if (!TerminateProcess(__hprocThis, 0))
        TKrnlError::ThrowKrnlError();
}

tCIDLib::TVoid 
TKrnlExtProcess::Start( const   tCIDLib::Tch* const pszPath
                        , const tCIDLib::Tch**      apszParms
                        , const tCIDLib::TCard4     c4ParmCount
                        , const tCIDLib::Tch**      apszEnviron
                        , const tCIDLib::TCard4     c4EnvironCount
                        , const tCIDLib::TBoolean   bDetached)
{
    //
    //  Ok, first check to make sure that we are not already running. If
    //  we are, then that's this is not legal.
    //
    if (bIsRunning())
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyRunning);

    //
    //  Ok, so its legal to start it up again. So lets build up the
    //  information in the required platform format. 
    //
    tCIDLib::Tch* pszEnvBuffer = 0;
    tCIDLib::Tch* pszCmdLine = 0;
    try
    {
        tCIDLib::TCard4 c4BufSize;
        tCIDLib::TCard4 c4Index;
        tCIDLib::Tch*   pszTmp;

        //
        //  First we will build up the environment block. If the passed
        //  apszEnviron parameter is 0, then we just pass a nul block to
        //  inherit this process' environment. Otherwise, we have to build
        //  up a buffer.
        //
        if (c4EnvironCount)
        {
            //
            //  First lets see how much space we need to create the environment
            //  buffer. It consists of the passed strings plus terminating
            //  nuls, plus an extra terminating nul (so we start the size at
            //  1.)
            //
            c4BufSize = 1;
            for (c4Index = 0; c4Index < c4EnvironCount; c4Index++)
                c4BufSize += TRawStr::c4StrLen(apszEnviron[c4Index]) + 1;

            // Allocate the buffer
            pszEnvBuffer = new tCIDLib::Tch[c4BufSize];

            #if CID_DEBUG_ON
            tCIDLib::TCard4 c4CharsLeft = c4BufSize;
            #endif

            // And now fill it in
            pszTmp = pszEnvBuffer;
            for (c4Index = 0; c4Index < c4EnvironCount; c4Index++)
            {
                // Get the length of this string
                tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(apszEnviron[c4Index]);

                // And copy it into the buffer
                TRawMem::CopyMemBuf
                (
                    pszTmp
                    , apszEnviron[c4Index]
                    , (c4Len+1) * kCIDLib::c4CharBytes
                );

                // Move the temp pointer past this string's nul
                pszTmp += c4Len+1;

                // If debug then keep track of chars used so far
                #if CID_DEBUG_ON
                if (c4CharsLeft < c4Len+1)
                    TKrnlError::ThrowKrnlError(kKrnlErrors::errcBufOverflow);

                c4CharsLeft -= c4Len+1;
                #endif
            }

            // Add in the second terminating nul for the end of buffer
            #if CID_DEBUG_ON
            if (c4CharsLeft != 1)
                TKrnlError::ThrowKrnlError(kKrnlErrors::errcBufOverflow);
            #endif

            *pszTmp = kCIDLib::chNull;
        }

        //
        //  If we need to create a command line, then let's do that now.
        if (c4ParmCount)
        {
            //
            //  So lets build up the command line buffer. Its just the
            //  conglomerated command strings, with a space between them
            //  and a single terminating nul. We start the counter at 1
            //  for the terminating nul.
            //
            c4BufSize = 1;
            for (c4Index = 0; c4Index < c4ParmCount; c4Index++)
                c4BufSize += TRawStr::c4StrLen(apszParms[c4Index]) + 1;

            // Allocate the buffer
            pszCmdLine = new tCIDLib::Tch[c4BufSize];

            #if CID_DEBUG_ON
            tCIDLib::TCard4 c4CharsLeft = c4BufSize;
            #endif

            // And now build them all into the buffer
            pszTmp = pszCmdLine;
            for (c4Index = 0; c4Index < c4ParmCount; c4Index++)
            {
                // Get the length of this string (without the null
                tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(apszParms[c4Index]);

                // And copy it into the buffer
                TRawMem::CopyMemBuf
                (
                    pszTmp
                    , apszParms[c4Index]
                    , c4Len * kCIDLib::c4CharBytes
                );

                //
                //  Move the temp pointer past the last character of the
                //  buffer and put a space there.
                //
                pszTmp += c4Len;
                *pszTmp = kCIDLib::chSpace;

                // If debug then keep track of chars used so far
                #if CID_DEBUG_ON
                if (c4CharsLeft < c4Len+1)
                    TKrnlError::ThrowKrnlError(kKrnlErrors::errcBufOverflow);

                c4CharsLeft -= c4Len+1;
                #endif

                // And move up to the next position
                pszTmp++;
            }

            #if CID_DEBUG_ON
            if (c4CharsLeft != 1)
                TKrnlError::ThrowKrnlError(kKrnlErrors::errcBufOverflow);
            #endif

            // Put the terminating null on it
            *pszTmp = kCIDLib::chNull;
        }

        // Set up some default startup info
        STARTUPINFO Startup = {0};
        Startup.cb = sizeof(STARTUPINFO);

        //
        //  Ok now we can finally create the process. We store away the
        //  process handle and the process id.
        //
        PROCESS_INFORMATION ProcInfo;
        if (!CreateProcessW
        (
            pszPath
            , pszCmdLine
            , 0
            , 0
            , 0
            , CREATE_DEFAULT_ERROR_MODE
              | CREATE_UNICODE_ENVIRONMENT
              | (bDetached ? DETACHED_PROCESS : 0)
            , pszEnvBuffer
            , 0
            , &Startup
            , &ProcInfo))
        {
            TKrnlError::ThrowKrnlError();
        }

        // Store away the process info
        __hprocThis = ProcInfo.hProcess;
        __pidThis = ProcInfo.dwProcessId;
    }

    catch(...)
    {
        delete pszEnvBuffer;
        delete pszCmdLine;
        throw;
    }

    delete pszEnvBuffer;
    delete pszCmdLine;
}
