//
//  FILE NAME: CIDKernel_EventSemaphore.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/12/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_EventSemaphore.Cpp module. This
//  module implements the TKrnlEvent class, which is an edge triggered
//  semaphore for holding back other threads until something happens.
//
//  CAVEATS/GOTCHAS:
//
//  1)  errcDuplicate() will duplicate an event. This is the only way to
//      open multiple copies of an unnamed event. For named events, you
//      can either create another event with the same name, or use the
//      duplicate method.
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"



// ----------------------------------------------------------------------------
//   CLASS: TKrnlEvent
//  PREFIX: kev
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlEvent: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlEvent::TKrnlEvent() :

    __eShareState(tCIDLib::EShareState_Unshared)
    , __hevThis(kCIDLib::hevInvalid)
    , __pszName(0)
{
}

TKrnlEvent::TKrnlEvent(const tCIDLib::EShareStates eShared) :

    __eShareState(eShared)
    , __hevThis(kCIDLib::hevInvalid)
    , __pszName(0)
{
}

TKrnlEvent::TKrnlEvent(const tCIDLib::Tch* const pszName) :

    __eShareState(tCIDLib::EShareState_Shared)
    , __hevThis(kCIDLib::hevInvalid)
    , __pszName(0)
{
    if (pszName)
        __pszName = TRawStr::pszReplicate(pszName);
}

TKrnlEvent::~TKrnlEvent()
{
    if (__pszName)
    {
        delete __pszName;
        __pszName = 0;
    }

    if (__hevThis != kCIDLib::hevInvalid)
        Close();
}


// ----------------------------------------------------------------------------
//  TKrnlEvent: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlEvent::Close()
{
    if (__hevThis != kCIDLib::hevInvalid)
    {
        tCIDLib::TEventHandle hTmp = __hevThis;
        __hevThis = kCIDLib::hevInvalid;

		if (!CloseHandle(hTmp))
            TKrnlError::ThrowKrnlError();
	}
}


tCIDLib::TVoid TKrnlEvent::Create(  const   tCIDLib::EEventStates   eInitState
                                    , const tCIDLib::EAutoModes     eAutoReset)
{
    if (__hevThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (__pszName)
    {
        __CreateNamed(eInitState, eAutoReset, kCIDLib::True);
        return;
    }

    tCIDLib::TEventHandle hTmp = CreateEvent(0, !eAutoReset, eInitState, 0);
    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    // We survived, so store the handle
    __hevThis = hTmp;
}

tCIDLib::TVoid
TKrnlEvent::CreateOrOpen(const  tCIDLib::EEventStates   eInitState
                        , const tCIDLib::EAutoModes     eAutoReset)
{
    if (__hevThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (!__pszName)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNullName);

    __CreateNamed(eInitState, eAutoReset, kCIDLib::False);
}

tCIDLib::TVoid TKrnlEvent::Duplicate(const TKrnlEvent& kevToDup)
{
    // Close it if open
    if (__hevThis != kCIDLib::hevInvalid)
        Close();

    // Clean up the name
    delete __pszName;
    __pszName = 0;

    // Copy over the info
    __eShareState = kevToDup.__eShareState;

    // Recreate the name
    if (kevToDup.__pszName)
        __pszName = TRawStr::pszReplicate(kevToDup.__pszName);

    // Duplicate the handle
    tCIDLib::TEventHandle hTmp;
    if (!DuplicateHandle
    (
        kevToDup.__hevThis
        , GetCurrentProcess()
        , GetCurrentProcess()
        , &hTmp
        , 0
        , 0
        , DUPLICATE_SAME_ACCESS))
    {
        TKrnlError::ThrowKrnlError();
    }
    __hevThis = hTmp;
}

tCIDLib::TVoid TKrnlEvent::Open()
{
    if (__hevThis)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    if (!__pszName)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcNullName);

    tCIDLib::TMtxHandle hTmp = OpenEvent
    (
        EVENT_ALL_ACCESS | SYNCHRONIZE
        , 0
        , __pszName
    );

    if (!hTmp)
        TKrnlError::ThrowKrnlError();

    // We survived, so store the handle
    __hevThis = hTmp;
}


tCIDLib::TVoid TKrnlEvent::Pulse()
{
    if (__hevThis == kCIDLib::hevInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

	if (!PulseEvent(__hevThis))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid TKrnlEvent::Reset()
{
    if (__hevThis == kCIDLib::hevInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    if (!ResetEvent(__hevThis))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid TKrnlEvent::SetName(const tCIDLib::Tch* const pszToSet)
{
    if (__hevThis != kCIDLib::hevInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);

    delete __pszName;
    __pszName = 0;
    __pszName = TRawStr::pszReplicate(pszToSet);
}


tCIDLib::TVoid TKrnlEvent::Trigger()
{
    if (__hevThis == kCIDLib::hevInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    if (!SetEvent(__hevThis))
        TKrnlError::ThrowKrnlError();
}


tCIDLib::TVoid TKrnlEvent::WaitFor(const tCIDLib::TCard4 c4Wait)
{
    if (__hevThis == kCIDLib::hevInvalid)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcInvalidHandle);

    tCIDLib::TCard4 c4Reason = WaitForSingleObject(__hevThis, c4Wait);

    if (c4Reason == WAIT_FAILED)
        TKrnlError::ThrowKrnlError();

    if (c4Reason == WAIT_ABANDONED)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcOwnerDied);

    if (c4Reason == WAIT_TIMEOUT)
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcTimeout);
}


// ----------------------------------------------------------------------------
//  TKrnlEvent: Private, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __CreateNamed
//
// DESCRIPTION:
//
//  These guys will create mutex, using the different rules that named
//  resources require.
// -------------------------------------
//   INPUT: eInitState indicates the initial lock state.
//          eAutoReset indicates what its reset mode is to be if created.
//          bFailIfExists indicates whether this is a create operation
//              or a create w/ optional open.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TKrnlEvent::__CreateNamed(  const   tCIDLib::EEventStates   eState
                            , const tCIDLib::EAutoModes     eAutoReset
                            , const tCIDLib::TBoolean       bFailIfExists)
{
    tCIDLib::TEventHandle hevTmp;

    //
    //  First try to open it. If that succeeds, then see if we need to
    //  fail if it exists. If so, then close it and return an error.
    //
    hevTmp = OpenEvent(EVENT_ALL_ACCESS, eState, __pszName);
    if (hevTmp)
    {
        if (bFailIfExists)
        {
            CloseHandle(hevTmp);
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyExists);
        }
        __hevThis = hevTmp;
        return;
    }

    // The open did not work, so try to create it
    hevTmp = CreateEvent
    (
        0
        , (eAutoReset == tCIDLib::EAutoMode_Automatic)
                            ? kCIDLib::False : kCIDLib::True
        , eState
        , __pszName
    );

    if (!hevTmp)
        TKrnlError::ThrowKrnlError();

    __hevThis = hevTmp;
}
