//
//  FILE NAME: CIDKernel_Error.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 04/30/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDKernel_Error.Cpp module. This module
//  implements the TKrnlError class, which is the kernel exception class
//  that is thrown for all kernel exceptions. Higher levels catch these
//  and translate them into the publically visible exception class.
//
//  CAVEATS/GOTCHAS:
//
//  1)  The default constructor gets the last host error on the calling
//      thread, stores it, and looks up the CIDLib error that it maps to.
//      If none, then its set to the generic kKrnlErrors::errcHostErr.
//
//  2)  The other constructor assumes that the passed error is a CIDKernel
//      error already and stores it as the error, setting __errcHostId to
//      0.
//
//  3)  This guy is kept minimal so as to minimize overhead for having
//      an exception based kernel.
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TKrnlError
//  PREFIX: kerr
// ----------------------------------------------------------------------------
class KRNLEXPORT TKrnlError
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKrnlError();

        TKrnlError
        (
            const   tCIDLib::TErrCode       errcId
        );

        TKrnlError
        (
            const   TKrnlError&             kerrToCopy
        );

        ~TKrnlError();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TKrnlError& operator=
        (
            const   TKrnlError&             kerrToAssign
        );


        // --------------------------------------------------------------------
        //  Public, static methods
        // --------------------------------------------------------------------
        static tCIDLib::TVoid ThrowKrnlError();

        static tCIDLib::TVoid ThrowKrnlError
        (
            const   tCIDLib::TErrCode       errcId
        );
 

        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TErrCode errcId() const;

        tCIDLib::TOSErrCode errcHostId() const;


    private :
        // --------------------------------------------------------------------
        //  Private data
        //
        //  __errcId
        //      This is the error id of the error.
        //
        //  __errcHostId
        //      This is the underlying host OS error, if any, that caused it.
        // --------------------------------------------------------------------
        tCIDLib::TErrCode       __errcId;
        tCIDLib::TOSErrCode     __errcHostId;
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Constructors and destructors
// ----------------------------------------------------------------------------
inline TKrnlError::TKrnlError(const tCIDLib::TOSErrCode errcId) :

    __errcId(errcId)
    , __errcHostId(0)
{
}

inline TKrnlError::TKrnlError(const TKrnlError& kerrToCopy) :

    __errcId(kerrToCopy.__errcId)
    , __errcHostId(kerrToCopy.__errcHostId)
{
}

inline TKrnlError::~TKrnlError()
{
}


// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------

inline TKrnlError& TKrnlError::operator=(const TKrnlError& kerrToAssign)
{
    if (this == &kerrToAssign)
        return *this;
    __errcId = kerrToAssign.__errcId;
    __errcHostId = kerrToAssign.__errcHostId;
    return *this;
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------

inline tCIDLib::TErrCode TKrnlError::errcId() const
{
    return __errcId;
}

inline tCIDLib::TOSErrCode TKrnlError::errcHostId() const
{
    return __errcHostId;
}
