//
//  FILE NAME: CIDKernel_Error.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 04/30/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the the TKrnlError class, the core exception
//  class. Most of it is inline.
//
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//  Local types
//
//  TErrMap
//      This is a small structure used to map host errors to our errors.
//      We maintain a list of these, so that they are arranged in increasing
//      values for the host error. This allows for a fast binary lookup of
//      the host error. Then its just a matter of looking into the second
//      entry to get the CIDLib error.
// ----------------------------------------------------------------------------
struct TErrMap
{
    tCIDLib::TOSErrCode     errcHostErr;
    tCIDLib::TErrCode       errcCIDErr;
};


// ----------------------------------------------------------------------------
//  Local data
//
//  __aerrmList
//      This is the list of error mapping structures that are used to map
//      host errors into our errors.
//
//      !!These MUST be in ascending order for the first column, because a
//      binary search is used to find the CIDKernel error that maps to the
//      first column.
// ----------------------------------------------------------------------------
static const TErrMap __aerrmList[] = 
{
        { NO_ERROR                      , kKrnlErrors::errcNoError }
    ,   { ERROR_INVALID_FUNCTION        , kKrnlErrors::errcNotSupported }
    ,   { ERROR_FILE_NOT_FOUND          , kKrnlErrors::errcFileNotFound }
    ,   { ERROR_PATH_NOT_FOUND          , kKrnlErrors::errcPathNotFound }
    ,   { ERROR_TOO_MANY_OPEN_FILES     , kKrnlErrors::errcTooManyOpenFiles }
    ,   { ERROR_ACCESS_DENIED           , kKrnlErrors::errcAccessDenied }
    ,   { ERROR_INVALID_HANDLE          , kKrnlErrors::errcInvalidHandle }
    ,   { ERROR_NOT_ENOUGH_MEMORY       , kKrnlErrors::errcNotEnoughMemory }
    ,   { ERROR_BAD_ENVIRONMENT         , kKrnlErrors::errcBadEnvironment }
    ,   { ERROR_BAD_FORMAT              , kKrnlErrors::errcBadFormat }
    ,   { ERROR_INVALID_ACCESS          , kKrnlErrors::errcInvalidAccess }
    ,   { ERROR_INVALID_DATA            , kKrnlErrors::errcInvalidData }
    ,   { ERROR_OUTOFMEMORY             , kKrnlErrors::errcOutOfMemory }
    ,   { ERROR_INVALID_DRIVE           , kKrnlErrors::errcInvalidDrive }
    ,   { ERROR_CURRENT_DIRECTORY       , kKrnlErrors::errcCurrentDirectory }
    ,   { ERROR_NOT_SAME_DEVICE         , kKrnlErrors::errcNotSameDevice }
    ,   { ERROR_NO_MORE_FILES           , kKrnlErrors::errcNoMoreFiles }
    ,   { ERROR_WRITE_PROTECT           , kKrnlErrors::errcWriteProtect }
    ,   { ERROR_NOT_READY               , kKrnlErrors::errcNotReady }
    ,   { ERROR_BAD_COMMAND             , kKrnlErrors::errcBadCommand }
    ,   { ERROR_CRC                     , kKrnlErrors::errcCRC }
    ,   { ERROR_SEEK                    , kKrnlErrors::errcSeekOnDevice }
    ,   { ERROR_NOT_DOS_DISK            , kKrnlErrors::errcInvalidDrive }
    ,   { ERROR_OUT_OF_PAPER            , kKrnlErrors::errcOutOfPaper }
    ,   { ERROR_WRITE_FAULT             , kKrnlErrors::errcWriteFault }
    ,   { ERROR_READ_FAULT              , kKrnlErrors::errcReadFault }
    ,   { ERROR_GEN_FAILURE             , kKrnlErrors::errcGenFailure }
    ,   { ERROR_SHARING_VIOLATION       , kKrnlErrors::errcSharingViolation }
    ,   { ERROR_WRONG_DISK              , kKrnlErrors::errcWrongDisk }
    ,   { ERROR_DUP_NAME                , kKrnlErrors::errcDuplicateName }
    ,   { ERROR_BAD_NETPATH             , kKrnlErrors::errcBadNetPath }
    ,   { ERROR_NETWORK_BUSY            , kKrnlErrors::errcNetworkBusy }
    ,   { ERROR_DEV_NOT_EXIST           , kKrnlErrors::errcNoSuchDevice }
    ,   { ERROR_PRINTQ_FULL             , kKrnlErrors::errcPrintQFull }
    ,   { ERROR_NO_SPOOL_SPACE          , kKrnlErrors::errcNoSpoolSpace }
    ,   { ERROR_PRINT_CANCELLED         , kKrnlErrors::errcPrintCancelled }
    ,   { ERROR_FILE_EXISTS             , kKrnlErrors::errcFileExists }
    ,   { ERROR_INVALID_PASSWORD        , kKrnlErrors::errcInvalidPassword }
    ,   { ERROR_INVALID_PARAMETER       , kKrnlErrors::errcInvalidParameter }
    ,   { ERROR_EXCL_SEM_ALREADY_OWNED  , kKrnlErrors::errcAlreadyOwned }
    ,   { ERROR_SEM_IS_SET              , kKrnlErrors::errcAlreadySet }
    ,   { ERROR_TOO_MANY_SEM_REQUESTS   , kKrnlErrors::errcTooManySemRequests }
    ,   { ERROR_SEM_OWNER_DIED          , kKrnlErrors::errcOwnerDied }
    ,   { ERROR_DRIVE_LOCKED            , kKrnlErrors::errcDriveLocked }
    ,   { ERROR_BROKEN_PIPE             , kKrnlErrors::errcBrokenPipe }
    ,   { ERROR_OPEN_FAILED             , kKrnlErrors::errcOpenFailed }
    ,   { ERROR_BUFFER_OVERFLOW         , kKrnlErrors::errcBufferOverflow }
    ,   { ERROR_DISK_FULL               , kKrnlErrors::errcDiskFull }
    ,   { ERROR_NO_MORE_SEARCH_HANDLES  , kKrnlErrors::errcNoMoreSearchHandles }
    ,   { ERROR_SEM_TIMEOUT             , kKrnlErrors::errcTimeout }
    ,   { ERROR_INSUFFICIENT_BUFFER     , kKrnlErrors::errcInsufficientBuffer }
    ,   { ERROR_INVALID_NAME            , kKrnlErrors::errcInvalidName }
    ,   { ERROR_NEGATIVE_SEEK           , kKrnlErrors::errcNegativeSeek }
    ,   { ERROR_SEEK_ON_DEVICE          , kKrnlErrors::errcSeekOnDevice }
    ,   { ERROR_DIR_NOT_ROOT            , kKrnlErrors::errcNotRootDirectory }
    ,   { ERROR_DIR_NOT_EMPTY           , kKrnlErrors::errcDirNotEmpty }
    ,   { ERROR_BAD_ARGUMENTS           , kKrnlErrors::errcBadArguments }
    ,   { ERROR_TOO_MANY_TCBS           , kKrnlErrors::errcMaxThreadsReached }
    ,   { ERROR_BAD_PATHNAME            , kKrnlErrors::errcBadPathName }
    ,   { ERROR_MAX_THRDS_REACHED       , kKrnlErrors::errcMaxThreadsReached }
    ,   { ERROR_BUSY                    , kKrnlErrors::errcBusy }
    ,   { ERROR_ALREADY_EXISTS          , kKrnlErrors::errcAlreadyExists }
    ,   { ERROR_SEM_NOT_FOUND           , kKrnlErrors::errcNotFound }
    ,   { ERROR_ENVVAR_NOT_FOUND        , kKrnlErrors::errcNotFound }
    ,   { ERROR_FILENAME_EXCED_RANGE    , kKrnlErrors::errcBadPathName }
    ,   { ERROR_BAD_PIPE                , kKrnlErrors::errcBadPipe }
    ,   { ERROR_PIPE_BUSY               , kKrnlErrors::errcPipeBusy }
    ,   { ERROR_NO_DATA                 , kKrnlErrors::errcNoData }
    ,   { ERROR_PIPE_NOT_CONNECTED      , kKrnlErrors::errcPipeNotConnected }
    ,   { ERROR_MORE_DATA               , kKrnlErrors::errcMoreData }
    ,   { ERROR_DIRECTORY               , kKrnlErrors::errcBadPathName }
    ,   { ERROR_NOT_OWNER               , kKrnlErrors::errcNotOwner }
    ,   { ERROR_TOO_MANY_POSTS          , kKrnlErrors::errcTooManyPosts }
    ,   { ERROR_INVALID_ADDRESS         , kKrnlErrors::errcInvalidAddress }
    ,   { ERROR_PIPE_CONNECTED          , kKrnlErrors::errcPipeConnected }
    ,   { ERROR_PIPE_LISTENING          , kKrnlErrors::errcPipeListening }
    ,   { ERROR_STACK_OVERFLOW          , kKrnlErrors::errcStackOverflow }
    ,   { ERROR_UNRECOGNIZED_VOLUME     , kKrnlErrors::errcUnsupportedFileSystem }
    ,   { ERROR_REGISTRY_RECOVERED      , kKrnlErrors::errcRegistryRecovered }
    ,   { ERROR_REGISTRY_CORRUPT        , kKrnlErrors::errcRegistryCorrupt }
    ,   { ERROR_KEY_DELETED             , kKrnlErrors::errcKeyDeleted }
    ,   { ERROR_KEY_HAS_CHILDREN        , kKrnlErrors::errcKeyHasChildren }
    ,   { ERROR_DLL_INIT_FAILED         , kKrnlErrors::errcDLLInitFailed }
    ,   { ERROR_SHUTDOWN_IN_PROGRESS    , kKrnlErrors::errcShutdownInProgress }
    ,   { ERROR_POSSIBLE_DEADLOCK       , kKrnlErrors::errcPossibleDeadlock }
    ,   { ERROR_APP_WRONG_OS            , kKrnlErrors::errcWrongOS }
    ,   { ERROR_SINGLE_INSTANCE_APP     , kKrnlErrors::errcTooManyInstances }
    ,   { ERROR_BAD_USERNAME            , kKrnlErrors::errcInvalidUserName }
    ,   { ERROR_INVALID_GROUPNAME       , kKrnlErrors::errcInvalidGroupName }
    ,   { ERROR_INVALID_COMPUTERNAME    , kKrnlErrors::errcInvalidComputerName }
    ,   { ERROR_INVALID_DOMAINNAME      , kKrnlErrors::errcInvalidDomainName }
    ,   { ERROR_INVALID_NETNAME         , kKrnlErrors::errcBadNetName }
    ,   { ERROR_INVALID_SHARENAME       , kKrnlErrors::errcInvalidShareName }
    ,   { ERROR_INVALID_PASSWORDNAME    , kKrnlErrors::errcInvalidPasswordName }
    ,   { ERROR_NO_NETWORK              , kKrnlErrors::errcNoNetwork }
    ,   { ERROR_NO_SUCH_PRIVILEGE       , kKrnlErrors::errcNoSuchPrivilege }
    ,   { ERROR_PRIVILEGE_NOT_HELD      , kKrnlErrors::errcPrivilegeNotHeld }
    ,   { ERROR_INVALID_ACCOUNT_NAME    , kKrnlErrors::errcInvalidAccountName }
    ,   { ERROR_USER_EXISTS             , kKrnlErrors::errcAlreadyExists }
    ,   { ERROR_NO_SUCH_USER            , kKrnlErrors::errcNotFound }
    ,   { ERROR_GROUP_EXISTS            , kKrnlErrors::errcAlreadyExists }
    ,   { ERROR_NO_SUCH_GROUP           , kKrnlErrors::errcNotFound }
    ,   { ERROR_MEMBER_IN_GROUP         , kKrnlErrors::errcMemberOfGroup }
    ,   { ERROR_MEMBER_NOT_IN_GROUP     , kKrnlErrors::errcNotMemberOfGroup }
    ,   { ERROR_WRONG_PASSWORD          , kKrnlErrors::errcInvalidPassword }
    ,   { ERROR_LOGON_FAILURE           , kKrnlErrors::errcLogonFailed }
};
static const tCIDLib::TCard4 __c4ErrCount = c4ArrayElems(__aerrmList);


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

static tCIDLib::TErrCode __errcMapError(const tCIDLib::TErrCode errcToMap)
{
    // Set up the two end points that are used to subdivide the list
    tCIDLib::TCard4 c4End = __c4ErrCount - 1;
    tCIDLib::TCard4 c4Begin = 0;

    while (c4Begin < c4End)
    {
        // Divide the current range
        tCIDLib::TCard4 c4MidPoint = (c4Begin + c4End) / 2;

        // Check this guy. If this is it, then return the mapped error
        if (__aerrmList[c4MidPoint].errcHostErr == errcToMap)
            return __aerrmList[c4MidPoint].errcCIDErr;

        // Didn't find it, so see which way to go and adjust begin/end
        if (errcToMap < __aerrmList[c4MidPoint].errcHostErr)
            c4End = c4MidPoint - 1;
        else
            c4Begin = c4MidPoint + 1;
    }

    // Just return generic error
    return kKrnlErrors::errcHostErr;
}



// ----------------------------------------------------------------------------
//  Intra-facility functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid
_InitTermError( const   tCIDLib::EInitTerm      eInitTerm
                , const tCIDLib::EGlobalStates  eGlobals
                , const tCIDLib::TModHandle     hmodThis
                , const tCIDLib::TCard4         c4MaxChars
                ,       tCIDLib::Tch* const     pszFailReason)
{
    #if CID_DEBUG_ON
    //
    //  If we debugging, then make sure that the error id array above really
    //  is in sorted order. If not, then we are in trouble because we use
    //  a binary search to search it.
    //
    if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
    &&  (eGlobals == tCIDLib::EGlobalState_Before))
    {
        for (tCIDLib::TCard4 c4Index = 0; c4Index < (__c4ErrCount-1); c4Index++)
        {
            if (__aerrmList[c4Index].errcHostErr
                                        > __aerrmList[c4Index+1].errcHostErr)
            {
                TRawStr::CopyStr
                (
                    pszFailReason
                    , kMessages::pszErr_OutOfOrder
                    , c4MaxChars
                );
                TKrnlError::ThrowKrnlError(kKrnlErrors::errcInitError);
            }
        }
    }
    #endif
}


// ----------------------------------------------------------------------------
//   CLASS: TKrnlError
//  PREFIX: kerr
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlError: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlError::TKrnlError() :

    __errcHostId(TKrnlThread::errcGetLast())
    , __errcId(kKrnlErrors::errcHostErr)
{
    __errcId = __errcMapError(__errcHostId);
}


// ----------------------------------------------------------------------------
//  TKrnlError: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TKrnlError::ThrowKrnlError()
{
    throw TKrnlError();
}

tCIDLib::TVoid TKrnlError::ThrowKrnlError(const tCIDLib::TErrCode errcId)
{
    throw TKrnlError(errcId);
}
